#include <unistd.h>

#include "framecpp/Common/MD5Sum.hh"

#include "genericAPI/Logging.hh"

#include "DeviceIOConfiguration.hh"
#include "RDSStreamFile.hh"
#include "util.hh"

using FrameCPP::Common::MD5Sum;
using FrameAPI::DeviceIOConfiguration;
using FrameAPI::LogMD5Sum;

RDSStreamFile::
RDSStreamFile( const std::string& OutputDirectory,
	       const std::string& OutputType,
	       const std::string& MD5SumOutputDirectory )
  : m_md5sum_output_directory( MD5SumOutputDirectory ),
    m_output_directory( OutputDirectory ),
    m_output_type( OutputType ),
    m_rds_level( 0 )
{
#if 0
  //   
  // Regex details for the name field representing RDS token.
  //
  // "^"                    The beginning of the buffer   
  // "(RDS_)"               RDS_ keyword
  // "([^_]+_L)"            File description, followed by "_L"    
  // "([0-9]+)"             Level
  // "$"                    The end.
  //   
  static const Regex re_rds_name( "^(RDS_)([^_]+_L)([0-9]+)$" );

  RegexMatch rm( 5 );  

  if ( rm.match( re_rds_name, m_output_type.c_str() ) )
  {
    // It's RDS frame, increment number at the end of the string
    std::istringstream	s_level( rm.getSubString( 3 ) );

    s_level >> m_rds_level;

    ++m_rds_level;
   
    std::ostringstream tmp;

    tmp << rm.getSubString( 1 )
	<< rm.getSubString( 2 )
	<< m_rds_level
      ;
   
    m_output_type = tmp.str( );
  }
#endif /* 0 */
}

void RDSStreamFile::
Abandon( )
{
  std::string	tmpfile( ( m_writer )
			 ? m_writer->GetFilename( )
			 : std::string( "<no file>" ) );

  m_writer.reset( (FileWriter*)NULL );	// Free up resourses
  unlink( tmpfile.c_str( ) );		// Remove file
}

void RDSStreamFile::
Close( bool Final )
{
  QUEUE_LOG_MESSAGE( "ENTRY: Final: " << Final,
		     MT_DEBUG, 30,
		     "RDSStreamFile::Close",
		     "RDS" );
  closeFrameFile( generateOutputName( m_frame_file_start,
				      m_frame_file_deltaT ),
		  Final );
  QUEUE_LOG_MESSAGE( "EXIT",
		     MT_DEBUG, 30,
		     "RDSStreamFile::Close",
		     "RDS" );
}

bool RDSStreamFile::
Next( const LDASTools::AL::GPSTime& FrameStart,
      const REAL_8 Dt,
      INT_4U& FramesPerStream )
{
  bool	retval = false;

  if ( ( m_writer.get( ) == (FileWriter*)NULL ) &&
       ( FrameStart >= m_user_start ) &&
       ( ( FrameStart + Dt ) <= m_user_stop ) )
  {
    openFrameFile( generateTmpOutputName( FrameStart.GetSeconds( ) ) );
    m_frame_file_start = FrameStart.GetSeconds( );
    m_frame_file_deltaT = 0;
    const INT_4U max( INT_4U( ( m_user_stop.GetSeconds( ) - FrameStart.GetSeconds( ) ) / Dt ) );
    if ( FramesPerStream > max )
    {
      FramesPerStream = max;
    }
    retval = true;
    QUEUE_LOG_MESSAGE( " m_frame_file_start: " << m_frame_file_start
		       << " m_frame_file_deltaT: " << m_frame_file_deltaT
		       << " FramesPerStream: " << FramesPerStream,
		       MT_DEBUG, 30,
		       "RDSStreamFile::Next",
		       "RDS" );
  }
  else
  {
    FramesPerStream = 0;
  }
  return retval;
}

void RDSStreamFile::
Write( frame_h_type Frame,
       compression_scheme_type CompressionScheme,
       compression_level_type CompressionLevel,
       chkSum_type CheckSum )
{
  if ( m_writer.get( ) )
  {
    //-----------------------------------------------------------------
    // Write Frame to the frame file
    //-----------------------------------------------------------------
    m_writer->WriteFrame( Frame,
			  CompressionScheme,
			  CompressionLevel,
			  CheckSum );
    m_frame_file_deltaT += Frame->GetDt( );
  }
}

void RDSStreamFile::
appendOutputFrameFilename( const std::string& Filename )
{
  m_output_frame_names.push_back( Filename );
}

void RDSStreamFile::
closeFrameFile( const std::string& Filename,
	       bool Final )
{
  std::cerr << "DEBUG:"
	    << "RDSStreamFileReduce::closeFrameFile:"
	    << std::endl
    ;
  closeFrameWriter( m_writer, Filename );
  m_writer.reset( (FileWriter*)NULL );	// Free up resourses
}

void RDSStreamFile::
closeFrameWriter( file_writer_type Writer,
		  const std::string& Filename )
{
  std::cerr << "DEBUG:"
	    << "RDSStreamFileReduce::closeFrameWriter:"
	    << std::endl
    ;
   if ( Writer )
   {
      //-----------------------------------------------------------------
      // Close output frame file, renaming it in accordance to
      //   user's request
      //-----------------------------------------------------------------
      Writer->Close( );

      std::string	tmpfile( Writer->GetFilename( ) );
      bool		calculating_md5_sum = true;
      MD5Sum		md5;

      try
      {
	 md5 = Writer->GetMD5Sum( );
	 md5.Finalize( );
      }
      catch( ... )
      {
	 calculating_md5_sum = false;
      }

      try
      {
	 ensureNoSuchFile( Filename );
      }
      catch( ... )
      {
	 unlink( tmpfile.c_str( ) );	// Remove the tempory file
	 throw;
      }
      if ( rename( tmpfile.c_str( ), Filename.c_str( ) ) != 0 )
      {
	 //--------------------------------------------------------------
	 // Cannot rename the file. Must remove the temporary file
	 //   and inform the call about the error.
	 //--------------------------------------------------------------
	 unlink( tmpfile.c_str( ) );	// Remove the tempory file
	 std::ostringstream	oss;
	 
	 oss << "Unable to create: " << Filename
	     << " src: " << tmpfile
	     << " perror: " << strerror( errno );
	 throw std::runtime_error( oss.str( ) );
      }
      if ( calculating_md5_sum )
      {
	 LogMD5Sum( Filename, md5, m_md5sum_output_directory );
      }
      appendOutputFrameFilename( Filename );
   } // if ( Writer )
}

void RDSStreamFile::
ensureNoSuchFile( const std::string& Filename ) const
{
   std::ifstream	test_file;
      
   try
   {
      test_file.open( Filename.c_str( ) );
   }
   catch( ... )
   {
      // File does not exist == continue
      return;
   }
   if ( test_file.is_open( ) )
   {
      test_file.close( );
   }
   else
   {
      return;
   }
   std::ostringstream	err;
   err << "Frame file already exists: " << Filename;
   throw SWIGEXCEPTION( err.str( ) );
}

std::string RDSStreamFile::
generateOutputName( INT_4U FileStart, REAL_8 FileDeltaT ) const
{
   //--------------------------------------------------------------------
   // Generate an output file
   //--------------------------------------------------------------------
   std::ostringstream	oss;
   oss << m_output_directory << "/"
       << IFOList( )
       << "-" << m_output_type
       << "-" << FileStart
       << "-" << FileDeltaT
       << ".gwf";
   return oss.str( );
}

std::string RDSStreamFile::
generateTmpOutputName( const INT_4U FrameStart ) const
{
   //--------------------------------------------------------------------
   // Generate the tmp output file name
   //--------------------------------------------------------------------
   std::ostringstream	oss;
   oss << m_output_directory
       << "/rds"
       << FrameStart
       << "_"
       << getpid( )
       << "_"
       << pthread_self( )
       << ".gwf.tmp";
   return oss.str( );
}

void RDSStreamFile::
openFrameFile( const std::string& Filename )
{
  unsigned int	buffer_size;
  bool		enable_memory_mapped_io;

  DeviceIOConfiguration::GetConfiguration( Filename.c_str( ),
					   buffer_size,
					   enable_memory_mapped_io );

   m_writer.reset( new FileWriter( Filename.c_str( ),
				  buffer_size,
				  (char*)NULL,
				  enable_memory_mapped_io ) );
    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    const std::string& dir( m_md5sum_output_directory );

    if ( ::access( dir.c_str( ), R_OK ) == 0 )
    {
       //----------------------------------------------------------------
       // Setup to calculate the MD5Sum
       //----------------------------------------------------------------
       m_writer->ResetMD5Sum( true );
    }
    else
    {
       m_writer->ResetMD5Sum( false );
    }
}

