#include <ldas_tools_config.h>

#include "ldastoolsal/mutexlock.hh"
#include "ldastoolsal/Singleton.hh"
#include "ldastoolsal/gpstime.hh"

#include "framecpp/Common/Description.hh"
#include "framecpp/Common/FrameStream.hh"
#include "framecpp/Common/FrameSpec.tcc"

#include "framecpp/Version8/FrProcData.hh"
#include "framecpp/Version8/FrSE.hh"
#include "framecpp/Version8/FrSH.hh"

#include "framecpp/Version8/PTR_STRUCT.hh"

using std::ceil;
using std::floor;

using FrameCPP::Common::Description;
using FrameCPP::Common::FrameSpec;

//=======================================================================
// Static
//=======================================================================

static const FrameSpec::Info::frame_object_types s_object_id
= FrameSpec::Info::FSI_FR_PROC_DATA;

//=======================================================================
// Anonymous
//=======================================================================
namespace {
  typedef FrameCPP::Version_8::FrProcData::type_type type_type;

  class IDTypeToStringMapping
  {
    SINGLETON_TS_DECL( IDTypeToStringMapping );
  public:
    const std::string& GetString( type_type Id ) const;

  private:
    std::vector< std::string >	m_types;
  };

  class IDSubTypeToStringMappingFrequency
  {
    SINGLETON_TS_DECL( IDSubTypeToStringMappingFrequency );
  public:
    const std::string& GetString( type_type Id ) const;

  private:
    std::vector< std::string >	m_types;
  };

} // namespace - anonymous


namespace FrameCPP
{
  namespace Version_8
  {
    //=======================================================================
    // FrProcData
    //=======================================================================
    FrProcData::
    FrProcData( )
      : object_type( s_object_id, StructDescription( ) ),
	m_synced_with_vector( false )
    {
    }

    FrProcData::
    FrProcData( const FrProcData& ProcData )
      : object_type( s_object_id, StructDescription( ) ),
	Common::TOCInfo( ProcData ),
	FrProcDataNPS( ProcData ),
	FrProcDataPS( ProcData ),
	m_synced_with_vector( false )
    {
    }

    FrProcData::
    FrProcData( const std::string& Name,
		const std::string& Comment,
		type_type Type,
		subType_type SubType,
		timeOffset_type TimeOffset,
		tRange_type TRange,
		fShift_type FShift,
		phase_type Phase,
		fRange_type FRange,
		BW_type BW )
      : object_type( s_object_id, StructDescription( ) ),
	FrProcDataNPS( Name,
		       Comment,
		       Type,
		       SubType,
		       TimeOffset,
		       TRange,
		       FShift,
		       Phase,
		       FRange,
		       BW ),
	m_synced_with_vector( false )
    {
    }

#if 0
    FrProcData::
    FrProcData( const std::string& name,
		REAL_8 sampleRate, const GPSTime& timeOffset,
		REAL_8 fShift, REAL_4 phase )
      : object_type( s_object_id, StructDescription( ) ),
	m_synced_with_vector( false )

    {
      m_data.name = name;
      m_data.comment = std::string( "" );
      m_data.type = 0;
      m_data.subType = 0;
      m_data.timeOffset = timeOffset - GPSTime( 0, 0 );
      m_data.tRange = 0;
      m_data.fShift = fShift;
      m_data.phase = phase;
      m_data.fRange = 0;
      m_data.BW = 0;
    }
#endif /* 0 */

    FrProcData::
    FrProcData( Previous::FrProcData& Source, istream_type* Stream )
      : object_type( s_object_id, StructDescription( ) ),
	FrProcDataNPS( Source ),
	FrProcDataPS( Source, Stream ),
	m_synced_with_vector( false )
    {
    }

    FrProcData::
    FrProcData( istream_type& Stream )
      : object_type( s_object_id, StructDescription( ) ),
	FrProcDataNPS( Stream ),
	FrProcDataPS( Stream ),
	m_synced_with_vector( false )
    {
      Stream.Next( this );
    }

    FrProcData::
    ~FrProcData( )
    {
    }

    const std::string& FrProcData::
    IDTypeToString( type_type Type )
    {
      return IDTypeToStringMapping::Instance( ).GetString( Type );
    }

    const char* FrProcData::
    ObjectStructName( ) const
    {
      return StructName( );
    }

    const std::string& FrProcData::
    GetNameSlow( ) const
    {
      return GetName( );
    }

    const Description* FrProcData::
    StructDescription( )
    {
      static Description ret;

      if ( ret.size( ) == 0 )
      {
	ret( FrSH( FrProcData::StructName(), s_object_id,
		   "Post-processed Data Structure" ) );

	ret( FrSE( "name", "STRING",
		   "" ) );
	ret( FrSE( "comment", "STRING",
		   "" ) );
	ret( FrSE( "type", "INT_2U",
		   "" ) );
	ret( FrSE( "subType", "INT_2U",
		   "" ) );
	ret( FrSE( "timeOffset", "REAL_8",
		   "" ) );
	ret( FrSE( "tRange", "REAL_8",
		   "" ) );
	ret( FrSE( "fShift", "REAL_8",
		   "" ) );
	ret( FrSE( "phase", "REAL_4",
		   "" ) );
	ret( FrSE( "fRange", "REAL_8",
		   "" ) );
	ret( FrSE( "BW", "REAL_8",
		   "" ) );
	ret( FrSE( "nAuxParam", "INT_2U",
		   "" ) );
	ret( FrSE( "auxParam", "REAL_8[nAuxParam]",
		   "" ) );
	ret( FrSE( "auxParamNames", "STRING[nAuxParam]",
		   "" ) );

	ret( FrSE( "data",	PTR_STRUCT::Desc( FrVect::StructName( ) ) ) );
	ret( FrSE( "aux", PTR_STRUCT::Desc( FrVect::StructName( ) ) ) );
	ret( FrSE( "table", PTR_STRUCT::Desc( FrTable::StructName( ) ) ) );
	ret( FrSE( "history", PTR_STRUCT::Desc( FrHistory::StructName( ) ) ) );

	ret( FrSE( "next", PTR_STRUCT::Desc( FrProcData::StructName( ) ) ) );

	ret( FrSE( "chkSum", CheckSumDataClass( ), CheckSumDataComment( ) ) );
      }

      return &ret;
    }

    const std::string& FrProcData::
    IDSubTypeToString( type_type Type, subType_type SubType )
    {
      static const std::string	empty( "" );
      switch ( Type )
      {
      case FREQUENCY_SERIES:
	return IDSubTypeToStringMappingFrequency::Instance( ).GetString( Type );
	break;
      default:
	return empty;
	break;
      }
    }

    FrProcData& FrProcData::
    Merge( const FrProcData& RHS )
    {
      //:TODO: Need to implement Merge routine
      std::string msg( "Merge currently not implemented for " );
      msg += StructName( );

      throw std::domain_error( msg );
      return *this;
    }

    FrProcData::subset_ret_type FrProcData::
    Subset( REAL_8 Offset, REAL_8 Dt ) const
    {
      subset_ret_type	retval;

      //---------------------------------------------------------------------
      // Process the types that are understood
      //---------------------------------------------------------------------
      switch( GetType( ) )
      {
      case TIME_SERIES:
	retval = sub_time_series( Offset, Dt );
	break;
      default:
	{
	  //-----------------------------------------------------------------
	  // This type is not supported so throw an exception
	  //-----------------------------------------------------------------
	  std::ostringstream	msg;
	  msg << "Currently cannot take a subsection of type: "
	      << IDTypeToString( GetType( ) );
	  const std::string& subtype( IDSubTypeToString( GetType( ),
							 GetSubType( ) ) );
	  if ( subtype.length( ) > 0 )
	  {
	    msg << " subType: " << subtype;
	  }
	  throw std::runtime_error( msg.str( ) );
	}
	break;
      }

      return retval;
    }

    bool FrProcData::
    operator==( const Common::FrameSpec::Object& Obj ) const
    {
      return ( *this == Obj );
    }

    void FrProcData::
    copy_core( const FrProcData& Source )
    {
      FrProcDataNPS::operator=( Source );
      FrProcDataPS::copy_core( Source );
    }

    FrProcData::demote_ret_type FrProcData::
    demote( INT_2U Target,
	    demote_arg_type Obj,
	    istream_type* Stream ) const
    {
      if ( Target >= DATA_FORMAT_VERSION )
      {
	return Obj;
      }
      try
      {
	//-------------------------------------------------------------------
	// Copy non-reference information
	//-------------------------------------------------------------------
	// Do actual down conversion
	/// \todo See if the actual sample rate can be computed
	Previous::GPSTime t;
	t += GetTimeOffset( );
	LDASTools::AL::SharedPtr< Previous::FrProcData >
	  retval( new Previous::FrProcData( GetName( ),
					    GetComment( ),
					    GetType( ),
					    GetSubType( ),
					    GetTimeOffset( ),
					    GetTRange( ),
					    GetFShift( ),
					    GetPhase( ),
					    GetFRange( ),
					    GetBW( )
					    ) )
	  ;
	if ( Stream )
	{
	  //-----------------------------------------------------------------
	  // Modify references
	  //-----------------------------------------------------------------
	  Stream->ReplaceRef( retval->RefAux( ), RefAux( ), MAX_REF );
	  Stream->ReplaceRef( retval->RefData( ), RefData( ), MAX_REF );
	  Stream->ReplaceRef( retval->RefTable( ), RefTable( ), MAX_REF );
	  Stream->ReplaceRef( retval->RefHistory( ), RefHistory( ), MAX_REF );
	}
	//-------------------------------------------------------------------
	// Return demoted object
	//-------------------------------------------------------------------
	return retval;
      }
      catch( ... )
      {
      }
      throw
	Unimplemented( "Object* FrProcData::demote( Object* Obj ) const",
		       DATA_FORMAT_VERSION, __FILE__, __LINE__ );
    }

    FrProcData::promote_ret_type FrProcData::
    promote( INT_2U Target,
	     promote_arg_type Obj,
	     istream_type* Stream ) const
    {
      return Promote( Target, Obj, Stream );
    }

    FrameCPP::cmn_streamsize_type FrProcData::
    pBytes( const Common::StreamBase& Stream ) const
    {
      return
	FrProcDataNPS::bytes( )
	+ FrProcDataPS::bytes( Stream )
	+ Stream.PtrStructBytes( ) // next
	;
    }

    FrProcData* FrProcData::
    pCreate( istream_type& Stream ) const
    {
      return new FrProcData( Stream );
    }

    void FrProcData::
    pWrite( ostream_type& Stream ) const
    {
      FrProcDataNPS::write( Stream );
      FrProcDataPS::write( Stream );
      WriteNext( Stream );
    }

    void FrProcData::
    sync_with_vector( )
    {
      if ( ( m_synced_with_vector == false ) &&
	   ( RefData( ).size( ) > 0 ) )
      {
	switch( GetType( ) )
	{
	case TIME_SERIES:
	  SetTRange( ( RefData( )[ 0 ]->GetDim( 0 ).GetNx( ) *
		       RefData( )[ 0 ]->GetDim( 0 ).GetDx( ) ) );
	  break;
	case FREQUENCY_SERIES:
	  SetFRange( ( RefData( )[ 0 ]->GetDim( 0 ).GetNx( ) *
		       RefData( )[ 0 ]->GetDim( 0 ).GetDx( ) ) );
	  break;
	}
	m_synced_with_vector = true;
      }
    }

    // Routine to take a subsection when the data type is TimeSeries data
    FrProcData::subset_ret_type FrProcData::
    sub_time_series( REAL_8 Offset, REAL_8 Dt ) const
    {
      //---------------------------------------------------------------------
      // Sanity checks
      //---------------------------------------------------------------------
      // Verify the existance of only a single FrVect component
      //---------------------------------------------------------------------
      if ( RefData( ).size( ) != 1 )
      {
	std::ostringstream	msg;

	msg << "METHOD: frameAPI::FrProcData::SubFrProcData: "
	    << "data vectors of length " << RefData( ).size( )
	    << " currently are not supported";
	throw std::logic_error( msg.str( ) );
      }
      INT_4U	num_elements = RefData( )[0]->GetDim( 0 ).GetNx( );
      //---------------------------------------------------------------------
      //:TODO: Ensure the dimension of the FrVect is 1
      //---------------------------------------------------------------------
      //---------------------------------------------------------------------
      //:TODO: Ensure the requested range is valid
      //---------------------------------------------------------------------
      REAL_8	sample_rate = 1.0 / RefData( )[ 0 ]->GetDim( 0 ).GetDx( );
      if ( Offset > ( GetTimeOffset( ) +
		      ( num_elements / sample_rate ) ) )
      {
	std::ostringstream msg;
	msg << "METHOD: frameAPI::FrProcData::SubFrProcData: "
	    << "The value for the parameter Offset (" << Offset
	    << ") is out of range ([0-"
	    << ( GetTimeOffset( ) + ( num_elements / sample_rate ) )
	    << "))";
	throw std::range_error( msg.str( ) );
      }
      //---------------------------------------------------------------------
      // Start moving of data
      //---------------------------------------------------------------------
      subset_ret_type	retval( new FrProcData );
      // Copy in the header information
      retval->copy_core( *this );

      //:TODO: Get the slice of FrVect
      REAL_8	startSampleReal =
	( Offset * sample_rate ) -
	( retval->GetTimeOffset( ) * sample_rate );

      INT_4U	endSample =
	INT_4U( ceil ( startSampleReal +
		       ( Dt * sample_rate ) ) );
      INT_4U	startSample = ( startSampleReal > REAL_8( 0 ) )
	? INT_4U( floor( startSampleReal ) )
	: INT_4U( 0 );

      {
	LDASTools::AL::SharedPtr< FrVect >
	  subvect( RefData( )[ 0 ]->
		   SubFrVect( startSample, endSample ).release( ) );
	retval->RefData( ).append( subvect );
      }

      //---------------------------------------------------------------------
      // Shift the offset
      //---------------------------------------------------------------------
      if ( startSampleReal > 0 )
      {
	// There is no gap
	retval->SetTimeOffset( REAL_8( 0 ) );
      }
      else
      {
	retval->timeOffset += Offset;
      }
      //---------------------------------------------------------------------
      // :TODO: Recalculate tRange
      //---------------------------------------------------------------------
      retval->SetTRange( retval->RefData( )[ 0 ]->GetDim( 0 ).GetNx( ) * 
			 retval->RefData( )[ 0 ]->GetDim( 0 ).GetDx( ) );
      //---------------------------------------------------------------------
      // Return the new FrProcData
      //---------------------------------------------------------------------
      return retval;
    }

  } // namespace - Version_8
} // namespace - FrameCPP

//=======================================================================
// Anonymous
//=======================================================================
namespace
{
  //---------------------------------------------------------------------
  // Initialization of singleton
  //---------------------------------------------------------------------
  SINGLETON_TS_INST( IDTypeToStringMapping );
  //---------------------------------------------------------------------
  // Default constructor
  //---------------------------------------------------------------------
  IDTypeToStringMapping::
  IDTypeToStringMapping( )
  {
    m_types.push_back( "Unknown/user defined" );
    m_types.push_back( "Time Series" );
    m_types.push_back( "Frequency Series" );
    m_types.push_back( "Other 1D Series data" );
    m_types.push_back( "Time-Frequency" );
    m_types.push_back( "Wavelets" );
    m_types.push_back( "Multi-dimensional" );
  }

  const std::string& IDTypeToStringMapping::
  GetString( type_type Type ) const
  {
    if ( Type >= m_types.size( ) )
    {
      std::ostringstream	msg;
      msg << "Cannot map FrProcData type value of " << Type
	  << " to string value because it is out of range";
      throw std::range_error( msg.str( ) );
    }
    return m_types[ Type ];
  }

  //---------------------------------------------------------------------
  // Initialization of singleton
  //---------------------------------------------------------------------
  SINGLETON_TS_INST( IDSubTypeToStringMappingFrequency );
  //---------------------------------------------------------------------
  // Default constructor
  //---------------------------------------------------------------------
  IDSubTypeToStringMappingFrequency::
  IDSubTypeToStringMappingFrequency( )
  {
    m_types.push_back( "Unknown/user defined" );
    m_types.push_back( "Amplitude Spectral Density" );
    m_types.push_back( "Power Spectral Density" );
    m_types.push_back( "Cross Spectral Density" );
    m_types.push_back( "Coherence" );
    m_types.push_back( "Transfer Function" );
  }

  const std::string& IDSubTypeToStringMappingFrequency::
  GetString( INT_2U Type ) const
  {
    if ( Type >= m_types.size( ) )
    {
      std::ostringstream	msg;
      msg << "Cannot map FrProcData type value of " << Type
	  << " to string value because it is out of range";
      throw std::range_error( msg.str( ) );
    }
    return m_types[ Type ];
  }
} // namespace - anonymous

namespace FrameCPP
{
  namespace Version_8
  {
    void FrProcData::
#if WORKING_VIRTUAL_TOCQUERY
    TOCQuery( int InfoClass, ... ) const
#else /*  WORKING_VIRTUAL_TOCQUERY */
    vTOCQuery( int InfoClass, va_list vl ) const
#endif /*  WORKING_VIRTUAL_TOCQUERY */
    {
      using Common::TOCInfo;

#if WORKING_VIRTUAL_TOCQUERY
      va_list	vl;
      va_start( vl, InfoClass );
#endif /*  WORKING_VIRTUAL_TOCQUERY */

      while ( InfoClass != TOCInfo::IC_EOQ )
      {
	int data_type = va_arg( vl, int );
	switch( data_type )
	{
	case TOCInfo::DT_STRING_2:
	  {
	    STRING* data = va_arg( vl, STRING* );
	    switch( InfoClass )
	    {
	    case TOCInfo::IC_NAME:
	      *data = GetName( );
	      break;
	    default:
	      goto cleanup;
	      break;
	    }
	  }
	  break;
	default:
	  // Stop processing
	  goto cleanup;
	}
	InfoClass = va_arg( vl, int );
      }
    cleanup:
#if WORKING_VIRTUAL_TOCQUERY
      va_end( vl )
#endif /*  WORKING_VIRTUAL_TOCQUERY */
	;
    }
  } // namespace - Version_8
} // namespace - FrameCPP

