/* -*- mode: c++ -*- */
#ifndef FR_EVENT_I
#define FR_EVENT_I

%module frameCPP

%include "std_pair.i"
%include "std_string.i"
%include "std_vector.i"

%{
#include "ldastoolsal/types.hh"
#include "framecpp/FrEvent.hh"

// SWIG thinks that Param_type is a global class, so we need to trick the C++
// compiler into understanding this so called global type.
typedef FrameCPP::FrEvent::Param_type FrEventParam;

// SWIG thinks that ParamList_type is a global class, so we need to trick the C++
// compiler into understanding this so called global type.
typedef FrameCPP::FrEvent::ParamList_type FrEventParamList;

%}

%import "framecpp/FrameCPP.hh"
%import "framecpp/FrEvent.hh"

%import "Container.i"

%apply double { REAL_8 };
%define FR_EVENT_PARAM_SWIG_DEF
std::pair< FR_EVENT_PARAM_NAME_TYPE,
	   FR_EVENT_PARAM_VALUE_TYPE >
%enddef

%template(FrEventParam) FR_EVENT_PARAM_SWIG_DEF;

%template(FrEventParamList) std::vector< std::pair< std::string, double > >;

%feature("autodoc",
"""
FrEvent (Event) implementation of the frame specification.

  Attributes:

    name         Name of event.
    comment      Description of event.
    inputs       Input channels and filter parameters to event
                 process.
    GTime        GPS time corresponiding to reference value of
                 event, as defined by the search algorithm.
    timeBefore   Signal duration before (seconds).
    timeAfter    Signal duration after (seconds).
    eventStatus  Defined by event search algorithm.
    amplitude    Continuous output amplitude returned by event.
    probability  Likelihood estimate of event, if available
                 (probability = -1 if cannot be estimated).
    statistics   Statistical description of event, if relevant
                 or available.
    params       Array of additional event parameters.
""" ) FrEvent;

class FrEvent
{
public:
  typedef FR_EVENT_NAME_TYPE		name_type;
  typedef FR_EVENT_COMMENT_TYPE		comment_type;
  typedef FR_EVENT_INPUTS_TYPE		inputs_type;
  typedef FR_EVENT_TIME_TYPE		time_type;
  typedef FR_EVENT_TIME_BEFORE_TYPE	timeBefore_type;
  typedef FR_EVENT_TIME_AFTER_TYPE	timeAfter_type;
  typedef FR_EVENT_EVENT_STATUS_TYPE	eventStatus_type;
  typedef FR_EVENT_AMPLITUDE_TYPE	amplitude_type;
  typedef FR_EVENT_PROBABILITY_TYPE	probability_type;
  typedef FR_EVENT_STATISTICS_TYPE	statistics_type;
  typedef std::vector< FR_EVENT_PARAM_SWIG_DEF > ParamList_type;

  FrEvent( );

  FrEvent( const name_type& name,
	   const comment_type& comment,
	   const inputs_type& inputs,
	   const time_type& time,
	   const timeBefore_type timeBefore,
	   const timeAfter_type timeAfter,
	   const eventStatus_type eventStatus,
	   const amplitude_type amplitude,
	   const probability_type prob,
	   const statistics_type& statistics,
	   const ParamList_type& parameters );

  const name_type& GetName() const;
  const comment_type& GetComment( ) const;
  const inputs_type& GetInputs( ) const;
  const time_type& GetGTime( ) const;
  timeBefore_type GetTimeBefore( ) const;
  timeAfter_type GetTimeAfter( ) const;
  eventStatus_type GetEventStatus( ) const;
  amplitude_type GetAmplitude( ) const;
  probability_type GetProbability( ) const;
  const statistics_type& GetStatistics( ) const;
  const ParamList_type& GetParam( ) const;

  LDAS_PROPERTY_READ_ONLY("name",GetName)
  LDAS_PROPERTY_READ_ONLY("comment",GetComment)
  LDAS_PROPERTY_READ_ONLY("inputs",GetInputs)
  LDAS_PROPERTY_READ_ONLY("GTime",GetGTime)
  LDAS_PROPERTY_READ_ONLY("timeBefore",GetTimeBefore)
  LDAS_PROPERTY_READ_ONLY("timeAfter",GetTimeAfter)
  LDAS_PROPERTY_READ_ONLY("eventStatus",GetEventStatus)
  LDAS_PROPERTY_READ_ONLY("amplitude",GetAmplitude)
  LDAS_PROPERTY_READ_ONLY("probability",GetProbability)
  LDAS_PROPERTY_READ_ONLY("statistics",GetStatistics)
  LDAS_PROPERTY_READ_ONLY("params",GetParam)
};

CONTAINER_WRAP(FrEvent,FrEvent)

#endif /* FR_EVENT_I */
