#ifndef WINDOW_HH
#define WINDOW_HH

//!ppp: {eval `cat $ENV{SRCDIR}/cc_rules.pl`}

#include <string>
#include <complex>
#include <valarray>

namespace Filters {

    //
    //: An abstract base class for window functions
    //
    class Window {

    public:
	//
	//: Default constructor
	//
	Window();

	//
	//: Destructor
	//
	virtual ~Window();

	//
	//: Copy assignment 
	//
	//!param: r - Window that is copied
	//
	//!return: const Window&
	//
	const Window& operator=(const Window& r);
    
	//
	//: Return window length
	//
	//!return: size_t - length of window
	//
	size_t size() const;

	//
	//: Return mean value of window
	//
	//!return: double - mean value of window elements
	//
	double mean() const;
	
	//
	//: Return RMS value of window
	//
	//!return: double - root mean square value of window elements
	//
	double rms() const;

	//
	//: Return window name
	//
	//!return: string - window name
	//
	virtual std::string name() const = 0;

	//
	//: Return window parameter
	//
	//!return: double - window parameter
	//
	virtual double param() const = 0;

	//
	//: Change the size of the window
	//
	//!param: n - desired length of window
	//
	//!exc: std::length_error - thrown if desired window length is 
	//+ greater than maximum allowed value 
	//
	//!exc: std::bad_alloc - thrown if not able to allocate memory for
	//+ the window 
	//
	void resize(const size_t n);
      
	//
	//: Apply a window to the data in-place
	//
	//!param: x  - input/output sequence
	//
	//!exc: std::length_error - thrown if desired window length is 
	//+ greater than maximum allowed value 
	//
	//!exc: std::bad_alloc - thrown if not able to allocate memory for 
	//+ window
	//
	template<class T>
	void apply(std::valarray<T>& x); 

	//
	//: Apply a window to the data out-of-place
	//
	//!param: out - windowed sequence
	//!param: in  - input sequence
	//
	//!exc: std::length_error - thrown if desired window length is 
	//+ greater than maximum allowed value 
	//
	//!exc: std::bad_alloc - thrown if not able to allocate memory for 
	//+ window
	//
	template<class TOut, class TIn>
	void apply(std::valarray<TOut>& out, 
		   const std::valarray<TIn>&  in);

	//
	//: Synonym for apply
	//
	//!param: out - windowed sequence
	//!param: in  - input sequence
	//
	//!exc: std::length_error - thrown if desired window length is 
	//+ greater than maximum allowed value 
	//
	//!exc: std::bad_alloc - thrown if not able to allocate memory for 
	//+ window
	//
	template<class TOut, class TIn>
	void operator()(std::valarray<TOut>& out, 
			const std::valarray<TIn>& in);

	//
	//: Clone a window
	//
	//!return: Window* - copy of current window
	//
	virtual Window* Clone() const = 0;

    protected:
	//
	//: Populate the elements of the window
	//
	// This function will usually need to be called by a derived
	// class whenever the the window is changed in any way which
	// affects the element() function, such as when some window
	// parameter is altered.
	//
	void populate();

    private:
	//
	//: Get the ith element of the window
	//
	// This function returns the value corresponding to the
	// ith position in the window. 
	// It is the window function evaluated at x = i/N where N is 
	// the length of the window.
	//
	//!param: i - element at which the window is evaluated
	//
	//!return: double - the window element w(i/N)
	//
	virtual double element(const size_t i) const = 0;

	//
	//: Multiply a sequence of floats by the window
	//
	//!param: out - input (and output) sequence
	//
	void multiply_by_window(std::valarray<float>& out);

	//
	//: Multiply a sequence of doubles by the window
	//
	//!param: out - input (and output) sequence
	//
	void multiply_by_window(std::valarray<double>& out);

	//
	//: Multiply a sequence of complex floats by the window
	//
	//!param: out - input (and output) sequence
	//
	void multiply_by_window(std::valarray<std::complex<float> >& out);
	
	//
	//: Multiply a sequence of complex doubles by the window
	//
	//!param: out - input (and output) sequence
	//
	void multiply_by_window(std::valarray<std::complex<double> >& out);

	//: a sequence for holding a window of floats
	std::valarray<float>  m_f_window;
	
	//: a sequence for holding a window of doubles
	std::valarray<double> m_d_window;

	//: mean value of window elements
	double m_mean;
	
	//: rms value of window elements
	double m_rms;
	
	//: true if window elements are floats
	bool   m_is_float;
    };

    inline
    size_t Window::size() const
    {
	return (m_is_float ? m_f_window.size() : m_d_window.size());
    }

    inline
    double Window::mean() const 
    { 
	return m_mean; 
    }
    
    inline
    double Window::rms()  const 
    { 
	return m_rms; 
    }

} // namespace Filters

#endif // WINDOW_HH
