#ifndef FrameCPP_VERSION_8_FrTOCProcData_HH
#define FrameCPP_VERSION_8_FrTOCProcData_HH

#if ! defined(SWIGIMPORTED)
#include "framecpp/Version7/FrTOCProcData.hh"

#include "framecpp/Version8/STRING.hh"
#endif /* ! defined(SWIGIMPORTED) */

namespace FrameCPP
{
#if ! defined(SWIG)
  namespace Common
  {
    class TOCInfo;
  }
#endif /* ! defined(SWIG) */

  namespace Version_8
  {
    //===================================================================
    /// \brief Indexed elements of the FrProcData structure.
    //===================================================================
    class FrTOCProcData
    {
    public:
      typedef INT_4U	nproc_type;
      typedef STRING	name_type;
      typedef INT_8U	position_type;

      typedef std::vector< name_type > key_container_type;
      typedef std::vector< position_type > proc_info_type;
      typedef std::map< name_type, proc_info_type >
      MapProc_type;

      //-----------------------------------------------------------------
      /// \brief Default constructor.
      //-----------------------------------------------------------------
      FrTOCProcData( );

      //-----------------------------------------------------------------
      /// \brief Number of bytes needed to write this structure
      ///
      /// \param[in] Stream
      ///     The stream from which to the object is being read or
      ///     written.
      ///
      /// \return
      ///     The number of bytes need to read or write this object.
      //-----------------------------------------------------------------
      Common::FrameSpec::size_type
      Bytes( const Common::StreamBase& Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the keys
      ///
      /// \return
      ///     Constant container of keys.
      //-----------------------------------------------------------------
      const key_container_type& GetKeys( ) const;

      //-----------------------------------------------------------------
      /// \brief Return all channels
      ///
      /// \return
      ///     Constant container to the indexed FrProcData elements.
      //-----------------------------------------------------------------
      const MapProc_type& GetProc( ) const;

      //-----------------------------------------------------------------
      /// \brief Return the indexed data for a named channel.
      ///
      /// \param[in] Name
      ///     The name of the channel.
      ///
      /// \return
      ///     Constant iterator to the indexed FrProcData element.
      //-----------------------------------------------------------------
      MapProc_type::const_iterator GetProc( const std::string& Name ) const;

      //-----------------------------------------------------------------
      /// \brief Return the indexed data for a named channel.
      ///
      /// \param[in] index
      ///     The numeric offset of the channel.
      ///
      /// \return
      ///     Constant iterator to the indexed FrProcData element.
      //-----------------------------------------------------------------
      MapProc_type::const_iterator GetProc( INT_4U index ) const;

      //-----------------------------------------------------------------
      /// \brief Gather TOC info for FrProcData being written.
      ///
      /// \param[in] Info
      ///     Information
      ///
      /// \param[in] FrameOffset
      ///     The frame offset of the frame being written.
      ///     The frame offsets start at 0 (zero).
      ///
      /// \param[in] Position
      ///     The byte offset from the start of the file
      ///     where the structure is written.
      //-----------------------------------------------------------------
      void QueryProc( const Common::TOCInfo& Info,
		      INT_4U FrameOffset,
		      INT_8U Position );

#if ! defined(SWIG)
      //-----------------------------------------------------------------
      /// \brief asignment operator
      ///
      /// \param[in] Source
      ///     The source to be copied.
      //-----------------------------------------------------------------
      const FrTOCProcData& operator=( const Previous::FrTOCProcData& Source );
#endif /* ! defined(SWIG) */

      //-----------------------------------------------------------------
      /// \brief equality operator
      ///
      /// \param[in] RHS
      ///     The FrTOCAdcData object to be compared.
      ///
      /// \return
      ///     The value true is returned if this object is equivelent
      ///     to the RHS instance; false otherwise.
      //-----------------------------------------------------------------
      bool operator==( const FrTOCProcData& RHS ) const;

      //-----------------------------------------------------------------
      /// \brief The description of structure
      ///
      /// \param[out] Desc
      ///     Storage for the description of the structure.
      ///
      /// \return
      ///     A Description object which describes this structure as
      ///     specified by the frame specification.
      //-----------------------------------------------------------------
      template< typename SE >
      static void Description( Common::Description& Desc );

    protected:
      key_container_type	m_keys;
      MapProc_type		m_info;

      //-----------------------------------------------------------------
      /// \brief Stream constructor
      ///
      /// \param[in] Stream
      ///     The input stream from where the object is being read.
      ///
      /// \param[in] FrameCount
      ///     The frame offset.
      //-----------------------------------------------------------------
      FrTOCProcData( Common::IStream& Stream, INT_4U FrameCount );

      //-----------------------------------------------------------------
      /// \brief Iterate over contents.
      ///
      /// \param[in] Info
      ///     Specifies the type of information to be searched.
      ///
      /// \param[in] Action
      ///     Action to be taken for each piece of information found.
      //-----------------------------------------------------------------
      void forEach( Common::FrTOC::query_info_type Info,
		    Common::FrTOC::FunctionBase& Action ) const;

      //-----------------------------------------------------------------
      /// \brief Write the structure to the stream
      ///
      /// \param[in] Stream
      ///     The output stream where the object is to be written.
      //-----------------------------------------------------------------
      void write( Common::OStream& Stream ) const;
    };

    inline Common::FrameSpec::size_type FrTOCProcData::
    Bytes( const Common::StreamBase& Stream ) const
    {
      Common::FrameSpec::size_type 
	retval = sizeof( nproc_type );
      if ( m_info.size( ) )
      {
	retval
	  += ( m_info.size( )
	       * ( sizeof( position_type )
		   * m_info.begin( )->second.size( )
		   ) );
      }
      for ( MapProc_type::const_iterator
	      cur = m_info.begin( ),
	      last = m_info.end( );
	    cur != last;
	    ++cur )
      {
	retval += cur->first.Bytes( );
      }
      return retval;
    }

    template< typename SE >
    void FrTOCProcData::
    Description( Common::Description& Desc )
    {
      Desc( SE( "nProc", "INT_4U",
		"Number of unique FrProcData names in file." ) );
      Desc( SE( "nameProc", "STRING[nProc]", "Array of FrProcData names" ) );
      Desc( SE( "positionProc", "INT_8U[nProc][nFrame]",
		"Array of lists of FrProcData offset positions, in bytes,"
		" from beginning of file (size of nFrame*nProc)"
		) );
    }

    inline const FrTOCProcData::key_container_type& FrTOCProcData::
    GetKeys( ) const
    {
      return m_keys;
    }

    inline const FrTOCProcData::MapProc_type& FrTOCProcData::
    GetProc( ) const
    {
      return m_info;
    }

  } // namespace - Version_8
} // namespace - FrameCPP

#endif /* FrameCPP_VERSION_8_FrTOCProcData_HH */
