#include <stdlib.h>
#include <string.h>
#include <iostream>
#include <list>
#include <map>

#include "valgrind/valgrind.h"

#include "ldastoolsal/MemChecker.hh"
#include "ldastoolsal/mutexlock.hh"
#include "ldastoolsal/ReadWriteLock.hh"

#define EXITING_LOCK 1

namespace
{
  using LDASTools::AL::MemChecker;
  using LDASTools::AL::MutexLock;
  using LDASTools::AL::ReadWriteLock;

  //---------------------------------------------------------------------
  // This code is derived from:
  //  http://www.drdobbs.com/the-standard-librarian-what-are-allocato/184403759?pgno=1
  //
  //---------------------------------------------------------------------
  template <class T> class malloc_allocator
  {
  public:
    typedef T                 value_type;
    typedef value_type*       pointer;
    typedef const value_type* const_pointer;
    typedef value_type&       reference;
    typedef const value_type& const_reference;
    typedef std::size_t       size_type;
    typedef std::ptrdiff_t    difference_type;
  
    template <class U> 
    struct rebind {
      typedef malloc_allocator<U> other;
    };

    malloc_allocator() {}
    malloc_allocator(const malloc_allocator&)
    {
    }

    template <class U> 
    malloc_allocator(const malloc_allocator<U>&)
    {
    }

    ~malloc_allocator()
    {
    }

    pointer address(reference x) const
    {
      return &x;
    }

    const_pointer address(const_reference x) const
    { 
      return x;
    }

    pointer allocate(size_type n, const_pointer = 0)
    {
      void* p = std::malloc(n * sizeof(T));
      if (!p)
	throw std::bad_alloc();
      return static_cast<pointer>(p);
    }

    void deallocate(pointer p, size_type)
    {
      std::free(p);
    }

    size_type max_size() const
    { 
      return static_cast<size_type>(-1) / sizeof(T);
    }

    void construct(pointer p, const value_type& x)
    { 
      new(p) value_type(x); 
    }

    void destroy(pointer p)
    {
      p->~value_type();
    }

  private:
    void operator=(const malloc_allocator&);
  };

  template<> class malloc_allocator<void>
  {
    typedef void        value_type;
    typedef void*       pointer;
    typedef const void* const_pointer;

    template <class U> 
    struct rebind { typedef malloc_allocator<U> other; };
  };


  template <class T>
  inline bool operator==(const malloc_allocator<T>&, 
			 const malloc_allocator<T>&)
  {
    return true;
  }

  template <class T>
  inline bool operator!=(const malloc_allocator<T>&, 
			 const malloc_allocator<T>&)
  {
    return false;
  }

  //---------------------------------------------------------------------
  //
  //---------------------------------------------------------------------
  class GCQueueNode
  {
  public:
    ~GCQueueNode( )
    {
      while( data_container.size( ) > 0 )
      {
	data_container.back( ).purge( );
	data_container.pop_back( );
      }
    }

    void Append( MemChecker::CleanupFunction Function,
		 const char* Name, bool Always )
    {
      if ( RUNNING_ON_VALGRIND || Always )
      {
	data_container.push_back( data_type( Function, Name ) );
      }
    }

    void
    Info( const char* Leader ) const
    {
      for ( data_container_type::const_iterator
	      cur = data_container.begin( ),
	      last = data_container.end( );
	    cur != last;
	    ++cur )
      {
	std::cerr << Leader << cur->name
		  << std::endl
		  << std::flush
	  ;
      }
    }

  private:
    class func_node {
    public:
      typedef MemChecker::CleanupFunction function_type;

      function_type	function;
      const char*	name;

      func_node( function_type Function,
		 const char* Name )
	: function( Function ),
	  name( ::strdup( Name ) )
      {
      }

      void
      purge( )
      {
	if ( function )
	{
	  if ( MemChecker::Trigger::IsRegistered( ) )
	  {
	    //-----------------------------------------------------------
	    // Cleanup is being called within main() so it is safe.
	    // If not called from within main(), then shared
	    // objects may be unloaded before reaching here.
	    //-----------------------------------------------------------
	    (*function)();
	  }
	  function = (function_type)NULL;
	}
	if ( name )
	{
	  ::free( const_cast< char* >( name ) );
	  name = (const char*)NULL;
	}
      }
    };

    typedef func_node	data_type;
    typedef std::list< data_type, malloc_allocator< data_type > >
    data_container_type;

    data_container_type	data_container;
  };

  class GCQueue
  {
  public:
    GCQueue( )
      :
#if EXITING_LOCK
        is_exiting_baton( /*__FILE__, __LINE__,*/ false ),
#endif /* EXITING_LOCK */
	gc_queue_baton( __FILE__, __LINE__, false )
    {
    }

    inline void
    Append( MemChecker::CleanupFunction Function,
	    const char* Name,
	    int Ring,
	    bool Always )
    {
      MutexLock	l( gc_queue_baton,
		   __FILE__, __LINE__ );

      if ( IsExiting( ) )
      {
	(*Function)( );
	return;
      }
      gc_queue[ Ring ].Append( Function, Name, Always );
      
    }

    inline void
    Cleanup( )
    {
      MutexLock	l( gc_queue_baton,
		   __FILE__, __LINE__ );

      bool			state;

      state = IsExiting( true );

      while( gc_queue.size( ) > 0 )
      {
	gc_queue_type::iterator
	  cur = gc_queue.end( );
	--cur;
	gc_queue.erase( cur );
      }

      IsExiting( state );

    }

    inline void
    Info( ) const
    {
      //-----------------------------------------------------------------
      // Print the queue in the order that it will be deallocated
      //-----------------------------------------------------------------
      std::cerr << "Exiting Status: " << is_exiting
		<< std::endl
		<< std::flush
	;
      for ( gc_queue_type::const_reverse_iterator
	      cur = gc_queue.rbegin( ),
	      last = gc_queue.rend( );
	    cur != last;
	    ++cur )
      {
	std::cerr << "Exit Queue Level: " << cur->first
		  << std::endl
		  << std::flush
	  ;
	cur->second.Info( "\t" );
      }
    }

    inline bool
    IsExiting( ) const
    {
#if EXITING_LOCK
      ReadWriteLock	l( is_exiting_baton,
			   ReadWriteLock::READ,
			   __FILE__, __LINE__ );
#endif /* EXIING_LOCK */

      return is_exiting;
    }

    inline bool
    IsExiting( bool Value )
    {
#if EXITING_LOCK
      ReadWriteLock	l( is_exiting_baton,
				   ReadWriteLock::WRITE,
				   __FILE__, __LINE__ );
#endif /* EXITING_LOCK */

      bool retval = is_exiting;

      is_exiting = Value;

      return retval;
    }

  private:
    typedef std::map< int, GCQueueNode, std::less< int >,
		      malloc_allocator< std::pair< int, GCQueueNode > > >
    gc_queue_type;

    bool			is_exiting;
#if EXITING_LOCK
    ReadWriteLock::baton_type	is_exiting_baton;
#endif /* EXITING_LOCK */

    gc_queue_type		gc_queue;
    MutexLock::baton_type	gc_queue_baton;
  };

  GCQueue&
  gc_queue( )
  {
    static GCQueue	eq;

    return eq;
  }
} // namespace - anonymous

namespace LDASTools
{
  namespace AL
  {
    bool MemChecker::Trigger::registered = false;

    MemChecker::Trigger::
    Trigger( bool Final )
      : final( Final )
    {
      registered = true;
    }

    MemChecker::Trigger::
    ~Trigger( )
    {
      DoGarbageCollection( );
    }

    void MemChecker::Trigger::
    DoGarbageCollection( ) const
    {
      if ( registered )
      {
	if ( final )
	{
	  MemChecker::is_exiting( true );
	}
	MemChecker::cleanup( );
	registered = false;
      }
    }

    //-------------------------------------------------------------------
    /// \note
    ///     On systems that support cleaning of the memory prior to exit,
    ///     setting the environment variable 'MEMCHECK_GARBAGE_COLLECTION'
    ///     prevents the system from registering cleanup helpers.
    //-------------------------------------------------------------------
    void MemChecker::
    Append( MemChecker::CleanupFunction Function,
	    const std::string& Name,
	    int Ring,
	    bool Always )
    {
      static bool init = false;
      static bool do_memcheck = true;

      if ( ! init )
      {
	const char* val = ::getenv( "MEMCHECK_GARBAGE_COLLECTION" );
	if ( val )
	{
	  switch( *val )
	  {
	  case '0':
	  case 'f':
	  case 'F':
	    do_memcheck = false;
	    break;
	  default:
	    do_memcheck = (RUNNING_ON_VALGRIND);
	  }
	}
	init = true;
      }
      if ( do_memcheck )
      {
	gc_queue( ).Append( Function, Name.c_str( ), Ring, Always );
      }
    }

    bool MemChecker::
    IsExiting( )
    {
      return gc_queue( ).IsExiting( );
    }

    void MemChecker::
    Info( )
    {
      gc_queue( ).Info( );
    }

    void MemChecker::
    cleanup( )
    {
      gc_queue( ).Cleanup( );
    }

    void MemChecker::
    is_exiting( bool Value )
    {
      gc_queue( ).IsExiting( Value );
    }

  } // namespace - AL
} // namespace - LDASTools
