#ifndef GENERIC_API__LSTAT_POOL_HH
#define GENERIC_API__LSTAT_POOL_HH

#include <list>
#include <iostream>

#include "ldastoolsal/ErrorLog.hh"
#include "ldastoolsal/Memory.hh"
#include "ldastoolsal/mutexlock.hh"

namespace GenericAPI
{

  template< typename StatInfoType >
  class StatPoolT
  {
  public:
    typedef typename LDASTools::AL::SharedPtr< StatInfoType > element_type;

    static void Cleanup( );

    static void Destroy( element_type Buffer );

    static std::ostream& Dump( std::ostream& Stream );

    static bool Interuptable( );

    static element_type Request( );

    static void Release( element_type Buffer );

    static void Trash( element_type Buffer );

  private:
    typedef std::list< element_type > pool_type;

    LDASTools::AL::MutexLock::baton_type	pool_lock;
    pool_type					available;
    pool_type					in_use;
    pool_type					trash_;

    void cleanup( );

    void destroy( element_type Buffer );

    std::ostream& dump( std::ostream& Stream );

    void release( element_type Buffer );

    void trash( element_type Buffer );

    element_type request( );
  };

  template< typename StatInfoType >
  void StatPoolT< StatInfoType >::
  Cleanup( )
  {
    LDASTools::AL::Singleton< StatPoolT< StatInfoType > >::Instance( ).cleanup( );
  }
  
  template< typename StatInfoType >
  bool StatPoolT< StatInfoType >::
  Interuptable( )
  {
    return true;
  }

  template< typename StatInfoType >
  typename StatPoolT< StatInfoType >::element_type StatPoolT< StatInfoType >::
  Request( )
  {
    return LDASTools::AL::Singleton< StatPoolT< StatInfoType > >::Instance( ).request( );
  }

  template< typename StatInfoType >
  void StatPoolT< StatInfoType >::
  Release( element_type Buffer )
  {
    LDASTools::AL::Singleton< StatPoolT< StatInfoType > >::Instance( ).release( Buffer );
  }

  template< typename StatInfoType >
  void StatPoolT< StatInfoType >::
  Trash( element_type Buffer )
  {
    LDASTools::AL::Singleton< StatPoolT< StatInfoType > >::Instance( ).trash( Buffer );
  }

  template< typename StatInfoType >
  void StatPoolT< StatInfoType >::
  Destroy( element_type Buffer )
  {
    LDASTools::AL::Singleton< StatPoolT< StatInfoType > >::Instance( ).destroy( Buffer );
  }

  template< typename StatInfoType >
  std::ostream& StatPoolT< StatInfoType >::
  Dump( std::ostream& Stream )
  {
    return LDASTools::AL::Singleton< StatPoolT< StatInfoType > >::Instance( ).dump( Stream );
  }

  template< typename StatInfoType >
  void StatPoolT< StatInfoType >::
  cleanup( )
  {
    using namespace LDASTools::AL;

    static bool has_been_cleaned = false;

    if ( has_been_cleaned == false )
    {
      MutexLock	lock( pool_lock,
		      __FILE__, __LINE__ );

      if ( has_been_cleaned == false )
      {
	if ( StdErrLog.IsOpen( ) )
	{
	  std::ostringstream	msg;
      
	  msg << "StatPool::~StatPool";
	  StdErrLog( ErrorLog::DEBUG,
		     __FILE__, __LINE__,
		     msg.str( ) );
	}
	available.erase( available.begin( ), available.end( ) );
	in_use.erase( in_use.begin( ), in_use.end( ) );
	has_been_cleaned = true;
      } // if ( has_been_cleaned == false ) (after Mutex lock)
    } // if ( has_been_cleaned == false )
  }

  template< typename StatInfoType >
  void StatPoolT< StatInfoType >::
  destroy( element_type Buffer )
  {
    using namespace LDASTools::AL;

    MutexLock	lock( pool_lock, __FILE__, __LINE__ );

    if ( StdErrLog.IsOpen( ) )
    {
      std::ostringstream	msg;
      
      msg << "StatPool::destroy: Buffer: " << (void*)(Buffer.get( ));
      StdErrLog( ErrorLog::DEBUG,
		 __FILE__, __LINE__,
		 msg.str( ) );
    }
    //-------------------------------------------------------------------
    // Take out of m_in_use list
    //-------------------------------------------------------------------
    typename pool_type::size_type	size = in_use.size( );

    /* erase the element */
    in_use.remove( Buffer );
    if ( size != in_use.size( ) )
    {
      //-----------------------------------------------------------------
      // Destroy the process
      //-----------------------------------------------------------------
      if ( StdErrLog.IsOpen( ) )
      {
	std::ostringstream	msg;
      
	msg << "StatPool::release: deleting: Buffer: " << (void*)(Buffer.get( ));
	StdErrLog( ErrorLog::DEBUG,
		   __FILE__, __LINE__,
		   msg.str( ) );
      }
    }
  }

  template< typename StatInfoType >
  std::ostream& StatPoolT< StatInfoType >::
  dump( std::ostream& Stream )
  {
    using namespace LDASTools::AL;

    MutexLock	lock( pool_lock, __FILE__, __LINE__ );

    typename pool_type::size_type	used = in_use.size( );
    typename pool_type::size_type	avail = available.size( );

    Stream << "Queue: " << used << " of " << ( used + avail ) << " slots are used" << std::endl;

    for ( typename pool_type::const_iterator
	    cur = in_use.begin( ),
	    last = in_use.end( );
	  cur != last;
	  ++cur )
    {
      Stream << "\tStatting directory: " << *cur << std::endl;
    }
    return Stream;
  }

  template< typename StatInfoType >
  void StatPoolT< StatInfoType >::
  release( element_type Buffer )
  {
    using namespace LDASTools::AL;

    MutexLock	lock( pool_lock, __FILE__, __LINE__ );

    if ( StdErrLog.IsOpen( ) )
    {
      std::ostringstream	msg;
      
      msg << "StatPool::release";
      StdErrLog( ErrorLog::DEBUG,
		 __FILE__, __LINE__,
		 msg.str( ) );
    }
    //-------------------------------------------------------------------
    // Take out of in_use list
    //-------------------------------------------------------------------
    typename pool_type::size_type size = in_use.size( );

    /* erase the element */
    in_use.remove( Buffer );
    if ( size != in_use.size( ) )
    {
      //-----------------------------------------------------------------
      // Put into available queue
      //-----------------------------------------------------------------
      if ( StdErrLog.IsOpen( ) )
      {
	std::ostringstream	msg;
      
	msg << "StatPool::release: returned to a available pool";
	StdErrLog( ErrorLog::DEBUG,
		   __FILE__, __LINE__,
		   msg.str( ) );
      }
      if ( ! MemChecker::IsExiting( ) )
      {
	available.push_back( Buffer );
      }
    }
  }

  template< typename StatInfoType >
  typename StatPoolT< StatInfoType >::element_type StatPoolT< StatInfoType >::
  request( )
  {
    using namespace LDASTools::AL;

    MutexLock	lock( pool_lock, __FILE__, __LINE__ );
    element_type	retval;

    if ( available.size( ) == 0 )
    {
      //-----------------------------------------------------------------
      // Create new thread
      //-----------------------------------------------------------------
      retval.reset( new typename element_type::element_type );

      if ( StdErrLog.IsOpen( ) )
      {
	std::ostringstream	msg;
	
	msg << "StatPool::request: Created new";
	StdErrLog( ErrorLog::DEBUG,
		   __FILE__, __LINE__,
		   msg.str( ) );
      }
    }
    else
    {
      //-----------------------------------------------------------------
      // Take one from the available list
      //-----------------------------------------------------------------
      retval = available.front( );
      available.pop_front( );
      if ( StdErrLog.IsOpen( ) )
      {
	std::ostringstream	msg;
	
	msg << "StatPool::request: Reusing";
	StdErrLog( ErrorLog::DEBUG,
		   __FILE__, __LINE__,
		   msg.str( ) );
      }
    }
    in_use.push_back( retval );
    return retval;
  }

  template< typename StatInfoType >
  void StatPoolT< StatInfoType >::
  trash( element_type Buffer )
  {
    using namespace LDASTools::AL;

    MutexLock	lock( pool_lock, __FILE__, __LINE__ );

    if ( StdErrLog.IsOpen( ) )
    {
      std::ostringstream	msg;
      
      msg << "StatPool::trash";
      StdErrLog( ErrorLog::DEBUG,
		 __FILE__, __LINE__,
		 msg.str( ) );
    }
    //-------------------------------------------------------------------
    // Take out of in_use list
    //-------------------------------------------------------------------
    typename pool_type::size_type size = in_use.size( );

    /* erase the element */
    in_use.remove( Buffer );
    if ( size != in_use.size( ) )
    {
      //-----------------------------------------------------------------
      // Put into available queue
      //-----------------------------------------------------------------
      if ( StdErrLog.IsOpen( ) )
      {
	std::ostringstream	msg;
      
	msg << "StatPool::trash: trashing";
	StdErrLog( ErrorLog::DEBUG,
		   __FILE__, __LINE__,
		   msg.str( ) );
      }
      if ( ! MemChecker::IsExiting( ) )
      {
	trash_.push_back( Buffer );
      }
    }
  }

}

namespace std
{
  template< typename StatInfoType > ostream&
  operator<<( ostream& Stream,	const GenericAPI::StatPoolT< StatInfoType >& Pool )
  {
    return Pool.Dump( Stream );
  }
}
#endif /* GENERIC_API__LSTAT_POOL_HH */
