#include <ldas_tools_config.h>

#include "regex.hh"
#include "errors.hh"

using LDASTools::Errors::getErrorString;

//-----------------------------------------------------------------------------
/// \brief  Constructor
///
/// This constructor compiles the given regular expression.
///
///
Regex::Regex( )
  : mRegex( ),
    mPattern( ),
    mFlags( REG_EXTENDED ),
    mCompiled( false )
{
}

//-----------------------------------------------------------------------------
/// \brief  Constructor
///
/// This constructor compiles the given regular expression.
///
/// \param regex
/// \param cflags
///
/// \exception LdasException
///
Regex::
Regex( const std::string& regex, int cflags )
  : mRegex( regex ),
    mPattern( ),
    mFlags( cflags ),
    mCompiled( false )
{
  compile( );
}

//-----------------------------------------------------------------------------
/// \brief  Constructor
///
/// This constructor compiles the given regular expression.
///
/// \param regex
/// \param cflags
///
/// \exception LdasException
///
Regex::
Regex( const char* regex, int cflags )
  : mRegex( regex ),
    mPattern( ),
    mFlags( cflags ),
    mCompiled( false )
{
  compile( );
}

//-----------------------------------------------------------------------------
/// \brief  Constructor
///
/// This constructor compiles the given regular expression.
///
/// \param Source
///
/// \exception LdasException
///
Regex::
Regex( const Regex& Source )
  : mRegex( Source.mRegex ),
    mPattern( ),
    mFlags( Source.mFlags ),
    mCompiled( false )
{
  compile( );
}


//-----------------------------------------------------------------------------
/// \brief  Destructor.
///
/// This just frees the compiled regex pattern.
///
Regex::
~Regex() throw()
{
  if ( mCompiled )
  {
    regfree( &mPattern );
  }
}

Regex& Regex::
operator=( const Regex& Source )
{
  mRegex = Source.mRegex;
  mFlags = Source.mFlags;
  mCompiled = false;
  compile( );
  return *this;
}


void Regex::
compile( )
{
  if ( mCompiled )
  {
    mCompiled = false;
    regfree( &mPattern );
  }
  if ( mRegex.length( ) > 0 )
  {
    int error = regcomp( &mPattern, mRegex.c_str(), mFlags );
    if ( error != 0 )
    {
      size_t msgSize( regerror( error, &mPattern, 0, 0 ) );
      char* msg = new char[ msgSize ];
      regerror( error, &mPattern, msg, msgSize );
        
      regfree( &mPattern );

      LdasException e( Library::GENERAL, LDASTools::Errors::INVALID_REGEX,
		       getErrorString( LDASTools::Errors::INVALID_REGEX ),
		       msg,
		       __FILE__, __LINE__ );
      delete[] msg;
      msg = 0;

      throw e;
    }
    mCompiled = true;
  }
}
