#include <signal.h>

#include "SignalHandler.hh"

SINGLETON_INSTANCE_DEFINITION(SingletonHolder<LDASTools::AL::SignalHandler>)

typedef int os_signal_type;

HASH_NAMESPACE_BEGIN
{
  template<>
    struct hash<LDASTools::AL::SignalHandler::signal_type>
  {
    size_t
    operator()( LDASTools::AL::SignalHandler::signal_type Key ) const
    {
      return static_cast<size_t>( Key );
    }
  };
}
HASH_NAMESPACE_END

namespace
{
  static const os_signal_type OS_SIGNAL_UNKNOWN = 0;

  os_signal_type os_signal( LDASTools::AL::SignalHandler::signal_type Signal );

  LDASTools::AL::SignalHandler::signal_type
  os_reverse_signal( os_signal_type Signal );

  void handler( os_signal_type Signal );
}

namespace LDASTools
{
  namespace AL
  {
    SignalHandler::Callback::
    ~Callback( )
    {
    }

    void SignalHandler::
    Callbacks( signal_type Signal )
    {
      Instance( ).callbacks( Signal );
    }

    void SignalHandler::
    Register( Callback* CB, signal_type Signal )
    {
      Instance( ).reg( CB, Signal );
    }

    void SignalHandler::
    Unregister( Callback* CB, signal_type Signal )
    {
      Instance( ).unreg( CB, Signal );
    }

    void SignalHandler::
    callbacks( signal_type Signal )
    {
      callbacks_type& callbacks = signal_queue[ Signal ];

      for ( callbacks_type::iterator
	      cur = callbacks.begin( ),
	      last = callbacks.end( );
	    cur != last;
	    ++cur )
      {
	(*cur)->SignalCallback( Signal );
      }
      
    }

    //-------------------------------------------------------------------
    /// Register a callback
    //-------------------------------------------------------------------
    void SignalHandler::
    reg( Callback* CB, signal_type Signal )
    {
      callbacks_type& callbacks = signal_queue[ Signal ];

      if ( callbacks.size( ) == 0 )
      {
	//---------------------------------------------------------------
	// Need to register handler since there is there will be 
	// something to handle
	//---------------------------------------------------------------
	struct sigaction	sa;

	sa.sa_handler = handler;
	sigemptyset( &sa.sa_mask );
	sa.sa_flags = SA_RESTART;

	if ( sigaction( os_signal( Signal ), &sa, NULL ) == -1 )
	{
	  std::cerr << "Unable to establish signal handler"
		    << std::endl
	    ;
	}
      }
      for ( callbacks_type::iterator
	      cur = callbacks.begin( ),
	      last = callbacks.end( );
	    cur != last;
	    ++cur )
      {
	if ( *cur == CB )
	{
	  //-------------------------------------------------------------
	  // Already in the queue
	  //-------------------------------------------------------------
	  return;
	}
      }
      //-----------------------------------------------------------------
      // Add to list
      //-----------------------------------------------------------------
      callbacks.push_back( CB );
    }

    //-------------------------------------------------------------------
    /// Unregister a callback
    //-------------------------------------------------------------------
    void SignalHandler::
    unreg( Callback* CB, signal_type Signal )
    {
      callbacks_type& callbacks = signal_queue[ Signal ];

      for ( callbacks_type::iterator
	      cur = callbacks.begin( ),
	      last = callbacks.end( );
	    cur != last;
	    ++cur )
      {
	if ( *cur == CB )
	{
	  //-------------------------------------------------------------
	  // Found the callback. Remove it and break out of the loop.
	  //-------------------------------------------------------------
	  callbacks.erase( cur );
	  break;
	}
      }
      if ( callbacks.size( ) == 0 )
      {
	//---------------------------------------------------------------
	// Need to remove the signal handler since there is no
	// reason to keep processing
	//---------------------------------------------------------------
	struct sigaction	sa;

	sa.sa_handler = SIG_DFL;
	sigemptyset( &sa.sa_mask );
	sa.sa_flags = SA_RESTART;

	if ( sigaction( os_signal( Signal ), &sa, NULL ) == -1 )
	{
	  std::cerr << "Unable to establish signal handler"
		    << std::endl
	    ;
	}
      }
    }

  } // namespace - AL
} // namespace - LDASTools

namespace
{
  using LDASTools::AL::SignalHandler;

  typedef LDASTools::AL::unordered_map< SignalHandler::signal_type,
					os_signal_type >
  os_signal_map_type;

  typedef LDASTools::AL::unordered_map< os_signal_type,
					SignalHandler::signal_type >
  os_reverse_signal_map_type;

  os_signal_type
  os_signal( SignalHandler::signal_type Signal )
  {
    static os_signal_map_type sm;

    if ( sm.empty( ) )
    {
      sm[ SignalHandler::SIGNAL_HANGUP ] = SIGHUP;
    }
    os_signal_map_type::iterator pos = sm.find( Signal );
    if ( pos == sm.end( ) )
    {
      return OS_SIGNAL_UNKNOWN;
    }
    return pos->second;
  }

  SignalHandler::signal_type
  os_reverse_signal( os_signal_type Signal )
  {
    static os_reverse_signal_map_type sm;

    if ( sm.empty( ) )
    {
      sm[ SIGHUP ] = SignalHandler::SIGNAL_HANGUP;
    }
    os_reverse_signal_map_type::const_iterator pos = sm.find( Signal );
    if ( pos == sm.end( ) )
    {
      return SignalHandler::SIGNAL_UNKNOWN;
    }
    return pos->second;
  }

  void handler( os_signal_type Signal )
  {
    SignalHandler::Instance( ).Callbacks( os_reverse_signal( Signal ) );
  }
}
