#ifndef GENERIC_API__LOGGING_HH
#define GENERIC_API__LOGGING_HH

#include <string>

#include "ldastoolsal/gpstime.hh"
#include "ldastoolsal/SharedPtr.hh"

#include "genericAPI/LogLDAS.hh"

namespace GenericAPI
{
  typedef LDASTools::AL::SharedPtr< GenericAPI::Log::LDAS > log_type;

  typedef Log::LDAS LogEntryGroup_type;

  //---------------------------------------------------------------------
  /// \brief Retrieve the default log file
  //---------------------------------------------------------------------
  const std::string&
  LogFileDefault( );

  //---------------------------------------------------------------------
  /// \brief Check if the message would be logged
  ///
  /// \param[in] Group
  ///     The group classification of the message to be logged.
  /// \param[in] Level
  ///     The level at which the message will be logged.
  /// \param[in] Filename
  ///     The filename of the stream to send the message.
  ///
  /// \return
  ///     True if a message would be logged for the give Group at the
  ///     given Level; false otherwise.
  //---------------------------------------------------------------------
  bool
  IsLogging( const int Group,
	     const int Level,
	     const std::string& Filename = LogFileDefault( ) );

  //---------------------------------------------------------------------
  /// \brief Queue a message to be written to a log file
  ///
  /// \param[in] Message
  ///     The text of the message to be written to the log file.
  /// \param[in] Group
  ///     The group classification of the message to be logged.
  /// \param[in] Level
  ///     The level at which the message will be logged.
  /// \param[in] Caller
  ///     The function or method name of the routine logging the message.
  /// \param[in] JobInfo
  ///     Descriptive text of job logging the message.
  /// \param[in] Time
  ///     The time the message was logged.
  /// \param[in] Filename
  ///     The filename of the stream to send the message.
  //---------------------------------------------------------------------
  void
  queueLogEntry( const std::string& Message,
		 const std::string& Group,
		 const int Level,
		 const std::string& Caller,
		 const std::string& JobInfo,
		 const unsigned int Time = 0,
		 const std::string& Filename = LogFileDefault( ) );

  //---------------------------------------------------------------------
  /// \brief Queue a message to be written to a log file
  ///
  /// \param[in] Message
  ///     The text of the message to be written to the log file.
  /// \param[in] Group
  ///     The group classification of the message to be logged.
  /// \param[in] Level
  ///     The level at which the message will be logged.
  /// \param[in] Caller
  ///     The function or method name of the routine logging the message.
  /// \param[in] JobInfo
  ///     Descriptive text of job logging the message.
  /// \param[in] Time
  ///     The time the message was logged.
  /// \param[in] Filename
  ///     The filename of the stream to send the message.
  //---------------------------------------------------------------------
  void
  queueLogEntry( const std::string& Message,
		 const int Group,
		 const int Level,
		 const std::string& Caller,
		 const std::string& JobInfo,
		 const unsigned int Time = 0,
		 const std::string& Filename = LogFileDefault( ) );

  //---------------------------------------------------------------------
  /// \brief Establish level of debugging output
  //---------------------------------------------------------------------
  void
  setLogDebugLevel( const int Level,
		    const std::string& Filename = std::string( "" ) );

  //---------------------------------------------------------------------
  /// \brief Establish the directory in which to write logging information
  //---------------------------------------------------------------------
  void
  setLogOutputDirectory( const std::string& OutputDirectory );

  //---------------------------------------------------------------------
  /// \brief Establish how to format logging messages
  //---------------------------------------------------------------------
  void SetLogFormatter( const GenericAPI::Log::LDAS* Formatter );

  //---------------------------------------------------------------------
  /// \brief Gain access to the current log formatter.
  //---------------------------------------------------------------------
  log_type LogFormatter( const std::string& Filename = LogFileDefault( ) );

  //---------------------------------------------------------------------
  /// \brief Establish tag used to identify logging set
  //---------------------------------------------------------------------
  void
  setLogTag( const std::string& Tag );

  //---------------------------------------------------------------------
  /// \brief Establish the default output file for logging
  //---------------------------------------------------------------------
  void LogFileDefault( const std::string& LogFilename );
}

#define QUEUE_LOG_MESSAGE( MSG, GROUP, LEVEL, CALLER, JOBINFO ) \
  if ( GenericAPI::IsLogging( GenericAPI::LogEntryGroup_type:: GROUP, LEVEL ) ) \
  { \
    std::ostringstream msg; \
    msg << MSG; \
    GenericAPI::queueLogEntry( msg.str( ), \
			       GenericAPI::LogEntryGroup_type:: GROUP,	\
                               LEVEL, \
			       CALLER, \
			       JOBINFO );	\
  }

#endif /* GENERIC_API__LOGGING_HH */

