#ifndef FrameCPP_VERSION_8_FrEndOfFile_HH
#define FrameCPP_VERSION_8_FrEndOfFile_HH

#include "ldastoolsal/types.hh"

#include "framecpp/Version8/FrameSpec.hh"

#include "framecpp/Common/FrEndOfFile.hh"

namespace FrameCPP
{
  namespace Version_8
  {
    //-------------------------------------------------------------------
    /// \brief End of File data structure
    //-------------------------------------------------------------------
    class FrEndOfFile
      : public Common::FrEndOfFile
    {
    public:
      typedef INT_4U nFrames_type;
      typedef INT_8U nBytes_type;
      typedef INT_8U seekTOC_type;
      typedef INT_4U chkSumFrHeader_type;
      typedef INT_4U chkSum_type;
      typedef INT_4U chkSumFile_type;

      //-----------------------------------------------------------------
      /// \brief Default constructor
      ///
      /// \return
      ///     A new instance of this object.
      //-----------------------------------------------------------------
      FrEndOfFile( );

      //-----------------------------------------------------------------
      /// \brief Destructor.
      //-----------------------------------------------------------------
      virtual ~FrEndOfFile( );

      //-----------------------------------------------------------------
      /// \brief The name structure name of this object.
      ///
      /// \return
      ///     The name of the structure as specified by the frame
      ///     specification.
      //-----------------------------------------------------------------
      static const char* StructName( );

      //-----------------------------------------------------------------
      /// \brief The description of structure
      ///
      /// \return
      ///     A Description object which describes this structure as
      ///     specified by the frame specification.
      //-----------------------------------------------------------------
      static const Common::Description* StructDescription( );

      //-----------------------------------------------------------------
      /// \brief Number of bytes needed to write this structure
      ///
      /// \param[in] Stream
      ///     The stream from which to the object is being read or
      ///     written.
      ///
      /// \return
      ///     The number of bytes need to read or write this object.
      //-----------------------------------------------------------------
      virtual Common::FrameSpec::size_type
      Bytes( const Common::StreamBase& Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Virtual copy constructor
      //-----------------------------------------------------------------
      virtual FrEndOfFile* Clone( ) const;

      //-----------------------------------------------------------------
      /// \brief Virtual constructor
      ///
      /// \param[in] Stream
      ///     The input stream from where the object is being read.
      //-----------------------------------------------------------------
      virtual FrEndOfFile* Create( istream_type& Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the expected size of the stream
      //-----------------------------------------------------------------
      virtual nBytes_cmn_type NBytes( ) const;

      //-----------------------------------------------------------------
      /// \brief The name structure name of this object.
      ///
      /// \return
      ///     The name of the structure as specified by the frame
      ///     specification.
      //-----------------------------------------------------------------
      virtual const char* ObjectStructName( ) const;

      //-----------------------------------------------------------------
      /// \brief Write the structure to the stream
      ///
      /// \param[in] Stream
      ///     The output stream where the object is to be written.
      //-----------------------------------------------------------------
      virtual void Write( ostream_type& Stream ) const;

      //-----------------------------------------------------------------
      /// \brief equality operator
      ///
      /// \param[in] RHS
      ///     The FrAdcData object to be compared.
      ///
      /// \return
      ///     The value true is returned if this object is equivelent
      ///     to the RHS instance; false otherwise.
      //-----------------------------------------------------------------
      bool operator==( const FrEndOfFile& RHS ) const;

      //-----------------------------------------------------------------
      /// \brief equality operator for abstract data type
      ///
      /// \param[in] RHS
      ///     The object to be compared.
      ///
      /// \return
      ///     The value true is returned if this object is equivelent
      ///     to the RHS instance; false otherwise.
      //-----------------------------------------------------------------
      virtual bool operator==( const Common::FrameSpec::Object& RHS ) const;

      static const Common::FrameSpec::Info::frame_object_types s_object_id
      = Common::FrameSpec::Info::FSI_FR_END_OF_FILE;

      //-----------------------------------------------------------------
      /// \brief Retrieve the checksum.
      ///
      /// \return
      ///     The checksum.
      //-----------------------------------------------------------------
      FrEndOfFile::chkSum_type GetChkSum( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the file checksum.
      ///
      /// \return
      ///     The file checksum.
      //-----------------------------------------------------------------
      FrEndOfFile::chkSumFile_type GetChkSumFile( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the FrHeader checksum.
      ///
      /// \return
      ///     The FrHEader checksum.
      //-----------------------------------------------------------------
      FrEndOfFile::chkSumFrHeader_type GetChkSumFrHeader( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the total number of bytes in this file.
      ///
      /// \return
      ///     The total number of bytes in this file.
      //-----------------------------------------------------------------
      FrEndOfFile::nBytes_type GetNBytes( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the number of frames in this file.
      ///
      /// \return
      ///     The number of frames in this file.
      //-----------------------------------------------------------------
      FrEndOfFile::nFrames_type GetNFrames( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the location of the table of contents.
      ///
      /// \return
      ///     The location of the table of contents.
      //-----------------------------------------------------------------
      FrEndOfFile::seekTOC_type GetSeekTOC( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the location of the table of contents.
      ///
      /// \return
      ///     The location of the table of contents.
      ///
      /// \todo
      ///     This may be better placed in the protected section.
      //-----------------------------------------------------------------
      virtual FrEndOfFile::seekTOC_cmn_type SeekTOC( ) const;

      //-----------------------------------------------------------------
      /// \brief Verify the object.
      //-----------------------------------------------------------------
      virtual void VerifyObject( Common::Verify& Verifier,
				 Common::IFrameStream& Stream ) const;

    protected:
      //-----------------------------------------------------------------
      /// Number of frames in this file
      //-----------------------------------------------------------------
      nFrames_type		m_nFrames;
      //-----------------------------------------------------------------
      /// Total number of bytes in the file; zero if not computed.
      //-----------------------------------------------------------------
      nBytes_type		m_nBytes;
      //-----------------------------------------------------------------
      /// Bytes to back up to the Beginning of the table of contents
      /// structure.
      /// If seekTOC is zero, then there is no table of contents for
      /// this file.
      //-----------------------------------------------------------------
      seekTOC_type		m_seekTOC;
      //-----------------------------------------------------------------
      /// FrHeader checksum
      //-----------------------------------------------------------------
      chkSumFrHeader_type	m_chkSumFrHeader;
      //-----------------------------------------------------------------
      /// Structure checksum.
      ///
      /// \note
      ///     The chkSum AND chkSumFile ariables are not included in the
      ///     computation of checkSum,
      ///     but chkSumFrHeader is includeed
      //-----------------------------------------------------------------
      chkSum_type		m_chkSum;
      //-----------------------------------------------------------------
      /// File checksum.
      //-----------------------------------------------------------------
      chkSumFile_type		m_chkSumFile;

      //-----------------------------------------------------------------
      /// \brief Assign the contents of the instance
      ///
      /// \param[in] Stream
      ///     An input stream from which to read information for
      ///     the instance.
      ///
      //-----------------------------------------------------------------
      virtual void assign( assign_stream_type& Stream );

      //-----------------------------------------------------------------
      /// \brief Demotes object to previous version of the frame spec
      ///
      /// \param[in] Target
      ///     The version of the frame specification to demote too.
      /// \param[in] Obj
      ///     The version of the object to demote.
      /// \param[in] Stream
      ///     The input stream from which the original object was read.
      ///
      /// \return
      ///     An object of the previous generation.
      //-----------------------------------------------------------------
      virtual demote_ret_type
      demote( INT_2U Target,
	      demote_arg_type Obj,
	      istream_type* Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Promotes object to another version of the frame spec
      ///
      /// \param[in] Target
      ///     The version of the promoted frame specification.
      /// \param[in] Obj
      ///     The object to be promoted.
      /// \param[in] Stream
      ///     The input stream from which the original object was read.
      ///
      /// \return
      ///     An object promoted to the next generation.
      //-----------------------------------------------------------------
      virtual promote_ret_type
      promote( INT_2U Target,
	       promote_arg_type Obj,
	       istream_type* Stream ) const;

    private:
      using Common::FrameSpec::Object::Create;

      FrEndOfFile( istream_type& Source );

      chkSum_cmn_type Filter( const istream_type& Stream,
			      Common::CheckSum& Filt,
			      chkType_cmn_type& Type,
			      void* Buffer,
			      INT_8U Size ) const;
    }; // class - FrEndOfFile

    inline const char* FrEndOfFile::
    StructName( )
    {
      static const CHAR* class_name( "FrEndOfFile" );
      return class_name;
    }

    inline FrEndOfFile::chkSum_type FrEndOfFile::
    GetChkSum( ) const
    {
      return m_chkSum;
    }

    inline FrEndOfFile::chkSumFile_type FrEndOfFile::
    GetChkSumFile( ) const
    {
      return m_chkSumFile;
    }

    inline FrEndOfFile::chkSumFrHeader_type FrEndOfFile::
    GetChkSumFrHeader( ) const
    {
      return m_chkSumFrHeader;
    }

    inline FrEndOfFile::nBytes_type FrEndOfFile::
    GetNBytes( ) const
    {
      return m_nBytes;
    }

    inline FrEndOfFile::nFrames_type FrEndOfFile::
    GetNFrames( ) const
    {
      return m_nFrames;
    }

    inline FrEndOfFile::seekTOC_type FrEndOfFile::
    GetSeekTOC( ) const
    {
      return m_seekTOC;
    }

    inline Common::FrEndOfFile::seekTOC_cmn_type FrEndOfFile::
    SeekTOC( ) const
    {
      return GetSeekTOC( );
    }
  } // namespace - Version_8
} // namespace - FrameCPP

#endif /* FrameCPP_VERSION_8_FrEndOfFile_HH */
