/* -*- mode: c++ -*- */
#ifndef FRAME_H_I
#define FRAME_H_I


%{
#include <string>

#include "ldastoolsal/types.hh"
#include "framecpp/FrameH.hh"
#include "framecpp/FrRawData.hh"
#include "framecpp/FrVect.hh"

using namespace FrameCPP;

%}

%import "FrDetector.i"
%import "FrVect.i"
%import "GPSTime.i"

%SharedPtr(FrameH)

%feature("autodoc",
"""
Frame Header implementation of the frame specification

  Attributes:
     name         Name of project or other experiment descrition.
                  (e.g., GEO; LIGO; VIRGO; TAMA; ... )
     run          Run number (number < 0 reserved for simulated data);
                  monotonic for experimental runs.
     frame        Frame number, monotonically increasing until end of
                  run, re-starting from 0 wit heach new run.
     dataQuality  A logical 32-bit word to denote top level quality of data.
                  Lowest order bits are reserved in pairs for the various
                  GW detectors.
     ULeapS       The integer number of leap seconds between GPS/TAI and
                  UTC in the epoch when the frame is written:
                  ULeapS = Int[TAI - UTC]. eg, ULeapS was 32 at the end
                  of 1999/07
     dt           Frame length in seconds.
""" ) FrameH;

class FrameH
{
public:
  const std::string& GetName( ) const;
  INT_4S GetRun( ) const;
  INT_4U GetFrame( ) const;
  INT_4U GetDataQuality( ) const;
  const GPSTime& GetGTime( ) const;
  INT_2U GetULeapS( ) const;
  REAL_8 GetDt( ) const;

  void SetName( const std::string& Name );
  void SetRun( INT_4S Run );
  void SetGTime( const GPSTime& Start );
  void SetDt( REAL_8 Dt );

  %extend {
    void
    AppendFrAdcData( const FrAdcData& Data )
    {
      FrameH::rawData_type raw( self->GetRawData( ) );

      if ( ! raw )
      {
	raw.reset( new FrameH::rawData_type::element_type );

	self->SetRawData( raw );
      }
      raw->RefFirstAdc( ).append( Data );
    }

    void
    AppendFrAdcData( LDAS_SHARED_PTR_QCLASS< FrAdcData > Data )
    {
      FrameH::rawData_type raw( self->GetRawData( ) );

      if ( ! raw )
      {
	raw.reset( new FrameH::rawData_type::element_type );

	self->SetRawData( raw );
      }
      raw->RefFirstAdc( ).append( Data );
    }

    void
    AppendFrDetector( const FrDetector& Data )
    {
      self->RefDetectProc( ).append( Data );
    }

    void
    AppendFrDetector( LDAS_SHARED_PTR_QCLASS< FrDetector > Data )
    {
      self->RefDetectProc( ).append( Data );
    }

    void
    AppendFrEvent( const FrEvent& Data )
    {
      self->RefEvent( ).append( Data );
    }

    void
    AppendFrHistory( const FrHistory& Data )
    {
      self->RefHistory( ).append( Data );
    }

    void
    AppendFrHistory( LDAS_SHARED_PTR_QCLASS< FrHistory > Data )
    {
      self->RefHistory( ).append( Data );
    }

    void
    AppendFrProcData( const FrProcData& Data )
    {
      self->RefProcData( ).append( Data );
    }

    void
    AppendFrProcData( LDAS_SHARED_PTR_QCLASS< FrProcData > Data )
    {
      self->RefProcData( ).append( Data );
    }

    void
    AppendFrSimData( const FrSimData& Data )
    {
      self->RefSimData( ).append( Data );
    }

    void
      AppendFrSimData( LDAS_SHARED_PTR_QCLASS< FrSimData > Data )
    {
      self->RefSimData( ).append( Data );
    }

    void
      Write( OFrameFStream& Stream,
	     FrVect::compression_scheme_type Compression = FrVect::ZERO_SUPPRESS_OTHERWISE_GZIP,
	     FrVect::compression_level_type Level = 6 )
    {
      OFrameFStream::stream_type*	s = Stream.Stream( );

      if ( s )
      {
	LDAS_SHARED_PTR_QCLASS< FrameH >
	  frame( new FrameH( *self ) );

	s->WriteFrame( frame, Compression, Level );
      }
    }
  } // %extend

  LDAS_PROPERTY_READ_WRITE("name", GetName, SetName)
  LDAS_PROPERTY_READ_WRITE("run", GetRun, SetRun)
  LDAS_PROPERTY_READ_ONLY("frame", GetFrame)
  LDAS_PROPERTY_READ_ONLY("dataQuality", GetDataQuality)
  # GTimeS = property( fget=lambda d:d.GetGTime( ).Seconds( ) )
  # GTimeN = property( fget=lambda d:d.GetGTime( ).Nanoseconds( ) )
  LDAS_PROPERTY_READ_ONLY("ULeapS", GetULeapS)
  LDAS_PROPERTY_READ_WRITE("dt", GetDt, SetDt)
};

#endif /* FRAME_H_I */
