#ifndef FRAME_CPP_INTERFACE__VERIFY_HH
#define FRAME_CPP_INTERFACE__VERIFY_HH
#include <memory>
#include <stdexcept>
#include <sstream>
#include <string>

#include "ldastoolsal/types.hh"
#include "ldastoolsal/unordered_map.hh"
#include "ldastoolsal/util.hh"

#include "framecpp/Common/FrameSpec.hh"
#include "framecpp/Common/FrameFilename.hh"

namespace FrameCPP
{
  namespace Common
  {
    class CheckSum;
    class IFrameStream;
    class Verify;
    class FrHeader;

    class VerifyObject
    {
    public:
      virtual ~VerifyObject( );

      virtual void operator( )( Verify&, IFrameStream& ) const = 0;
    };

    class VerifyException
      : public std::runtime_error
    {
    public:
      enum error_type {
	NO_ERROR = 0,
	CHECKSUM_ERROR,
	FILE_OPEN_ERROR,
	UNSUPPORTED_FRAME_SPEC,
	FRAME_SPEC_CONFORMANCE,
	UNSUPPORTED_CHECKSUM_TYPE,
	NO_CHECKSUM,
	INVALID_FRAME_STRUCTURE,
	FILE_TRUNCATION,
	DATA_INVALID,
	METADATA_INVALID,
	MAX_ERROR = METADATA_INVALID
      };

      VerifyException( error_type ErrorCode, const std::string& What = "" );

      error_type ErrorCode( ) const;

      //: Return string describing error code
      static std::string StrError( error_type ErrorNum );

      //: Retrieve the string representation of the error_type
      static std::string StrErrorType( const error_type Error );

    private:
      static const char* m_error_strings[];
      static const char* m_error_type_strings[];

      const error_type	m_error_code;

      std::string build_error_string( error_type ErrorCode,
				      const std::string& Message );
    };

    inline VerifyException::
    VerifyException( error_type ErrorCode, const std::string& What )
      : std::runtime_error( build_error_string( ErrorCode, What ) ),
	m_error_code( ErrorCode )
    {
    }
			    
    inline VerifyException::error_type VerifyException::
    ErrorCode( ) const
    {
      return m_error_code;
    }

    inline std::string VerifyException::
    build_error_string( error_type ErrorCode, const std::string& What )
    {
      std::string retval( "VerifyException: " );

      retval += StrErrorType( ErrorCode );

      if ( What.length( ) > 0 )
      {
	retval += ": ";
	retval += What;
      }
      return retval;
    }

    class Verify
    {
    public:
      typedef VerifyException::error_type error_type;
      typedef std::unique_ptr< FrameFilename > frame_filename_ptr_type;

      typedef INT_4U chkType_type;
      typedef INT_4U chkSum_type;

      static frame_filename_ptr_type NULL_FRAME_FILENAME;

      //: Default constructor
      Verify( );

      //: Query the size of the I/O buffer
      INT_8U BufferSize( ) const;

      //-----------------------------------------------------------------
      /// \brief Change the size of the I/O buffer
      ///
      /// \param[in] Bytes
      ///     establish the size in bytes of the I/O buffer.
      ///     A value of 0 sets the buffer size to the
      ///     system specified default.
      //-----------------------------------------------------------------
      void BufferSize( INT_8U Bytes );

      //: Query if the data valid field of the FrAdcData must be zero
      bool CheckDataValid( ) const;

      /// \brief Request fast checking
      void CheckFast( bool Check );

      /// \brief Query if fast checking has been requested.
      bool CheckFast( ) const;

      //: Change requirement if the data valid field of the FrAdcData must be zero
      void CheckDataValid( bool Check );

      //: Query if checksums will be calculated for a frame file
      bool CheckFileChecksum( ) const;

      //: Change requirement for calculating the checksum of a frame file
      void CheckFileChecksum( bool Check );

      bool CheckFileChecksumOnly( ) const;

      void CheckFileChecksumOnly( bool Check );

      //: Change requirement for calculating the checksum of a frame file
      void CheckFileChecksumPreVersion8( bool Check );

      //: Query if checksums will be calculated for the frames of a frame file
      bool CheckFrameChecksum( ) const;

      //: Change requirement for calculating the checksum of frames of a frame file
      void CheckFrameChecksum( bool Check );

      //: Query if md5sum will be calculated for the frame file
      bool CheckMD5Sum( ) const;

      //: Change requirement for calculating the md5sum value for the frame file
      void CheckMD5Sum( bool Check );

      //-------------------------------------------------------------------------
      /// \brief Establish the Checksum value for the FrHeader
      //-------------------------------------------------------------------------
      void ChecksumHeader( chkSum_type Checksum );

      //-------------------------------------------------------------------------
      /// \brief Retrieve the value for the checksum scheme
      //-------------------------------------------------------------------------
      INT_4U ChecksumScheme( ) const;

      //-------------------------------------------------------------------------
      /// \brief Set the value for the checksum scheme
      //-------------------------------------------------------------------------
      void ChecksumScheme( INT_4U Value );

      //: Retrieve additional infomation about an error.
      std::string ErrorInfo( ) const;

      //-------------------------------------------------------------------------
      /// \brief Retrieve the value governing checking of compressed data
      ///
      /// \return
      ///      A value of true if compressed data should be expanded as part
      ///      of data validation.
      ///      A value of false is returned if this data check should not
      ///      be performed.
      //-------------------------------------------------------------------------
      bool Expandability( ) const;

      //-------------------------------------------------------------------------
      /// \brief Set the value governing validation of compressed data
      ///
      /// \param[in] Value
      ///     If Value is true, then compressed data should be expanded as part
      ///     of data validation.
      ///     If Value is false, then compressed data should not be expanded
      ///     as part of data validation.
      //-------------------------------------------------------------------------
      void Expandability( bool Value );

      //: Query if the frame file must have the EOF checksum structure.
      bool MustHaveEOFChecksum( ) const;

      //-------------------------------------------------------------------------
      /// \brief Change requirement of the frame file having the EOF checksum structure
      ///
      /// \param[in] Conformance
      ///     true if the frame file must have the EOF
      ///     checksum structure, false otherwise
      //-------------------------------------------------------------------------
      void MustHaveEOFChecksum( bool Conformance );

      //: Change status of having seen FrEndOfFile structure
      void SeenFrEndOfFile( bool Value );

      //: Query strict conformance requirement
      bool Strict( ) const;

      //: Change strict conformance requirement
      void Strict( bool Strictness );

      //: Query if memory mapped I/O will be used
      bool UseMemoryMappedIO( ) const;

      //: Specify if to use memory mapped I/O.
      void UseMemoryMappedIO( bool UseMemoryMappedIO );

      //: Query if metadata should be validated
      bool ValidateMetadata( ) const;

      //: Specify if to validate metadata
      void  ValidateMetadata( bool ValidateMetadata );

      //: Verify a frame file
      error_type operator()( const std::string& Filename );

      //: Verify a frame stream
      error_type operator()( IFrameStream& Stream,
			     frame_filename_ptr_type& Filename
			     = NULL_FRAME_FILENAME );

      void ExamineFrEndOfFileChecksum( IFrameStream& Stream,
				       chkType_type Type, chkSum_type Sum );

      //-----------------------------------------------------------------
      /// \brief Verify the checksum for the FrHeader structure
      ///
      /// \param[in] Checksum
      ///     The value of the expected checksum.
      //-----------------------------------------------------------------
      void ExamineFrHeaderChecksum( chkSum_type Checksum ) const;

      template< int >
      void ExamineMagicNumber( const unsigned char* Source );

    private:
#if 0
      template< typename T > friend
      void VerifyObject( const T&, Verify&, IFrameStream& );
#endif /* 0 */
      typedef LDASTools::AL::SharedPtr< FrHeader >	fr_header_type;

      friend void FrameSpec::ObjectInterface::VerifyObject( Verify& Verifier,
							    IFrameStream& Stream ) const;

      error_type crc_file_checksum_only( IStream& Stream,
					 fr_header_type Header );

      typedef LDASTools::AL::unordered_map< char, int >	detector_validation_type;

      detector_validation_type	m_detector_validation;
      detector_validation_type	m_detector_validation_excess;

      detector_validation_type	m_channel_validation;
      detector_validation_type	m_channel_validation_excess;

      INT_8U			m_buffer_size;
      bool			m_check_data_valid;
      //-----------------------------------------------------------------
      /// \brief Expandability checking
      ///
      /// When enabled (set to true), data that has been compressed
      /// will be expanded as part of validation.
      ///
      /// This check is enabled by default.
      //-----------------------------------------------------------------
      bool			m_check_expandability;
      //-----------------------------------------------------------------
      /// \brief Enable fast checking
      ///
      /// Fast checking is only available for version 8 frames
      /// and beyond since with version 8 frames each structure
      /// contains its own checksum.
      /// 
      /// When this option is set, file and frame checksums are not
      /// calculated.
      /// Validation of the other information makes use of the table
      /// of contents.
      //-----------------------------------------------------------------
      bool			m_check_fast;
      bool			m_check_file_checksum;
      bool			m_check_frame_checksum;
      bool			m_check_md5sum;
      bool			m_checksum_file_only;
      std::ostringstream	m_error_info;
      bool			m_must_have_eof_checksum;
      bool			m_strict;
      bool			m_use_memory_mapped_io;
      bool			m_validate_metadata;
      bool			m_state_has_seen_fr_end_of_file;

      //-------------------------------------------------------------------------
      /// Beginning with version eight frame specification, the
      /// checksum_scheme needs to be recorded from the FrHeader
      /// structure to be used by the FrEndOfFile structure.
      //-------------------------------------------------------------------------
      INT_4U			m_checksum_scheme;

      //-------------------------------------------------------------------------
      /// The checksum of the FrHeader structure upon reading the object.
      //-------------------------------------------------------------------------
      chkSum_type		m_checksum_frheader;

      //-------------------------------------------------------------------------
      /// Verify Checksum values
      ///
      /// param[in] ExpectedCRC
      ///     The checksum that was written to the stream origionally and used
      ///     for comparison to detect changes.
      /// param[in] CalculatedCRC
      ///     The checksum that was calculated while reading the current stream
      ///     and used for comparison to detect changes.
      /// param[in] FrStructure
      ///     The name of the structure being verified.
      ///     This allows for better error handling code as this detail becomes
      ///     part of any diagnostic message.
      /// param[in] MustBeNonZero
      ///     If true, then checksum values of zero are considered to be an
      ///     error.
      ///     This condition commonly comes into play when no checksum
      ///     information is recorded in the output stream usually to
      ///     provide a minimal performance boost.
      ///
      //-------------------------------------------------------------------------
      void verify_checksum( CheckSum* ExpectedCRC,
			    CheckSum* CalculatedCRC,
			    const std::string& FrStructure,
			    bool MustBeNonZero ) const;

      void set_channel_site( char Site );

      template< class T >
      void check_magic_number( const unsigned char* const Source,
			       const T Ref );
      
    };

    //!retrun: INT_8U - return size in bytes of the I/O buffer.
    inline INT_8U Verify::
    BufferSize( ) const
    {
      return m_buffer_size;
    }

    inline void Verify::
    BufferSize( INT_8U Bytes )
    {
      m_buffer_size = Bytes;
    }

    //!return: bool - state of checking that the data valid field is zero
    inline bool Verify::
    CheckDataValid( ) const
    {
      return m_check_data_valid;
    }

    //!param: bool Check - true if the data valid field must be zero, false
    //+	otherwise.
    inline void Verify::
    CheckDataValid( bool Check )
    {
      m_check_data_valid = Check;
    }

    //-------------------------------------------------------------------
    /// \return
    ///     True if fast checking should be attempted.
    //-------------------------------------------------------------------
    inline bool Verify::
    CheckFast( ) const
    {
      return m_check_fast;
    }

    //-------------------------------------------------------------------
    /// \param[in] Value
    ///     If Value is true, then fast checking is enabled;
    ///     If Value is false, then fast checking is disabled.
    //-------------------------------------------------------------------
    inline void Verify::
    CheckFast( bool Value )
    {
      m_check_fast = Value;
    }

    //!return: bool - true if checksums for frame files need to be
    //+	performed, false otherwise.
    inline bool Verify::
    CheckFileChecksum( ) const
    {
      return m_check_file_checksum;
    }

    //!param: bool Check - true if checksums for frame files need to be
    //+	performed, false otherwise.
    inline void Verify::
    CheckFileChecksum( bool Check )
    {
      m_check_file_checksum = Check;
    }

    inline bool Verify::
    CheckFileChecksumOnly( ) const
    {
      return m_checksum_file_only;
    }

    inline void Verify::
    CheckFileChecksumOnly( bool Check )
    {
      m_checksum_file_only = Check;
    }

    //!return: bool - true if checksums for individual frames need to be
    //+	performed, false otherwise.
    inline bool Verify::
    CheckFrameChecksum( ) const
    {
      return m_check_frame_checksum;
    }

    //!param: bool Check - true if checksums for individual frames need to be
    //+	performed, false otherwise.
    inline void Verify::
    CheckFrameChecksum( bool Check )
    {
      m_check_frame_checksum = Check;
    }

    //!return: bool - true if md5sum for the frame file will be
    //+	performed, false otherwise.
    inline bool Verify::
    CheckMD5Sum( ) const
    {
      return m_check_md5sum;
    }

    //!param: bool Check - true if md5sum for the frame file should be
    //+	performed, false otherwise.
    inline void Verify::
    CheckMD5Sum( bool Check )
    {
      m_check_md5sum = Check;
    }

    //---------------------------------------------------------------------------
    /// This estalishes the checksum calculated from the reading of the
    /// FrHeader structure.
    //---------------------------------------------------------------------------
    inline void Verify::
    ChecksumHeader( chkSum_type Checksum )
    {
      m_checksum_frheader = Checksum;
    }

    //---------------------------------------------------------------------------
    /// Retrieve the value for the checksum scheme
    //---------------------------------------------------------------------------
    inline INT_4U  Verify::
    ChecksumScheme( ) const
    {
      return m_checksum_scheme;
    }

    //---------------------------------------------------------------------------
    /// Establish the value for the checksum scheme
    //---------------------------------------------------------------------------
    inline void Verify::
    ChecksumScheme( INT_4U Value )
    {
      m_checksum_scheme = Value;
    }

    //!return: const std::string&  - Additional error information
    inline std::string Verify::
    ErrorInfo( ) const
    {
      return m_error_info.str( );
    }

    template<>
    inline void Verify::
    ExamineMagicNumber< 2 >( const unsigned char* Source )
    {
      if ( m_strict )
      {
	try
	{
	  check_magic_number( Source, INT_2U( 0x1234 ) );
	}
	catch( ... )
	{
	  throw VerifyException( VerifyException::FRAME_SPEC_CONFORMANCE,
				 "2 bytes container 0x1234" );
	}
      }
    }

    template<>
    inline void Verify::
    ExamineMagicNumber< 4 >( const unsigned char* Source )
    {
      if ( m_strict )
      {
	try
	{
	  check_magic_number( Source, INT_4U( 0x12345678 ) );
	}
	catch( ... )
	{
	  throw VerifyException( VerifyException::FRAME_SPEC_CONFORMANCE,
				 "4 bytes container 0x12345678" );
	}
      }
    }

    template<>
    inline void Verify::
    ExamineMagicNumber< 8 >( const unsigned char* Source )
    {
      if ( m_strict )
      {
	try
	{
	  check_magic_number( Source, INT_8U( 0x123456789abcdefULL ) );
	}
	catch( ... )
	{
	  throw VerifyException( VerifyException::FRAME_SPEC_CONFORMANCE,
				 "8 bytes container 0x123456789abcdef" );
	}
      }
    }

    inline void Verify::
    Expandability( bool Value )
    {
      m_check_expandability = Value;
    }

    inline bool Verify::
    Expandability( ) const
    {
      return m_check_expandability;
    }

    //!return: bool - true if the frame file must have the EOF checksum
    //+	 structure, false otherwise
    inline bool Verify::
    MustHaveEOFChecksum( ) const
    {
      return m_must_have_eof_checksum;
    }

    inline void Verify::
    MustHaveEOFChecksum( bool Confomance )
    {
      m_must_have_eof_checksum = Confomance;
    }

    inline void Verify::
    SeenFrEndOfFile( bool Value )
    {
      m_state_has_seen_fr_end_of_file = Value;
    }

    //!return: bool - state of strictness. True for strict conformance to
    //+	 the frame spec; false otherwise.
    inline bool Verify::
    Strict( ) const
    {
      return m_strict;
    }

    //!param: bool Strictness - true if the frame file must conform to all
    //+	details of the frame spec; false otherwise.
    inline void Verify::
    Strict( bool Strictness )
    {
      m_strict = Strictness;

    }
    //!return: bool - true if memory mapped I/O will be used
    inline bool Verify::
    UseMemoryMappedIO( ) const
    {
      return m_use_memory_mapped_io;
    }

    //!param: bool UseMemoryMappedIO - true if memory mapped I/O should be
    //+	 used; false otherwise.
    inline void Verify::
    UseMemoryMappedIO( bool UseMemoryMappedIO )
    {
      m_use_memory_mapped_io = UseMemoryMappedIO;
    }

    //!return: bool - true if metadata should be validated
    inline bool Verify::
    ValidateMetadata( ) const
    {
      return m_validate_metadata;
    }

    //!param: bool ValidateMetadata - true if frame metadata to be verified against
    //+	 against the filename
    inline void Verify::
    ValidateMetadata( bool ValidateMetadata )
    {
      m_validate_metadata = ValidateMetadata;
    }

    inline void Verify::
    set_channel_site( char Site )
    {
      detector_validation_type::iterator
	pos = m_channel_validation.find( Site );
      if ( pos == m_channel_validation.end( ) )
      {
	m_channel_validation_excess[ Site ]++;
      }
      else
      {
	(*pos).second++;
      }
    }

    template< class T >
    inline void Verify::
    check_magic_number( const unsigned char* const Source, const T Ref )
    {
      T buf;

      //-----------------------------------------------------------------
      // Load the buffer
      //-----------------------------------------------------------------
      memcpy(&buf, Source, sizeof( buf ) );
      //-----------------------------------------------------------------
      // Check the bytes
      //-----------------------------------------------------------------
      if ( Ref != buf )
      {
	//---------------------------------------------------------------
	// Reverse the bytes and check again
	//---------------------------------------------------------------
	reverse< sizeof( buf ) >( &buf, 1 );
	if ( Ref != buf )
	{
	  //-------------------------------------------------------------
	  // Pattern does not match either BIGENDIAN or LITTLEENDIAN so
	  //   it must not be a frame file.
	  //-------------------------------------------------------------
	  throw std::runtime_error( "non-frame frame file" );
	}
      }
    }

  } // namespace - Common
} // namespace - FrameCPP

#endif /* FRAME_CPP_INTERFACE__VERIFY_HH */
