#include <ldas_tools_config.h>

extern "C" {
#include <zlib.h>
} // extern "C"

#include <iostream>

#include <stdexcept>
#include <sstream>

#include "ldastoolsal/types.hh"
#include "ldastoolsal/autoarray.hh"

#include "framecpp/Common/CompressionGZip.hh"

using LDASTools::AL::AutoArray;

namespace FrameCPP
{
  namespace Compression
  {
    //-------------------------------------------------------------------
    /// The routines defined in this namespace are specific to the
    /// GZip compression algorithm.
    /// Routines are defined for both compression and decompression
    /// of data buffers.
    //-------------------------------------------------------------------
    namespace GZip
    {
      //-----------------------------------------------------------------
      /// The data must be in an uncompressed state before this is
      /// called.
      //-----------------------------------------------------------------
      void
      Compress( const CHAR_U* DataIn, INT_8U NBytesIn, size_t Level,
		AutoArray< CHAR_U >& DataOut, INT_8U& NBytesOut )
      {
	if ( Level == 0 )
	{
	  return;
	}
	if ( Level > 9 )
	{
	  Level = 9;
	}
    
	uLongf bufferSize = uLongf( NBytesIn * 1.001 + 13 );
    
	AutoArray< CHAR_U > buffer( new CHAR_U[ bufferSize ] );
#if HAVE_LIBZ_COMPRESS2
	::compress2( buffer.get( ), &bufferSize, DataIn, NBytesIn, Level );
#else
	::compress( buffer.get( ), &bufferSize, DataIn, NBytesIn );
#endif

	NBytesOut = bufferSize;
	DataOut.reset( new CHAR_U[ NBytesOut ] );
	std::copy( &buffer[ 0 ], &( buffer[ NBytesOut] ), &(DataOut[ 0 ]) );
      } // func - Compress

      //-----------------------------------------------------------------
      /// This will uncompress the data using the appropriate GZip
      /// call.
      //-----------------------------------------------------------------
      void
      Expand( const CHAR_U* DataIn, INT_8U NBytesIn,
	      AutoArray< CHAR_U >& DataOut, INT_8U& NBytesOut )
      {
	//---------------------------------------------------------------
	// This may throw std::bad_alloc
	//---------------------------------------------------------------
#if 0
	std::cerr << "DEBUG: GZIP: Expand:"
		  << " NBytesOut: " << NBytesOut
		  << std::endl
	  ;
#endif /* 0 */
	DataOut.reset( new CHAR_U[ NBytesOut ] );
	uLongf uncompressedLength( NBytesOut );

	int error = uncompress( DataOut.get( ),
				&uncompressedLength,
				DataIn,
				NBytesIn );

	//---------------------------------------------------------------
	// Check the return status from the uncompress
	//---------------------------------------------------------------
	if ( error != Z_OK )
	{
	  DataOut.reset( (CHAR_U*)NULL );
	  switch( error )
	  {
	  case Z_MEM_ERROR:
	    throw std::bad_alloc();
	    break;
	
	  case Z_BUF_ERROR:
	  case Z_DATA_ERROR:
	    throw std::runtime_error( "Compressed data is corrupt." );
	    break;
                    
	  default:
	    throw std::runtime_error( "An unknown error occurred during "
				      "uncompression." );
	  }
	}

	//---------------------------------------------------------------
	// Make sure we uncompressed the correct number of bytes.
	//---------------------------------------------------------------
	if ( uncompressedLength != NBytesOut )
	{
	  std::ostringstream	msg;

	  msg << "Compressed data is corrupt ("
	      << "Buffer expands to " << uncompressedLength
	      << " bytes in length instead of " << NBytesOut
	      << " bytes)"
	    ;
	  throw std::range_error( msg.str( ) );
	}

	//---------------------------------------------------------------
	// Successfully expanded the buffer, so return the results
	//---------------------------------------------------------------
      } // func - Exapnd

    } // namespace - GZip

  } // namespace - Compression

} // namespace - FrameCPP
