#include <ldas_tools_config.h>

#include <fstream>

#include "genericAPI/Logging.hh"

#include "diskcacheAPI/Cache/ExcludedDirectoriesSingleton.hh"

#include "diskcacheAPI/Streams/ASCII.hh"
#include "diskcacheAPI/Streams/Binary.hh"
#include "diskcacheAPI/Streams/FStream.hh"

#include "diskcacheAPI/MetaCommands.hh"

#include "Commands.hh"
#include "diskcachecmd.hh"
#include "DumpCacheDaemon.hh"
#include "IO.hh"
#include "MountPointScanner.hh"

//=======================================================================
//=======================================================================

using GenericAPI::queueLogEntry;

using diskCache::Commands::updateFileExtList;
using diskCache::Commands::updateMountPtList;

typedef diskCache::MetaCommand::client_type	client_type;
typedef diskCache::MetaCommand::ClientServerInterface::ServerInfo ServerInfo;

//=======================================================================
// Global variables
//=======================================================================

#if 0
static client_type	client_handle;
#endif /* 0 */

//=======================================================================
//=======================================================================
#if 0
static void send_request( const std::string& Command,
			  const ServerInfo& Server );
#endif /* 0 */

//=======================================================================
/// \brief Maintains a list of commands that are supported.
//=======================================================================
namespace diskCache
{
  namespace MetaCommand
  {
    //===================================================================
    // CommandTable
    //===================================================================

    // Instantiate generic to Singleton class methods:   
    // see "general" library   
    SINGLETON_TS_INST( CommandTable );

    /** \cond ignore_singleton_constructor */
    CommandTable::
    CommandTable( )
    {
      command_table[ "daemon" ] = CMD_DAEMON;
      command_table[ "dump" ] = CMD_DUMP;
      command_table[ "filenames" ] = CMD_FILENAMES;
      command_table[ "filenames-rds" ] = CMD_FILENAMES_RDS;
      command_table[ "intervals" ] = CMD_INTERVALS;
      command_table[ "mount-point-stats" ] = CMD_MOUNT_POINT_STATS;
      command_table[ "quit" ] = CMD_QUIT;
      command_table[ "scan" ] = CMD_SCAN;
    }
    /** \endcond */

    CommandTable::
    ~CommandTable( )
    {
      command_table.erase( command_table.begin( ),
			   command_table.end( ) );
    }

    ClientServerInterface::
    ~ClientServerInterface( )
    {
      if ( server_request_handle )
      {
	server_request_handle->close( );
      }
    }

    //===================================================================
    // MountPointStats
    //===================================================================
    OptionSet& MountPointStats::m_options( MountPointStats::init_options( ) );

    OptionSet& MountPointStats::
    init_options( )
    {
      static OptionSet	retval;

      retval.
	Synopsis( "Subcommand: mountPointStats" );

      retval.
	Summary( "The mount-point-stats sub command is intended to get some statistical information"
		 " the memory cache grouped by the mount points."
		 );

      retval.Add( Option( OPT_IFO,
			  "ifo",
			  Option::ARG_REQUIRED,
			  "IFO pattern to use for search. (Default: all)",
			  "pattern" ) );

      retval.Add( Option( OPT_TYPE,
			  "type",
			  Option::ARG_REQUIRED,
			  "Type pattern to use for search. (Default: all)",
			  "pattern" ) );

      return retval;
    }

    MountPointStats::
    MountPointStats( CommandLineOptions& Args )
      : m_args( Args ),
	m_ifo( "all" ),
	m_type( "all" )
    {
      if ( m_args.empty( ) == false )
      {
	//---------------------------------------------------------------
	// Parse the commands
	//---------------------------------------------------------------
	std::string	arg_name;
	std::string	arg_value;
	bool 		parsing( true );

	while( parsing )
	{
	  switch( m_args.Parse( m_options, arg_name, arg_value ) )
	  {
	  case CommandLineOptions::OPT_END_OF_OPTIONS:
	    parsing = false;
	    break;
	  case OPT_IFO:
	    m_ifo = arg_value;
	    break;
	  case OPT_TYPE:
	    m_type = arg_value;
	    break;
	  default:
	    break;
	  }
	}
      }
    }

    const OptionSet& MountPointStats::
    Options( )
    {
      return m_options;
    }

    void MountPointStats::
    operator()( )
    {
      diskCache::Cache::QueryAnswer	answer;

      getHashNumbers( answer, m_ifo.c_str( ), m_type.c_str( ) );
    }

    //===================================================================
    // Scan
    //===================================================================
    OptionSet& Scan::m_options( Scan::init_options( ) );

    OptionSet& Scan::
    init_options( )
    {
      static OptionSet	retval;

      retval.
	Synopsis( "Subcommand: scan" );

      retval.
	Summary( "The scan sub command is intended to"
		 " scan a set of directories for files of interest"
		 " and generate a memory cache image."
		 );

      retval.Add( Option( OPT_CONCURRENCY,
			  "concurrency",
			  Option::ARG_REQUIRED,
			  "Number of mount points to scan concurrently",
			  "integer" ) );

      retval.Add( Option( OPT_EXTENSIONS,
			  "extensions",
			  Option::ARG_REQUIRED,
			  "Comma seperated list of file extensions",
			  "list" ) );

      retval.Add( Option( OPT_MOUNT_POINTS,
			  "mount-points",
			  Option::ARG_REQUIRED,
			  "Comma seperated list of mount points to scan",
			  "list" ) );

      retval.Add( Option( OPT_OUTPUT_ASCII,
			  "output-ascii",
			  Option::ARG_REQUIRED,
			  "Filename for the ascii output; '-' to direct to standard output",
			  "filename" ) );

      retval.Add( Option( OPT_OUTPUT_BINARY,
			  "output-binary",
			  Option::ARG_REQUIRED,
			  "Filename for the binary output",
			  "filename" ) );

      retval.Add( Option( OPT_TYPE,
			  "type",
			  Option::ARG_REQUIRED,
			  "Type pattern to use for search. (Default: all)",
			  "pattern" ) );

      retval.Add( Option( OPT_VERSION_ASCII,
			  "version-ascii",
			  Option::ARG_REQUIRED,
			  "Version of the ascii diskcache dump format to output",
			  "version" ) );
      retval.Add( Option( OPT_VERSION_BINARY,
			  "version-binary",
			  Option::ARG_REQUIRED,
			  "Version of the binary diskcache dump format to output",
			  "version" ) );
      return retval;
    }

    Scan::
    Scan( CommandLineOptions& Args )
      : m_args( Args ),
	m_output_ascii( "" ),
	m_output_binary( "" ),
	m_version_ascii( VERSION_DEFAULT_ASCII ),
	m_version_binary( VERSION_DEFAULT_BINARY )
    {
      if ( m_args.empty( ) == false )
      {
	//---------------------------------------------------------------
	// Parse the commands
	//---------------------------------------------------------------
	std::string	arg_name;
	std::string	arg_value;
	bool 		parsing( true );

	while( parsing )
	{
	  switch( m_args.Parse( m_options, arg_name, arg_value ) )
	  {
	  case CommandLineOptions::OPT_END_OF_OPTIONS:
	    parsing = false;
	    break;
	  case OPT_CONCURRENCY:
	    {
	      std::istringstream	value_stream( arg_value );

	      INT_4U	concurrency;

	      value_stream >> concurrency;

	      ScanConcurrency( concurrency );
	    }
	    break;
	  case OPT_EXTENSIONS:
	    {
	      //-----------------------------------------------------------
	      // Generate list of extensions
	      //-----------------------------------------------------------
	      size_t pos = 0;
	      size_t end = 0;

	      while ( end != std::string::npos )
	      {
		end = arg_value.find_first_of( ",", pos );
		m_extensions.push_back( arg_value.substr( pos,( ( end == std::string::npos )
								? end
								: end - pos ) ) );
		pos = end + 1;
	      }
	    }
	    break;
	  case OPT_MOUNT_POINTS:
	    {
	      //-----------------------------------------------------------
	      // Generate list of mount points
	      //-----------------------------------------------------------
	      size_t pos = 0;
	      size_t end = 0;

	      while ( end != std::string::npos )
	      {
		end = arg_value.find_first_of( ",", pos );
		m_mount_points.push_back( arg_value.substr( pos,( ( end == std::string::npos )
								  ? end
								  : end - pos ) ) );
		pos = end + 1;
	      }
	    }
	    break;
	  case OPT_TYPE:
	    m_type = arg_value;
	    break;
	  case OPT_OUTPUT_ASCII:
	    m_output_ascii = arg_value;
	    break;
	  case OPT_OUTPUT_BINARY:
	    m_output_binary = arg_value;
	    break;
	  case OPT_VERSION_ASCII:
	    {
	      std::istringstream	version_stream( arg_value );

	      version_stream >> std::hex >> m_version_ascii >> std::dec;
	    }
	    break;
	  case OPT_VERSION_BINARY:
	    {
	      std::istringstream	version_stream( arg_value );

	      version_stream >> std::hex >> m_version_binary >> std::dec;
	    }
	    break;
	  default:
	    break;
	  }
	}
      }
    }

    const OptionSet& Scan::
    Options( )
    {
      return m_options;
    }

    void Scan::
    operator()( )
    {
      //-----------------------------------------------------------------
      // Add the extensions
      //-----------------------------------------------------------------
      updateFileExtList( m_extensions );

      //-----------------------------------------------------------------
      // Add the mount points
      //-----------------------------------------------------------------
      if ( m_mount_points.size( ) > 0 )
      {
	MountPointManagerSingleton::UpdateResults    status;

	updateMountPtList( status, m_mount_points, false );
      }

      //-----------------------------------------------------------------
      // Scan the directories
      //-----------------------------------------------------------------

      {
	diskCache::MountPointScanner::ScanResults	results;
	ScanMountPointList( results );

	/// \todo format results for output to stream
	/// diskCache::ASCII::Translate( &std::cout, results );
      }

      //-----------------------------------------------------------------
      // Write the scanned data
      //-----------------------------------------------------------------
      if ( m_output_ascii.empty( )
	   || ( m_output_ascii.compare( "-" ) == 0 ) )
      {
	//-------------------------------------------------------------
	// Dump the default mount point manager
	//-------------------------------------------------------------
	diskCache::Streams::OASCII	stream( std::cout, m_version_ascii );

	diskCache::Write( stream );
      }
      else if ( m_output_ascii.empty( ) == false )
      {
	//-------------------------------------------------------------
	/// \todo
	///     When the output is ascii, the query options should allow
	///     the user to request a subset of the entire cache.
	//-------------------------------------------------------------
	//-------------------------------------------------------------
	// Dump the default mount point manager
	//-------------------------------------------------------------
	diskCache::Streams::OFStream	file_stream( m_output_ascii );
	diskCache::Streams::OASCII	stream( file_stream, m_version_ascii );

	diskCache::Write( stream );
      }
      if ( m_output_binary.empty( ) == false )
      {
	//-------------------------------------------------------------
	/// \todo
	///     When the output is binary, the query options should allow
	///     the user to request a subset of the entire cache.
	//-------------------------------------------------------------
	//-------------------------------------------------------------
	// Dump the default mount point manager
	//-------------------------------------------------------------
	diskCache::Streams::OFStream	file_stream( m_output_binary );
	diskCache::Streams::OBinary	stream( file_stream, m_version_binary );

	diskCache::Write( stream );
      }
    }

    //
    ClientServerInterface::ServerInfo::
    ServerInfo( )
      : port( -1 )
    {
    }
  }
}

#if 0
//=======================================================================
//=======================================================================
static void
send_request( const std::string& Command,
	      const ServerInfo& Server )
{
  if ( ! client_handle )
  {
    client_handle = client_type( new client_type::element_type( ) );
    client_handle->open( Server.Hostname( ), Server.Port( ) );
  }
  (*client_handle) << INT_4U( Command.size( ) );
  client_handle->write( &(Command[0]), Command.size( ) );
  client_handle->flush( );
}
#endif /* 0 */
