/* -*- mode: C++ -*- */
#ifndef UNITTEST_H
#define	UNITTEST_H

#include <stdlib.h>
#include <strings.h>

#include <cstdlib>
#include <cstring>

#include <iostream>
#include <fstream>
#include <sstream>
#include <string>

#ifndef WITHOUT_AT_EXIT_CLASS
#include "ldastoolsal/AtExit.hh"
#endif /* WITHOUT_AT_EXIT_CLASS */

namespace LDASTools
{
  namespace Testing
  {
    /// \brief  Provide standard routines for unit testing of code.
    ///
    class UnitTest
    {
    public:
      /// \brief  Constructor
      UnitTest(void);
      /// \brief  Destructor
      ~UnitTest(void);
      /// \brief  Check for error conditions
      std::ostream& Check(bool TestResult);
      /// \brief  Check for error conditions
      void Check(bool TestResult, std::string Message);
      /// \brief  Exit with the appropriate exit code
      void Exit(void);
      /// \brief  Initialize the class with command line arguments
      void Init(int ArgC, char** ArgV);
      /// \brief  Check if in verbose mode
      inline bool IsVerbose( int Level = 0 ) const
      {
	return ( Level <= m_verbose );
      }   
      /// \brief  Output info message
      std::ostream& Message( bool Leader = true ) const;
      /// \brief  Output info message
      std::ostream& Message( int Level, bool Leader = true ) const;
      /// \brief  Retrieve the verbosity level
      inline int Verbosity( ) const
      {
	return ( m_verbose );
      }   

    private:
      /// \brief  Level of verbosity
      int			m_verbose;
      /// \brief  Failure status flag
      bool		m_status;
      /// \brief  Pointer to stream when in non-verbose mode
      mutable std::ostringstream	m_null_stream;
    }; // class - UnitTest

    inline UnitTest::
    UnitTest(void)
      : m_verbose( -1 ),
	m_status(true),
	m_null_stream( "" )
    {
      m_null_stream.clear( );
      if ( getenv("TEST_VERBOSE_MODE") )
      {
	if ( 0 == strcmp( getenv( "TEST_VERBOSE_MODE" ), "true" ) )
	{
	  m_verbose = 0;
	}
	else if ( 0 == strcmp( getenv( "TEST_VERBOSE_MODE" ), "false" ) )
	{
	  m_verbose = -1;
	}
	else
	{
	  m_verbose = atoi( getenv( "TEST_VERBOSE_MODE" ) );
	}
      }
    }
  
    inline UnitTest::
    ~UnitTest(void)
    {
    }

    inline std::ostream& UnitTest::
    Check(bool TestResult)
    {
      m_status = m_status && TestResult;
      if ( IsVerbose() )
      {
	std::cout << "-- " << ((TestResult) ? "PASS" : "FAIL") << ": ";
	return std::cout;
      }
      m_null_stream.str( "" );
      m_null_stream.clear( );
      return m_null_stream;
    }

    inline void UnitTest::
    Check(bool TestResult, std::string Message)
    {
      m_status = m_status && TestResult;
      if ( IsVerbose() )
      {
	std::cout << "-- " << ((TestResult) ? "PASS" : "FAIL") << ": " << Message
		  << std::endl << std::flush;
      }
    }

    inline void UnitTest::
    Exit(void)
    {
#ifndef WITHOUT_AT_EXIT_CLASS
      LDASTools::AL::AtExit::Cleanup( );
#endif /* WITHOUT_AT_EXIT_CLASS */
      exit ((m_status) ? 0 : 1);
    }

    inline void UnitTest::
    Init(int ArgC, char** ArgV)
    {
      for (int x = 1; x < ArgC; x++)
      {
	if (0 == (strcmp(ArgV[x], "--verbose")))
	{
	  m_verbose = 0;
	  continue;
	}
      }
    }

    inline std::ostream& UnitTest::
    Message( bool Leader ) const
    {
      if ( IsVerbose() )
      {
	if (Leader)
	{
	  std::cout << "-- MESG: ";
	}
	return std::cout;
      }
      m_null_stream.str( "" );
      m_null_stream.clear( );
      return m_null_stream;
    }

    inline std::ostream& UnitTest::
    Message( int Level, bool Leader ) const
    {
      if ( IsVerbose( Level ) )
      {
	return Message( Leader );
      }
      m_null_stream.str( "" );
      m_null_stream.clear( );
      return m_null_stream;
    }
  } // namespace - Testing
} // namespace - LDASTools

#endif	/* UNITTEST_H */
