// -*- mode: c++; c-basic-offset: 4; indent-tabs-mode: nil; -*-

#include <ldas_tools_config.h>

#include <sys/stat.h>

#include <algorithm>
#include <list>
#include <string>
#include <sstream>
#include <utility>

#include "ldastoolsal/Directory.hh"
#include "ldastoolsal/gpstime.hh"
#include "ldastoolsal/SharedPtr.hh"
#include "ldastoolsal/System.hh"
#include "ldastoolsal/Timeout.hh"

#include "genericAPI/Logging.hh"
#include "genericAPI/Stat.hh"

#include "diskcacheAPI/Cache/Directory.hh"
#include "diskcacheAPI/Cache/DirectoryManager.hh"
#include "diskcacheAPI/Cache/ExcludedDirectoriesSingleton.hh"
#include "diskcacheAPI/Cache/QueryAnswer.hh"
#include "diskcacheAPI/Cache/RegistrySingleton.hh"

#include "diskcacheAPI/Streams/ASCII.hh"
#include "diskcacheAPI/Streams/Binary.hh"

//-----------------------------------------------------------------------
// Frequently used namespaces.
//-----------------------------------------------------------------------

using std::string;
using LDASTools::AL::ErrorLog;
using LDASTools::AL::MutexLock;
using LDASTools::AL::StdErrLog;
using LDASTools::AL::TimeoutException;
using LDASTools::AL::WeakPtr;
using GenericAPI::LogEntryGroup_type;

using diskCache::Cache::ExcludedDirectoriesSingleton;

typedef diskCache::Cache::Directory                     Directory;
typedef diskCache::Cache::Directory::SymbolicLinkError  SymbolicLinkError;
typedef diskCache::Cache::Directory::children_type      children_type;
typedef diskCache::Cache::Directory::ignored_type       ignored_type;
typedef diskCache::Cache::Directory::dirref_type        dirref_type;
typedef diskCache::Cache::DirectoryScanData             DirectoryScanData;

typedef std::list< std::string >                        filename_container_type;
typedef std::string                                     directory_element_type;
typedef std::list< directory_element_type >             directory_container_type;

//-----------------------------------------------------------------------
// Debugging
//-----------------------------------------------------------------------
#if 0
#include "ldastoolsal/IOLock.hh"

#define VERBOSE_DEBUGGING 1
#define	AT_BASE( a ) \
{ \
   int oldstate; \
   pthread_setcancelstate( PTHREAD_CANCEL_DISABLE, &oldstate ); \
   MutexLock    lock( LDASTools::AL::IOLock::GetKey( std::cerr ) ); \
   pthread_setcancelstate( oldstate, &oldstate ); \
\
   std::cerr << a << __FILE__ << " " << __LINE__ << std::endl << std::flush; \
}

#else
#define	AT_BASE( a )
#endif
#define	AT( a )	AT_BASE( a << std::endl << "\t" )
#define	HERE( ) AT_BASE( "" )

//-----------------------------------------------------------------------
// Forward declaration of local functions
//-----------------------------------------------------------------------
namespace
{
  diskCache::Cache::Directory::timestamp_type
  do_stat( const std::string& Path, struct stat& StatStruct );

  const string errnoMessage( );

  void read_directory( const std::string& Path,
                       const std::string& RootDirectory,
                       directory_container_type& Children,
                       ignored_type& IgnoredDirectories,
                       Directory::index_container_type& OldData,
                       DirectoryScanData& ScanData,
                       const diskCache::Cache::StatInfo& Stat );

}

namespace diskCache
{
  namespace Streams
  {
    template<>
    IBinary& IBinary::
    operator>>( std::set<std::string>& Data )
    {
      //-----------------------------------------------------------------
      // Start with an empty container
      //-----------------------------------------------------------------
      Data.clear( );
      
      //-----------------------------------------------------------------
      // Read the number of elements
      //-----------------------------------------------------------------
      size_type	s;

      *this >> s;

      //-----------------------------------------------------------------
      // Read the elements out to the stream
      //-----------------------------------------------------------------
      std::string d;

      for ( size_type
	      cur = 0,
	      last = s;
	    cur != last;
	    ++cur )
      {
	//---------------------------------------------------------------
	// Extract from the stream
	//---------------------------------------------------------------
	*this >> d;
	//---------------------------------------------------------------
	// Insert into container
	//---------------------------------------------------------------
	Data.insert( d );
      }
      return *this;
    }
  }

  namespace Cache
  {
    //===================================================================
    //===================================================================
    class StatInfo
    {
    public:
        struct stat                     stat_buffer;
        Directory::timestamp_type       mod_time;
    };

    //===================================================================
    //===================================================================
    Directory::FileCacheError::
    FileCacheError( const std::string& Filename, const std::string& Reason )
      : std::runtime_error( format( Filename, Reason ) )
    {
    }
    
    Directory::FileCacheError::
    FileCacheError( const std::string& Reason )
      : std::runtime_error( Reason )
    {
    }
    
    std::string Directory::FileCacheError::
    format( const std::string& Path, const std::string& Reason )
    {
      std::ostringstream        msg;

      if ( Reason.size( ) > 0 )
      {
        
      }
      return msg.str( );
    }

    //===================================================================
    //===================================================================
    Directory::StartIsNotADirectoryError::
    StartIsNotADirectoryError( const std::string& Path, const char* MethodName )
      : std::runtime_error( format( Path, MethodName ) )
    {
    }
    
    std::string Directory::StartIsNotADirectoryError::
    format( const std::string& Path, const char* MethodName )
    {
      std::ostringstream        msg;

      if ( MethodName )
      {
        msg << MethodName;
      }
      msg << "Specified path \'"
          << Path
          << "\' is not a directory."
        ;

      return msg.str( );
    }

    Directory::SymbolicLinkError::
    SymbolicLinkError( const std::string& Path, const char* MethodName )
      : std::runtime_error( format( Path, MethodName ) )
    {
    }
    
    std::string Directory::SymbolicLinkError::
    format( const std::string& Path, const char* MethodName )
    {
      std::ostringstream        msg;

      if ( MethodName )
      {
        msg << MethodName;
      }
      msg << "API does not support symbolic links. Please remove \'"
          << Path
          << "\'"
        ;

      return msg.str( );
    }

    //===================================================================
    //===================================================================
    Directory::ScanResults::
    ScanResults( )
      : m_scanned_directories( 0 ),
        m_scanned_files( 0 )
    {
    }

    void Directory::ScanResults::
    Log( const std::string& Caller, const std::string& JobInfo ) const
    {
      for ( directory_container_type::const_iterator
              cur = m_results.begin( ),
              last = m_results.end( );
            cur != last;
            ++cur )
      {
        std::string 		state( "UNKNOWN" );
        std::ostringstream	added;
        std::ostringstream	removed;
        std::ostringstream	msg;

        switch( cur->second.s_state )
        {
        case Directory::DIRECTORY_NEW:
          state = "NEW";
          break;
        case Directory::DIRECTORY_UPDATED:
          state = "UPDATED";
          break;
        case Directory::DIRECTORY_REMOVED:
          state = "REMOVED";
          break;
        }
        for ( ScanResults::file_container_type::const_iterator
                fi_cur = cur->second.s_engines.begin( ),
                fi_last = cur->second.s_engines.end( );
              fi_cur != fi_last;
              ++fi_cur )
        {
          if ( fi_cur->second.s_added > 0 )
          {
            added << " " << fi_cur->second.s_added
                  << " " << ( ( fi_cur->second.s_added == 1 )
                              ? "FILE"
                              : "FILES" )
                  << " ADDED"
                  << " for engine " << fi_cur->first
              ;
          }
          if ( fi_cur->second.s_removed > 0 )
          {
            removed << " " << fi_cur->second.s_removed
                    << " " << ( ( fi_cur->second.s_removed == 1 )
                                ? "FILE"
                                : "FILES" )
                    << " REMOVED"
                    << " for engine " << fi_cur->first
              ;
          }
        }
        msg << "{"
            << cur->first // Directory name
            << " " << state
            << " DIRECTORY"
            << ( ( ( added.str( ).length( ) > 0 )
                   || ( removed.str( ).length( ) > 0 ) )
                 ? " WITH"
                 : "" )
            << added.str( )
            << removed.str( )
            << "}"
          ;
        GenericAPI::queueLogEntry( msg.str( ),
                                   GenericAPI::LogEntryGroup_type::MT_NOTE,
                                   0,
                                   Caller,
                                   JobInfo );
      }
    }

    //###################################################################
    //###################################################################
    MutexLock::baton_type              Directory::SDGTX_ID_baton;
    RegistrySingleton::id_type         Directory::SDGTx_ID = RegistrySingleton::KEY_NULL;
    RegistrySingleton::ascii_key_type  Directory::SDGTx_ID_KEY = "SDGTx";

    Directory::excluded_directories_rw_type::baton_type
    Directory::p_excluded_directories_baton;
    Directory::excluded_directories_type
    Directory::p_excluded_directories;

    typedef WeakPtr< Directory::dirref_type::element_type > weakdirref_type;
    //-------------------------------------------------------------------
    //-------------------------------------------------------------------


    Directory::
    Directory( )
      : m_last_time_modified( 0 ),
        m_state( DIRECTORY_NEW )
        
    {
    }
    
    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    Directory::
    Directory( const std::string& Path,
               const std::string& Root )
      : m_name( Path ),
        m_last_time_modified( 0 ),
        m_root( Root ),
        m_state( DIRECTORY_NEW )
    {
    }
    
    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    Directory::
    Directory( const std::string& Path,
               dirref_type Parent )
      : /* m_parent( Parent ), */
        m_name( Path ),
        m_last_time_modified( 0 ),
        m_state( DIRECTORY_NEW )
    {
    }
    
    //-------------------------------------------------------------------
    /// Implements the copy constructor for the class
    //-------------------------------------------------------------------
    Directory::
    Directory( const Directory& Source )
      : m_indexes( Source.m_indexes ),
        m_subdirs( Source.m_subdirs ),
        m_subdirs_ignored( Source.m_subdirs_ignored ),
        m_name( Source.m_name ),
        m_last_time_modified( Source.m_last_time_modified ),
        m_root( Source.m_root ),
        m_state( Source.m_state )
    {
    }
    
    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    Directory::
    ~Directory( )
    {
    }
    
    //-------------------------------------------------------------------
    /// Configuration of the directory cache understands the following
    /// variables:
    ///
    /// <ul>
    ///   <li><b>exclude</b> - A colon separated list of patterns
    ///       which describe directories to be ignored.
    ///   </li>
    /// </ul>
    //-------------------------------------------------------------------
    bool Directory::
    Configure( const std::string& Variable,
	       const std::string& Value )
    {
      bool	retval = false;

      return retval;
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    void Directory::
    Find( QueryAnswer& Answer )const
    {
      index_container_type::const_iterator
        pos( m_indexes.find( Answer.IndexId( ) ) );

      if ( pos != m_indexes.end( ) )
      {
        RegistrySingleton::info_type
          info( RegistrySingleton::GetInfo( Answer.IndexId( ) ) );

        if ( info )
        {
          reinterpret_cast< const Registry::Info * >( info.get( ) )->Find( *pos->second, *this, Answer );
        }
      }
    }

    //-------------------------------------------------------------------
    //
    //-------------------------------------------------------------------
    Directory::dirref_type Directory::
    Scan( DirectoryManager& DirectoryCollection,
          ScanResults& Results,
          const std::string& Caller,
          const std::string& JobInfo ) const
    {
      dirref_type       retval;

      static const CHAR* const method_name( "diskCache::Cache::Directory::scan( ): " );

      //-----------------------------------------------------------------
      //
      //-----------------------------------------------------------------
      StatInfo          stat_info;

      std::string       fullname( Fullname( ) );

      stat_info.mod_time = do_stat( fullname, stat_info.stat_buffer );
      if ( is_updated( m_last_time_modified,
                       stat_info.mod_time ) == true )
      {
        //---------------------------------------------------------------
        // Lock since it is most likely that the information will be
        // updated.
        // This serializes the access from the default directory scans
        // and other methods such as HotDirectory.
        //---------------------------------------------------------------
        MutexLock       lock( m_scan_baton, __FILE__, __LINE__ );

        stat_info.mod_time = do_stat( fullname,
                                      stat_info.stat_buffer );

        if ( is_updated( m_last_time_modified,
                         stat_info.mod_time ) == true )
        {
          //-------------------------------------------------------------
          // Verify that the path is a directory
          //-------------------------------------------------------------
          if ( S_ISDIR( stat_info.stat_buffer.st_mode ) == 0 )
          {
            //-----------------------------------------------------------
            // It is not a directory.
            // Check if it is a symbolic link
            //-----------------------------------------------------------
            if ( S_ISLNK( stat_info.stat_buffer.st_mode ) )
            {
              //---------------------------------------------------------
              // Any entry that is a symbolic link is an error
              //---------------------------------------------------------
              throw SymbolicLinkError( m_name, method_name );
            }
            else
            {
              //---------------------------------------------------------
              // It is something other than a directory or a symbolic
              // link.
              //---------------------------------------------------------
              if ( m_name.size( ) == m_root.size( ) )
              {
                throw StartIsNotADirectoryError( m_name, method_name );
              }
              return retval;
            }
          }
          retval = update( Results,
                           DirectoryCollection,
                           stat_info );

          if ( retval )
          {
            DirectoryCollection.OnUpdate( retval, Results );
            Results.Log( Caller, JobInfo );
          }
        } // if - mutex locked
      } // if
      return retval;
    } // method Directory::Scan( ScanResults& Results )

    Directory::dirref_type Directory::
    update( ScanResults& Results,
            const DirectoryManager& DirectoryCollection,
            const StatInfo& StatInfo ) const
    {
      dirref_type       retval( new Directory( *this ) );

      DirectoryScanData scan_data( m_indexes,
                                   retval->m_indexes,
                                   Results,
                                   DirectoryCollection,
                                   Root( ) );

      retval->read_directory( scan_data, StatInfo );

      return retval;
    }


    void Directory::
    read_directory( DirectoryScanData& ScanData,
                    const StatInfo& StatInfo )
    {
      //-----------------------------------------------------------------
      // Gather all the names in the directory skipping anything
      //     that matches the ignore pattern
      //-----------------------------------------------------------------
      directory_container_type  cur_children;
      ignored_type              cur_ignored;

      ::read_directory( Fullname( ),
                        Root( ),
                        cur_children, cur_ignored,
                        m_indexes,
                        ScanData,
                        StatInfo );
      ScanData.s_results.State( Fullname( ), m_state );
      if ( m_state == DIRECTORY_NEW )
      {
        m_state = DIRECTORY_UPDATED;
      }
      for ( DirectoryScanData::searches_scan_data_type::iterator
              cur = ScanData.s_searches_scan_data.begin( ),
              last = ScanData.s_searches_scan_data.end( );
            cur != last;
            ++cur )
      {
        m_indexes[ cur->first ] = cur->second->SearchData( );
      }

      //-----------------------------------------------------------------
      // Update with the information gleaned from reading the directory
      //-----------------------------------------------------------------
      {
        //---------------------------------------------------------------
        // Find the new verses the old directories
        //---------------------------------------------------------------
        {
          filename_container_type          additions;
          filename_container_type          deletions;

          directory_container_type::const_iterator
            current = cur_children.begin( ),
            current_end = cur_children.end( )
            ;

          children_type::const_iterator
            previous = children( ).begin( ),
            previous_end = children( ).end( )
            ;

          while( ( previous != previous_end )
                 && ( current != current_end ) )
          {
            int       status = previous->compare( *current );

            if ( status == 0 )
            {
              //---------------------------------------------------------
              // When the two are the same, then the data just needs
              // to migrate from the old to the new.
              //---------------------------------------------------------
              ++previous;
              ++current;
            }
            else if ( status < 0 )
            {
              //---------------------------------------------------------
              // Deletion
              //---------------------------------------------------------
              deletions.push_back( *previous );
              ++previous;
            }
            else
            {
              //---------------------------------------------------------
              // Addition
              //---------------------------------------------------------
              additions.push_back( *current );
              ++current;
            }
          }
          while( previous != previous_end )
          {
            //-----------------------------------------------------------
            // Anything left in the old tree is flagged for deletion
            //-----------------------------------------------------------
            deletions.push_back( *previous );
            ++previous;
          }
          while( current != current_end )
          {
            //-----------------------------------------------------------
            // Anything left in the new tree is flagged for addition
            //-----------------------------------------------------------
            additions.push_back( *current );
            ++current;
          }
          //-------------------------------------------------------------
          // Remove the directories that no longer exist
          //-------------------------------------------------------------
          for ( filename_container_type::const_iterator
                  cur = deletions.begin( ),
                  last = deletions.end( );
                cur != last;
                ++cur )
          {
            m_subdirs.erase( *cur );
          }
          ScanData.s_results.Removed( Fullname( ), deletions );
          //-------------------------------------------------------------
          // Add any directories that have been created
          //-------------------------------------------------------------
          for ( filename_container_type::const_iterator
                  cur = additions.begin( ),
                  last = additions.end( );
                cur != last;
                ++cur )
          {
            m_subdirs.insert( *cur );
          }
          ScanData.s_results.Added( Fullname( ), additions );
        }
        //---------------------------------------------------------------
        //---------------------------------------------------------------
        //---------------------------------------------------------------
        // Record the last time this directory was modified.
        //---------------------------------------------------------------
        m_last_time_modified = StatInfo.mod_time;
      }
    }

    DirectoryScanData::scan_data::
    ~scan_data( )
    {
    }

  } // namespace - Cache

  Streams::IBinary& 
  operator>>( Streams::IBinary& Stream, Cache::Directory::dirref_type& Data )
  {
    Data.reset( new Directory( ) );
    // Data.Read( Stream, Data );
    return Stream;
  }
} // namespace - diskCache

namespace
{
  inline diskCache::Cache::Directory::timestamp_type
  do_stat( const std::string& Path, struct stat& StatStruct )
  {
    static const CHAR* const method_name( __FILE__ "::anonymous::do_stat: " );

    int err = GenericAPI::LStat( Path, StatStruct );

    AT( "err: " << err );      

    if ( StdErrLog.IsOpen( ) )
    {
      HERE( );
      std::ostringstream	msg;
      
      msg << "doStat: " << Path
          << " err: " << err
          << " errno: " << errno;
      StdErrLog( ErrorLog::DEBUG,
                 __FILE__, __LINE__,
                 msg.str( ) );
    }
    if( err != 0 )
    {
      //-----------------------------------------------------------------
      // An error has occurred while trying to get information about
      // this directory.
      // 1) Create% a readable error message
      //-----------------------------------------------------------------
      HERE( );
      string exc( method_name );
      exc += "Could not execute lstat() for \'";
      exc += Path;
      exc += '\'';
      exc += errnoMessage( );
      exc += " ";

      //-----------------------------------------------------------------
      /// \todo
      ///     Remove all subdirectories recursively
      //-----------------------------------------------------------------
      throw std::runtime_error( exc );
    }
    //-------------------------------------------------------------------
    // Handling of the 1 second bug
    //-------------------------------------------------------------------
    static double BUG_WINDOW = 1.0;

    LDASTools::AL::GPSTime dir( StatStruct.st_mtime, 0, LDASTools::AL::GPSTime::UTC );
    LDASTools::AL::GPSTime now;

    now.Now( );
    double diff( now - dir );
    if ( diff < 0 )
    {
      diff *= -1;
    }
    if ( diff <= BUG_WINDOW )
    {
      //-----------------------------------------------------------------
      // The time difference is within the window of opportunity so
      //   need to flag the directory as needing to be rescanned.
      //-----------------------------------------------------------------
      return diskCache::Cache::Directory::MODIFIED_RESET;
    }
    return StatStruct.st_mtime;
  }

   //---------------------------------------------------------------------------
   //
   /// \brief Get errno string message.
   ///   
   /// This method interpret error code into the string message.
   ///
   /// \return
   ///     Formatted error message.
   //---------------------------------------------------------------------------

   inline const string
   errnoMessage( )
   {
     std::ostringstream msg;
     msg << " errno=" << errno << " (";
     
     string errno_msg( strerror( errno ) );
     msg << errno_msg << ')';
   
   
     return msg.str();
   }

  void
  read_directory( const std::string& Path,
                  const std::string& RootDirectory,
                  directory_container_type& Children,
                  ignored_type& IgnoredDirectories,
                  Directory::index_container_type& OldData,
                  DirectoryScanData& ScanData,
                  const diskCache::Cache::StatInfo& Stat )
  {
    static const CHAR* const method_name( __FILE__ "::anonymous::read_directory: " );

    //-------------------------------------------------------------------
    // Ensure starting with empty lists
    //-------------------------------------------------------------------
    Children.erase( Children.begin( ),
                    Children.end( ) );
    IgnoredDirectories.erase( IgnoredDirectories.begin( ),
                              IgnoredDirectories.end( ) );
    //-------------------------------------------------------------------
    // Collect all entries for the directory.
    //-------------------------------------------------------------------
    typedef LDASTools::AL::Directory::block_read_type         entries_type;
    entries_type                                        entries;
    
    {
      static const char*                                close_func = "Close()";
      static const char*                                next_func = "Next()";
      static const char*                                open_func = "Open()";
      const char*                                       active_func;


      LDASTools::AL::Directory        current_dir( Path, false );

      try
      {
        //-----------------------------------------------------------------
        // Open
        //-----------------------------------------------------------------
        active_func = open_func;
        current_dir.Open( );
        //-----------------------------------------------------------------
        // Read
        //-----------------------------------------------------------------
        active_func = next_func;
        current_dir.Next( entries );
        //-----------------------------------------------------------------
        // Close
        //-----------------------------------------------------------------
        active_func = close_func;
        current_dir.Close( );

        {
          std::ostringstream    msg;

          msg << "Completed read of directory: "
              << Path
            ;
          GenericAPI::queueLogEntry( msg.str( ),
                                     GenericAPI::LogEntryGroup_type::MT_DEBUG,
                                     30,
                                     method_name,
                                     "CXX" );
        }

      }
      catch( const TimeoutException& Exception )
      {
        std::ostringstream   msg;

        msg << "TimeoutException for: " << current_dir.Name( )
            << " - " << active_func << " " << Exception.what( )
            << ")"
	  ;
   
        GenericAPI::queueLogEntry( msg.str( ),
                                   GenericAPI::LogEntryGroup_type::MT_DEBUG,
                                   0,
                                   method_name,
                                   "CXX_INTERFACE" );
        //-----------------------------------------------------------------
        // Retrow the exception so others can act upon it
        //-----------------------------------------------------------------
        throw;
      }
    }
    //-------------------------------------------------------------------
    // Process the entries
    //-------------------------------------------------------------------
    string              temp_file;
    DirectoryScanData   dsd( OldData, OldData, ScanData.s_results,
                             ScanData.DirManager( ),
                             RootDirectory );

    dsd.s_directory_root = RootDirectory;
    for( entries_type::const_iterator
           entry_cur = entries.begin( ),
           entry_last = entries.end( );
         entry_cur != entry_last;
         ++entry_cur )
    {
      static const std::string tmp( ".tmp" );

      //-----------------------------------------------------------------
      // Within this loop, the file should be identified.
      // Subsequent checks will only be done if the previous check
      //   failed.
      //-----------------------------------------------------------------
      //-----------------------------------------------------------------
      // \todo Check if the file should be ignored
      //-----------------------------------------------------------------
      if ( ( ( entry_cur->size( ) > 0 )
             && ( (*entry_cur)[ 0 ] == '.' ) )
           || ( ( entry_cur->size( ) >= tmp.size( ) )
                && ( entry_cur->compare( entry_cur->size( ) -
                                         tmp.size( ),
                                         std::string::npos,
                                         tmp ) == 0 ) ) )
      {
        continue;
      }
                     
      //-----------------------------------------------------------------
      /// \todo
      /// Check if any registered indexing scheme indexes this type of
      ///   file and if so, what assumption does it make about the file
      ///   type.
      //-----------------------------------------------------------------
      dsd.s_filename = *entry_cur;
      if ( diskCache::Cache::RegistrySingleton::ScanForMatch( dsd ) )
      {
        continue;
      }
      //-----------------------------------------------------------------
      // Check for leaf optimization
      //
      // If the link count != 2, then this directory is not a leaf.
      // If the link count == 2, then the directory is a leaf
      //    When it is a leaf, then we skip the stat and hence
      //    the check for symlink is also disabled.
      //-----------------------------------------------------------------
      if ( Stat.stat_buffer.st_nlink != 2 )
      {
          //-------------------------------------------------------------
          /// \todo
          /// Check if entry should be excluded from the scan.
          /// \note
          ///    This check assumes that patterns that match
          ///    are not directories
          //-------------------------------------------------------------
          //-------------------------------------------------------------
          // The following checks need to know the file type so a system
          //   call is done.
          //-------------------------------------------------------------
          temp_file = Path;
          temp_file += "/";
          temp_file += *entry_cur;
          
          struct stat stat_struct;
          do_stat( temp_file, stat_struct );

          if ( S_ISLNK( stat_struct.st_mode ) )
          {
              throw SymbolicLinkError( temp_file, method_name );
          }
          if ( S_ISDIR( stat_struct.st_mode ) )
          {
              //---------------------------------------------------------
              // Check to see if the directory should be excluded
              //---------------------------------------------------------
              if ( ExcludedDirectoriesSingleton::IsExcluded( *entry_cur ) )
              {
                  std::ostringstream    msg;

                  msg << "Excluding directory: " << temp_file
                      ;
                  GenericAPI::queueLogEntry( msg.str( ),
                                             GenericAPI::LogEntryGroup_type::MT_NOTE,
                                             0,
                                             method_name,
                                             "CXX_INTERFACE" );
                  IgnoredDirectories.insert( *entry_cur );
                  continue;
              }
              //---------------------------------------------------------
              /// \todo
              /// Add to the list of directories
              //---------------------------------------------------------
              Children.push_back( directory_element_type( *entry_cur, false ) );
              continue;
          }
          //-------------------------------------------------------------
          // Everything else is a file that is unmanaged.
          //-------------------------------------------------------------
      }
      
    }
    //-------------------------------------------------------------------
    // Now have all search engines do any necessary cleanup.
    //-------------------------------------------------------------------
    dsd.s_filename = Path;

    diskCache::Cache::RegistrySingleton::OnDirectoryClose( dsd );
    //-------------------------------------------------------------------
    // Put the directories in order for easy post processing
    //-------------------------------------------------------------------
    Children.sort( );
  }
} // namespace - anonymous
