#include <ldas_tools_config.h>

#include <list>

#include "ldastoolsal/AtExit.hh"
#include "ldastoolsal/mutexlock.hh"
#include "ldastoolsal/TaskThread.hh"
#include "ldastoolsal/ThreadPool.hh"

using LDASTools::AL::AtExit;
using LDASTools::AL::MutexLock;
using LDASTools::AL::TaskThread;
using LDASTools::AL::ThreadPool;

typedef	std::list< TaskThread* >	thread_container_type;

static bool				at_exit_initialized = false;

static void on_exit( );

//-----------------------------------------------------------------------
/// \brief Obtain lock
///
/// The use of a function ensures the proper initialization of the data
/// without having to depend on the link order initialization.
//-----------------------------------------------------------------------
inline MutexLock::baton_type
baton( )
{
  static MutexLock::baton_type	baton;

  return baton;
}

//-----------------------------------------------------------------------
/// \brief Obtain the queue of available resources
///
/// The use of a function ensures the proper initialization of the data
/// without having to depend on the link order initialization.
//-----------------------------------------------------------------------
inline static thread_container_type&
available( )
{
  static thread_container_type retval;

  return retval;
}

//-----------------------------------------------------------------------
/// \brief Obtain the queue of resources currently in use.
///
/// The use of a function ensures the proper initialization of the data
/// without having to depend on the link order initialization.
//-----------------------------------------------------------------------
inline static thread_container_type&
in_use( )
{
  static thread_container_type retval;

  return retval;
}

namespace LDASTools
{
  namespace AL
  {
    //-------------------------------------------------------------------
    /// Secure a resource from the pool of resouces.
    /// If there is no resouce available at the time of the call,
    /// a new resouce is allocated and returned to the caller.
    //-------------------------------------------------------------------
    TaskThread* ThreadPool::
    Acquire( )
    {
      if ( AtExit::IsExiting( ) )
      {
	//---------------------------------------------------------------
	// Do not give out any more threads if the application is
	// in the process of shutting down.
	//---------------------------------------------------------------
	return (TaskThread*)NULL;
      }
      TaskThread*	retval = (TaskThread*)NULL;

      {
	MutexLock	lock( baton( ),
			      __FILE__, __LINE__ );

	if ( at_exit_initialized == false )
	{
	  AtExit::Append( on_exit,
			  "ThreadPool::on_exit",
			  150 );
	  at_exit_initialized = true;
	}
      
#if old
	if ( available( ).size( ) == 0 )
	{
	  available( ).push_front( new TaskThread );
	}
	in_use( ).push_front( available( ).front( ) );
	available( ).pop_front( );

	retval = in_use( ).front( );
#else
	if ( available( ).empty( ) == false )
	{
	  in_use( ).push_front( available( ).front( ) );
	  available( ).pop_front( );
	  retval = in_use( ).front( );
	}
#endif /* old */
      }

#if old
#else
      if ( retval == (TaskThread*)NULL )
      {
	retval = new TaskThread;

	MutexLock	lock( baton( ),
			      __FILE__, __LINE__ );
	in_use( ).push_front( retval );
      }

#endif
	
      return retval;
    }

    //-------------------------------------------------------------------
    /// Recyle the resource.
    //-------------------------------------------------------------------
    void ThreadPool::
    Relinquish( TaskThread* Resource )
    {
      if ( Resource )
      {
	bool exiting = true;
	{
	  MutexLock lock( baton( ),
			  __FILE__, __LINE__ );

	  in_use( ).remove( Resource );
	  if ( AtExit::IsExiting( ) == false )
	  {
	    available( ).push_back( Resource );
	    exiting = false;
	  }
	}
	if ( exiting )
	{
	  // delete Resource;
	}
      }
    }
  
    //-------------------------------------------------------------------
    /// \note
    ///     This method should not be called by others.
    ///     It is provided to help facilitate cleanup at program exit.
    //-------------------------------------------------------------------
    void ThreadPool::
    Reset( )
    {
      MutexLock	lock( baton( ),
		      __FILE__, __LINE__ );
      //-----------------------------------------------------------------
      /// Get rid of threads currently doing something interesting
      //-----------------------------------------------------------------
      for ( thread_container_type::const_iterator
	      cur = in_use( ).begin( ),
	      last = in_use( ).end( );
	    cur != last;
	    ++cur )
      {
	delete( *cur );
      }
      in_use( ).erase( in_use( ).begin( ), in_use( ).end( ) );
      //-----------------------------------------------------------------
      /// Get rid of threads waiting to do something interesting
      //-----------------------------------------------------------------
      for ( thread_container_type::const_iterator
	      cur = available( ).begin( ),
	      last = available( ).end( );
	    cur != last;
	    ++cur )
      {
	delete( *cur );
      }
      available( ).erase( available( ).begin( ), available( ).end( ) );
    }
  } // namespace - AL

} // namespace - LDASTools

static void
on_exit( )
{
  {
    MutexLock	lock( baton( ),
		      __FILE__, __LINE__ );

    for ( thread_container_type::iterator
	    cur = in_use( ).begin( ),
	    last = in_use( ).end( );
	  cur != last;
	  ++cur )
    {
      if ( *cur )
      {
	(*cur)->Halt( );
      }
    }
  }
  while( in_use( ).empty( ) == false )
  {
    //-------------------------------------------------------------------
    // Give time for tasks to complete
    //-------------------------------------------------------------------
    {
      for ( thread_container_type::iterator
	      cur = in_use( ).begin( ),
	      last = in_use( ).end( );
	    cur != last;
	     )
      {
#if 0
	std::cerr << "DEBUG: typeid: " << typeid( *cur ).name( )
		  << " ptr: " << (void*)(*cur)
		  << " name: " << ( ( (*cur)->Name( ) )
				    ? (*cur)->Name( )
				    : "unknown" )
		  << " IsAlive: " << ( (*cur)->IsAlive( )
				       ? "true" : "false" )
		  << " state: " << (*cur)->State( )
		  << " CancellationType: " << (*cur)->CancellationType( )
	  
		  << std::endl
	  ;
#endif /* 0 */
	if ( ( (*cur)->IsAlive( ) == true )
	     && (*cur)->State( ) != TaskThread::TASK_THREAD_EXITING )
	{
	  (*cur)->Halt( );
	}
	if ( (*cur)->IsAlive( ) == false )
	{
	  (*cur)->Join( );
	  cur = in_use( ).erase( cur );
	  continue;
	}
	++cur;
      }
    }
    struct timespec duration;

    duration.tv_sec = 5;
    duration.tv_nsec = 0;

    nanosleep( &duration, NULL );
  }
  {
    struct timespec duration;

    duration.tv_sec = 10;
    duration.tv_nsec = 0;

    nanosleep( &duration, NULL );
  }
  ThreadPool::Reset( );
}
