#include <stdlib.h>
#include <string.h>
#include <iostream>
#include <list>
#include <map>

#include "ldastoolsal/AtExit.hh"
#include "ldastoolsal/mutexlock.hh"
#include "ldastoolsal/ReadWriteLock.hh"

namespace anonymous
{
  void exit_handler( );
}

extern "C"
{
  typedef void(*atexit_parm_type)();
}

SINGLETON_INSTANCE_DEFINITION(SingletonHolder<AtExit>)

namespace
{
  using LDASTools::AL::AtExit;
  using LDASTools::AL::MutexLock;
  using LDASTools::AL::ReadWriteLock;

  static bool	at_exit_handler_registered = false;
  static void	cleanup( );

  //---------------------------------------------------------------------
  // This code is derived from:
  //  http://www.drdobbs.com/the-standard-librarian-what-are-allocato/184403759?pgno=1
  //
  //---------------------------------------------------------------------
  template <class T> class malloc_allocator
  {
  public:
    typedef T                 value_type;
    typedef value_type*       pointer;
    typedef const value_type* const_pointer;
    typedef value_type&       reference;
    typedef const value_type& const_reference;
    typedef std::size_t       size_type;
    typedef std::ptrdiff_t    difference_type;
  
    template <class U> 
    struct rebind {
      typedef malloc_allocator<U> other;
    };

    malloc_allocator() {}
    malloc_allocator(const malloc_allocator&)
    {
    }

    template <class U> 
    malloc_allocator(const malloc_allocator<U>&)
    {
    }

    ~malloc_allocator()
    {
    }

    pointer address(reference x) const
    {
      return &x;
    }

    const_pointer address(const_reference x) const
    { 
      return x;
    }

    pointer allocate(size_type n, const_pointer = 0)
    {
      void* p = std::malloc(n * sizeof(T));
      if (!p)
	throw std::bad_alloc();
      return static_cast<pointer>(p);
    }

    void deallocate(pointer p, size_type)
    {
      std::free(p);
    }

    size_type max_size() const
    { 
      return static_cast<size_type>(-1) / sizeof(T);
    }

    void construct(pointer p, const value_type& x)
    { 
      new(p) value_type(x); 
    }

    void destroy(pointer p)
    {
      p->~value_type();
    }

  private:
    void operator=(const malloc_allocator&);
  };

  template<> class malloc_allocator<void>
  {
    typedef void        value_type;
    typedef void*       pointer;
    typedef const void* const_pointer;

    template <class U> 
    struct rebind { typedef malloc_allocator<U> other; };
  };


  template <class T>
  inline bool operator==(const malloc_allocator<T>&, 
			 const malloc_allocator<T>&)
  {
    return true;
  }

  template <class T>
  inline bool operator!=(const malloc_allocator<T>&, 
			 const malloc_allocator<T>&)
  {
    return false;
  }

  //---------------------------------------------------------------------
  //
  //---------------------------------------------------------------------
  class ExitQueueNode
  {
  public:
    ~ExitQueueNode( )
    {
      while( data_container.size( ) > 0 )
      {
	data_container.back( ).purge( );
	data_container.pop_back( );
      }
    }

    void Append( AtExit::ExitFunction Function,
		 const char* Name )
    {
      data_container.push_back( data_type( Function, Name ) );
    }

  private:
    class func_node {
    public:
      typedef AtExit::ExitFunction function_type;

      func_node( function_type Function,
		 const char* Name )
	: function( Function ),
	  name( ::strdup( Name ) )
      {
      }

      void
      purge( )
      {
	if ( function )
	{
	  (*function)();
	  function = (function_type)NULL;
	}
	if ( name )
	{
	  ::free( const_cast< char* >( name ) );
	  name = (const char*)NULL;
	}
      }
    private:
      function_type	function;
      const char*	name;
    };

    typedef func_node	data_type;
    typedef std::list< data_type, malloc_allocator< data_type > >
    data_container_type;

    data_container_type	data_container;
  };

  class ExitQueue
  {
  public:
    ExitQueue( )
      :	is_exiting_baton( /*__FILE__, __LINE__,*/ false ),
	exit_queue_baton( __FILE__, __LINE__, false )
    {
    }

    inline void
    Append( AtExit::ExitFunction Function,
	    const char* Name,
	    int Ring )
    {
      MutexLock	l( exit_queue_baton,
				   __FILE__, __LINE__ );

      if ( IsExiting( ) )
      {
	(*Function)( );
	return;
      }
      if ( ! at_exit_handler_registered )
      {
	(void)atexit( (atexit_parm_type)(cleanup) );
	at_exit_handler_registered = true;
      }
      exit_queue[ Ring ].Append( Function, Name );
      
    }

    inline void
    Cleanup( )
    {
      MutexLock	l( exit_queue_baton,
				   __FILE__, __LINE__ );

      bool			state;

      state = IsExiting( true );

      while( exit_queue.size( ) > 0 )
      {
	exit_queue_type::iterator
	  cur = exit_queue.end( );
	--cur;
	exit_queue.erase( cur );
      }

      IsExiting( state );

    }

    inline bool
    IsExiting( ) const
    {
      ReadWriteLock	l( is_exiting_baton,
				   ReadWriteLock::READ,
				   __FILE__, __LINE__ );

      return is_exiting;
    }

    inline bool
    IsExiting( bool Value )
    {
      ReadWriteLock	l( is_exiting_baton,
				   ReadWriteLock::WRITE,
				   __FILE__, __LINE__ );

      bool retval = is_exiting;

      is_exiting = Value;

      return retval;
    }

  private:
    typedef std::map< int, ExitQueueNode, std::less< int >,
		      malloc_allocator< std::pair< int, ExitQueueNode > > >
    exit_queue_type;

    bool			is_exiting;
    ReadWriteLock::baton_type	is_exiting_baton;

    exit_queue_type		exit_queue;
    MutexLock::baton_type	exit_queue_baton;
  };

  ExitQueue&
  exit_queue( )
  {
    static ExitQueue	eq;

    return eq;
  }

  void
  cleanup( )
  {
    exit_queue( ).IsExiting( true );

    AtExit::Cleanup( );
  }

} // namespace - anonymous

namespace LDASTools
{
  namespace AL
  {
    void AtExit::
    Append( AtExit::ExitFunction Function,
	    const std::string& Name,
	    int Ring )
    {
      exit_queue( ).Append( Function, Name.c_str( ), Ring );
    }

    void AtExit::
    Cleanup( )
    {
      exit_queue( ).Cleanup( );
    }

    bool AtExit::
    IsExiting( )
    {
      return exit_queue( ).IsExiting( );
    }
  } // namespace - AL
} // namespace - LDASTools
