#ifdef HAVE_CONFIG_H
#include <ldas_tools_config.h>
#endif /* HAVE_CONFIG_H */

#include <signal.h>

#include <sstream>

#include "ldastoolsal/AtExit.hh"
#include "ldastoolsal/ErrorLog.hh"
#include "ldastoolsal/mutexlock.hh"
#include "ldastoolsal/Pool.hh"
#include "ldastoolsal/SystemCallTask.hh"
#include "ldastoolsal/TaskThread.hh"
#include "ldastoolsal/Timeout.hh"

#include "genericAPI/Logging.hh"
#include "genericAPI/MountPointStatus.hh"
#include "genericAPI/Stat.hh"
#include "genericAPI/StatFork.hh"
#include "genericAPI/StatDirect.hh"
#include "genericAPI/StatPool.hh"


using LDASTools::AL::AtExit;
using LDASTools::AL::ErrorLog;
using LDASTools::AL::MutexLock;
using LDASTools::AL::Pool;
using LDASTools::AL::SystemCallTask;
using LDASTools::AL::TaskThread;
using LDASTools::AL::Timeout;
using LDASTools::AL::TimeoutException;
using LDASTools::AL::TIMEOUT_COMPLETED;
using LDASTools::AL::StdErrLog;

using GenericAPI::MountPointStatus;
using GenericAPI::StatPool;
using GenericAPI::StatDirect;

// Timeout value for lstat
static int lstat_timeout_value = 5;

namespace
{
  void
  wakeup( int SignalNumber )
  {
    signal( SignalNumber, wakeup );
  }
}

namespace
{
  //---------------------------------------------------------------------
  /// \brief Perform stat or lstat system call
  ///
  /// This class wraps the lstat and stat system call.
  /// The benifit of using this call is that the caller is protected
  /// from NFS servers that are hung.
  //---------------------------------------------------------------------
  class lStat
    : public SystemCallTask
  {
  public:
    typedef struct stat buffer_type;

    //-------------------------------------------------------------------
    /// \brief Specify the type of request.
    //-------------------------------------------------------------------
    enum mode_type {
      //-----------------------------------------------------------------
      /// Special case to handle initial condition
      //-----------------------------------------------------------------
      MODE_NONE,
      //-----------------------------------------------------------------
      /// Perform stat()
      //-----------------------------------------------------------------
      MODE_STAT,
      //-----------------------------------------------------------------
      /// Perform lstat()
      //-----------------------------------------------------------------
      MODE_LSTAT
    };

    lStat( );

    virtual ~lStat( );

    //-------------------------------------------------------------------
    /// \brief Initialize the instance for use.
    ///
    /// \param[in] Path
    ///     The pathname of any file within the mounted filesystem.
    ///
    /// \param[in] Mode
    ///     Variation of stat call to use.
    //-------------------------------------------------------------------
    void Reset( const std::string& Path, mode_type Mode );

    //-------------------------------------------------------------------
    /// \brief Syncronize the callers buffer with the local buffer.
    //-------------------------------------------------------------------
    void Sync( buffer_type& StatBuffer );

  protected:
    //-------------------------------------------------------------------
    /// \brief Do the stat or lstat system call
    ///
    /// \return
    ///     Upon successful completion, the value zero is returned.
    ///     Upon failure, the value -1 is returned and errno
    ///     is set.
    //-------------------------------------------------------------------
    virtual int eval( );

  private:
    mode_type	m_mode;
    std::string	m_path;
    buffer_type	m_statbuf;
  };

  typedef LDASTools::AL::SharedPtr< lStat > value_type;

  //---------------------------------------------------------------------
  /// \brief Collection of reusable stat buffers
  //---------------------------------------------------------------------
  class StatPool
    : public Pool< value_type >
  {
    SINGLETON_TS_DECL( StatPool );
  };

  //---------------------------------------------------------------------
  /// \brief Collection of stat buffers that have timed out
  ///
  /// This is only a stop gap solution to prevent the memory resource
  /// from being recycled while being used by another thread.
  ///
  /// \todo
  ///    Instead of collecting these buffers in a pool where they will
  ///    be leaking memory, the thread needs to take ownership of the
  ///    object and destroy it once the thread completes.
  //---------------------------------------------------------------------
  class StatTrashPool
    : public Pool< value_type >
  {
    SINGLETON_TS_DECL( StatTrashPool );
  };

  //---------------------------------------------------------------------
  /// \brief Create a new instance for use in the pool
  //---------------------------------------------------------------------
  value_type create( );

  int stat_wrapper( const std::string& Path,
		    lStat::buffer_type& StatBuffer,
		    lStat::mode_type Mode,
		    const char* const MethodName );

} // namespace - anonymous

namespace GenericAPI
{
  int
  init_lstat_interupt( )
  {
    // static const int interupt = SIGTERM;
    static const int interupt = TaskThread::CANCEL_SIGNAL_ABANDON;
    static bool initialized = false;

    if ( initialized == false )
    {
      static MutexLock::baton_type baton;

      MutexLock l( baton,
		   __FILE__, __LINE__ );

      if ( initialized == false )
      {
	const StatPool::info_type lstat_base = StatPool::StatType( );
    
	if ( ( interupt > 0 )
	     && ( dynamic_cast< const StatDirect* >( lstat_base ) ) )
	{
	  wakeup( interupt );	// Initialize the signal handler
	}
	initialized = true;
      }
    }

    return interupt;
  }

  int
  lstat_timeout( )
  {
    return lstat_timeout_value;
  }

  int
  SetStatTimeout( int Value )
  {
    static const CHAR* const
      method_name( "GenericAPI::SetStatTimeout" );

    int	retval = lstat_timeout_value;

    std::ostringstream	msg;

    msg << "Modifying lstat timeout from: " << retval
	<< " to: " << Value
      ;
    queueLogEntry( msg.str( ),
		   GenericAPI::LogEntryGroup_type::MT_DEBUG,
		   0,
		   method_name,
		   "VARIABLE" );
    
    lstat_timeout_value = Value;

    return retval;
  }

  int
  Stat( const std::string& Filename, struct stat& StatBuffer )
  {
    return stat_wrapper( Filename, StatBuffer,
			 lStat::MODE_STAT,
			 "GenericAPI::Stat" );
  }

  int
  LStat( const std::string& Filename, struct stat& StatBuffer )
  {
    return stat_wrapper( Filename, StatBuffer,
			 lStat::MODE_LSTAT,
			 "GenericAPI::LStat" );
  }
}

namespace {
  lStat::
  lStat( )
    : m_mode( MODE_NONE )
  {
  }

  lStat::
  ~lStat( )
  {
  }

  //---------------------------------------------------------------------
  /// This actually does the system call.
  /// The local buffer is used to prevent memory corruuption in the
  /// instances where the system call times out.
  //---------------------------------------------------------------------
  int lStat::
  eval( )
  {
    int retcode = -1;
    // static const CHAR* method_name( "LDASTools::AL::lStat::eval" );

    switch( m_mode )
    {
    case MODE_STAT:
      retcode = stat( m_path.c_str( ), &m_statbuf );
      break;
    case MODE_LSTAT:
      retcode = lstat( m_path.c_str( ), &m_statbuf );
      break;
    default:
      break;
    }
    return retcode;
  }

  //---------------------------------------------------------------------
  /// Simply record caller's information for future use.
  //---------------------------------------------------------------------
  inline void lStat::
  Reset( const std::string& Path, mode_type Mode  )
  {
    m_path = Path;
    m_mode = Mode;
  }

  //---------------------------------------------------------------------
  /// Syncronize the callers buffer with the local buffer being
  /// careful to ensure that the caller has specified a buffer
  /// into which the results are to be saved.
  ///
  /// \note
  ///     This call should only be made if Request returns successful.
  ///     It is the responsibility of the developer to ensure this
  ///     condition.
  //---------------------------------------------------------------------
  inline void lStat::
  Sync( buffer_type& StatBuffer )
  {
    StatBuffer = m_statbuf;
  }

  //---------------------------------------------------------------------
  /// \brief Core to perform timed stat or lstat system calls.
  //---------------------------------------------------------------------
  inline int
  stat_wrapper( const std::string& Path,
		lStat::buffer_type& StatBuffer,
		lStat::mode_type Mode,
		const char* const MethodName )
  {
    int retval = -1;

    if ( MountPointStatus::Status( Path )
	 == MountPointStatus::STATE_OFFLINE )
    {
      errno = EINTR;
      return retval;
    }

    //-------------------------------------------------------------------
    // PR#3015 -
    // Be careful with the lstat call since it can lock up under
    //   nfs usage and not return until the server is once again
    //   available.
    // This condition is avoided by setting a maximum time for this
    //   call to return before causing the lstat call to be interupted.
    //-------------------------------------------------------------------
    //-----------------------------------------------------------
    // Get entry from pool to reduce system allocation overhead
    //-----------------------------------------------------------
    value_type	syscall( StatPool::Instance( ).Request( create ) );
    int	status = TIMEOUT_COMPLETED;

    if ( syscall )
    {
      syscall->Reset( Path, Mode );
      try
      {
	//---------------------------------------------------------
	// PR#341
	// Carefully setup to only wait a certain amount of time
	// for this call to complete.
	//---------------------------------------------------------
	Timeout( syscall,
		 GenericAPI::lstat_timeout( ),
		 status );
      }
      catch( const TimeoutException& Exception )
      {
	errno = EINTR;

	std::ostringstream   msg;

	msg << "TimeoutException for: " << Path
	    << " (" << Exception.what( )
	    << ")"
	  ;
   
	GenericAPI::queueLogEntry( msg.str( ),
				   GenericAPI::LogEntryGroup_type::MT_DEBUG,
				   0,
				   MethodName,
				   "CXX_INTERFACE" );
	if ( StdErrLog.IsOpen( ) )
	{
	  StdErrLog( ErrorLog::DEBUG,
		     __FILE__, __LINE__,
		     Exception.what( ) );
	}
      }
      catch( const std::exception& Exception )
      {
	std::ostringstream   msg;

	msg << "std::exception for: " << Path
	    << " (" << Exception.what( )
	    << ")"
	  ;
   
	GenericAPI::queueLogEntry( msg.str( ),
				   GenericAPI::LogEntryGroup_type::MT_DEBUG,
				   0,
				   MethodName,
				   "CXX_INTERFACE" );
	if ( StdErrLog.IsOpen( ) )
	{
	  StdErrLog( ErrorLog::DEBUG,
		     __FILE__, __LINE__,
		     Exception.what( ) );
	}
      }
      catch( ... )
      {
	std::ostringstream   msg;

	msg << "unknown exception for: " << Path
	  ;
   
	GenericAPI::queueLogEntry( msg.str( ),
				   GenericAPI::LogEntryGroup_type::MT_DEBUG,
				   0,
				   MethodName,
				   "CXX_INTERFACE" );
	// :TODO: Unknown exception
      }
      //-----------------------------------------------------------------
      // Need to see how the call completed.
      //-----------------------------------------------------------------
      if ( status & TIMEOUT_COMPLETED )
      {
	//---------------------------------------------------------------
	// Transfer results so caller can make use of them.
	//---------------------------------------------------------------
	retval = syscall->SystemReturnCode( );
	errno = syscall->SystemErrNo( );
	if ( retval == 0 )
	{
	  syscall->Sync( StatBuffer );
	}
	//---------------------------------------------------------------
	// Recycle this memory for future use.
	//---------------------------------------------------------------
	StatPool::Instance( ).Relinquish( syscall );
      }
      else
      {
	//---------------------------------------------------------------
	// Put onto the trash heap to prevent the memory from being
	// recycled and causing memory corruption.
	//---------------------------------------------------------------
	StatTrashPool::Instance( ).Relinquish( syscall );
      }
    }
    if ( errno == EINTR )
    {
      //---------------------------------------------------------
      // An interupted system call means that the task took too
      // long to complete. Register this file system as being
      // offline.
      //---------------------------------------------------------
      MountPointStatus::Offline( Path );
    }

    return retval;
  }

  //---------------------------------------------------------------------
  //---------------------------------------------------------------------
  SINGLETON_TS_INST( StatPool );

  StatPool::
  StatPool( )
  {
    AtExit::Append( singleton_suicide,
		    "<anonymous>::StatPool::singleton_suicide",
		    5 );
  }

  //---------------------------------------------------------------------
  //---------------------------------------------------------------------
  SINGLETON_TS_INST( StatTrashPool );

  StatTrashPool::
  StatTrashPool( )
  {
    AtExit::Append( singleton_suicide,
		    "<anonymous>::StatTrashPool::singleton_suicide",
		    5 );
  }

  //---------------------------------------------------------------------
  /// This routine is used to create an new instance of the lStat
  /// class if none are available in the pool.
  //---------------------------------------------------------------------
  value_type
  create( )
  {
    value_type	retval( new value_type::element_type );

    return retval;
  }

}
