/* -*- mode: c++; c-basic-offset: 3 -*- */
#ifndef RDS_FRAME_HH
#define RDS_FRAME_HH

#include <memory>   
#include <string>   
#include <vector>   

#include "ldastoolsal/types.hh"
#include "ldastoolsal/SharedPtr.hh"

#include "framecpp/FrameH.hh"

#include "framecpp/FrAdcData.hh"
#include "framecpp/FrProcData.hh"
#include "framecpp/FrRawData.hh"
#include "framecpp/FrVect.hh"
#include "framecpp/Time.hh"

#include "genericAPI/Logging.hh"

class RDSStream;

namespace FrameAPI
{
   namespace RDS
   {
      class Options
      {
      public:
	 typedef INT_4U seconds_per_frame_type;
	 typedef INT_4U start_type;
	 typedef INT_4U end_type;
	 typedef INT_2U compression_level_type;
	 typedef INT_4U frames_per_file_type;
	 typedef INT_4U rds_level_type;
	 typedef FrameCPP::FrVect::compression_scheme_type	compression_method_type;

	 Options( );

	 Options( const start_type start,
		  const end_type stop,
		  const char* type,
		  const char* compression_method,
		  const compression_level_type level,
		  const bool VerifyChecksum,
		  const bool VerifyFrameChecksum,
		  const bool VerifyTimeRange,
		  const bool VerifyDataValid,
		  const frames_per_file_type FramesPerFile,
		  const seconds_per_frame_type SecondsPerFrame,
		  const bool AllowShortFrames,
		  const bool GenerateFrameChecksum,
		  const bool FillMissingDataValidArray,
		  const bool VerifyFilenameMetadata );
	 Options( const Options& Source );

	 bool AllowShortFrames( ) const;

	 void AllowShortFrames( bool Value );

	 compression_level_type CompressionLevel( ) const;

	 void CompressionLevel( compression_level_type Value );

	 compression_method_type CompressionMethod( ) const;

	 void CompressionMethod( const std::string& Value );

	 bool CreateChecksumPerFrame( ) const;

	 void CreateChecksumPerFrame( bool Value );

	 bool FillMissingDataValidArray( ) const;

	 void FillMissingDataValidArray( bool Value );

	 frames_per_file_type FramesPerFile( ) const;

	 void FramesPerFile( frames_per_file_type FPF );

	 bool GenerateFrameChecksum( ) const;

	 void GenerateFrameChecksum( bool Value );

	 bool HistoryRecord( ) const;

	 void HistoryRecord( bool Value );

	 end_type OutputTimeEnd( ) const;

	 void OutputTimeEnd( end_type Value);

	 start_type OutputTimeStart( ) const;

	 void OutputTimeStart( start_type Value );

	 const std::string& OutputType( ) const;

	 void OutputType( const std::string& Value);

	 rds_level_type RDSLevel( ) const;

	 void RDSLevel( rds_level_type Level );

	 seconds_per_frame_type SecondsPerFrame( ) const;

	 void SecondsPerFrame( seconds_per_frame_type SPF );

	 bool VerifyChecksum( ) const;

	 bool VerifyChecksumOfFrame( ) const;

	 bool VerifyChecksumPerFrame( ) const;

	 void VerifyChecksumPerFrame( bool Value );

	 bool VerifyChecksumOfStream( ) const;

	 void VerifyChecksumOfStream( bool Value );

	 bool VerifyDataValid( ) const;

	 void VerifyDataValid( bool Value );

	 bool VerifyFilenameMetadata( ) const;

	 void VerifyFilenameMetadata( bool Value );

	 bool VerifyTimeRange( ) const;

	 void VerifyTimeRange( bool Value );

      private:
	 bool				history_record;
	 //: Method for compression
	 compression_method_type	m_compression_method;
	 //: Level for compression
	 compression_level_type		m_compression_level;
	 //: Number of frames per file
	 frames_per_file_type		m_frames_per_file;
	 //: If checksum for the input file should be verified
	 bool				m_input_verify_checksum;
	 //: If input frames' checksums should be verified
	 bool				m_input_verify_checksum_per_frame;
	 //: If input ADC channel's dataValid field should be checked
	 bool				m_input_verify_data_valid;
	 //: Verify that the frame's name and internal metadata agree
	 bool				m_input_verify_filename_metadata;
	 //: If input frame's filename should be validated against its time range
	 bool				m_input_verify_time_range;
	 //: Allow generation of short final frames
	 bool				m_output_allow_short_frames;
	 //: If the checksum should be created for each frame on output
	 bool				m_output_checksum_per_frame;
	 //: Allow creation of missing dataValid array
	 bool				m_output_fill_data_valid_array;
	 //: Time when to start writing data to frame file
	 INT_4U				m_output_start;
	 //: Time when to stop writing data to frame file
	 INT_4U				m_output_stop;
	 //: Use type of frame file name
	 std::string			m_output_type;
	 //: The current rds level
	 // Not really a command line option but is calculated from the type
	 //  field.
	 rds_level_type			m_rds_level;
	 //: Number of seconds of data for each frame
	 seconds_per_frame_type		m_seconds_per_frame;

	 //: The equal operator is not allowed since all data members are constant
	 const Options& operator=( const Options& RHS );

	 std::string validate_type( const char* Type );
      }; // Class - Options

      //-----------------------------------------------------------------
      //-----------------------------------------------------------------
      class FileOptions
	 : public Options
      {
      public:
	 FileOptions( );

	 FileOptions( const INT_4U Start,
		      const INT_4U Stop,
		      const char* dir,
		      const char* type,
		      const char* compression_method,
		      const INT_2U level,
		      const bool VerifyChecksum,
		      const bool VerifyFrameChecksum,
		      const bool VerifyTimeRange,
		      const bool VerifyDataValid,
		      const frames_per_file_type FramesPerFile,
		      const seconds_per_frame_type SecondsPerFrame,
		      const bool AllowShortFrames,
		      const bool GenerateFrameChecksum,
		      const bool FillMissingDataValidArray,
		      const bool VerifyFilenameMetadata,
		      const std::string& MD5SumOutputDirectory );
	 
	 FileOptions( const FileOptions& Source );

	 const std::string& DirectoryOutputFrames( ) const;

	 const std::string& DirectoryOutputMD5Sum( ) const;

	 const std::string& OutputDirectory( ) const;

	 const std::string& MD5SumOutputDirectory( ) const;

	 void DirectoryOutputFrames( const std::string& Value );

	 void DirectoryOutputMD5Sum( const std::string& Value );

      private:
	 //: Directory to write result frames to
	 std::string	m_output_directory;
	 //: Where to put the md5sum calculations
	 std::string	m_md5sum_output_directory;

      };
   } // namespace - RDS
} // namespace - FrameAPI

//-----------------------------------------------------------------------
///   
/// \brief Base functional to create RDS frames.
///   
//-----------------------------------------------------------------------
class RDSFrame
{
public:
   typedef std::vector< std::string > frame_file_container_type;
   typedef std::vector< std::string > channel_container_type;
   typedef ::FrameAPI::RDS::Options Options;
   
   typedef LDASTools::AL::SharedPtr< RDSStream >	stream_type;

   //--------------------------------------------------------------------
   ///   
   /// \brief Constructor.
   ///
   /// \param[in] frame_files
   ///     A list of frame file names.
   /// \param[in] channels
   ///     A list of channels to extract from original frames
   ///     (only channel names are allowed).
   /// \param[in] CommandOptions
   ///     Collection of flags used to direct the construction
   ///     of the Reduced Data Set results.
   ///
   //--------------------------------------------------------------------
   RDSFrame( const char* frame_files, 
             const char* channels,
	     const Options& CommandOptions );

   //--------------------------------------------------------------------
   ///   
   /// \brief Constructor.
   ///
   /// \param[in] frame_files
   ///     A list of frame file names.
   /// \param[in] channels
   ///     A list of channels to extract from original frames
   ///     (only channel names are allowed).
   /// \param[in] CommandOptions
   ///     Collection of flags used to direct the construction
   ///     of the Reduced Data Set results.
   ///
   //--------------------------------------------------------------------
   RDSFrame( const frame_file_container_type& frame_files, 
             const channel_container_type& channels,
	     const Options& CommandOptions );

   virtual ~RDSFrame();
   
   INT_4U GetNumberOfChannels( ) const;
   INT_4U GetNumberOfFrameGroups( ) const;
   INT_4U GetNumberOfFrameFiles( ) const;

   void ProcessRequest( stream_type Output );

protected:
   // Useful typedefs
   typedef LDASTools::AL::SharedPtr< FrameCPP::FrameH >	frame_h_type;
   typedef LDASTools::AL::SharedPtr< FrameCPP::FrAdcData >	fr_adc_data_type;
   typedef LDASTools::AL::SharedPtr< FrameCPP::FrProcData >	fr_proc_data_type;

   typedef std::vector< std::string >::const_iterator stringv_const_iterator;

   typedef enum {
      STOP_DATA,
      STOP_USER
   } stop_request_type;

   Options	m_options;

   RDSFrame( const Options& CommandOptions );

   virtual stop_request_type stopRequest( ) const = 0;

   virtual void processChannel( fr_adc_data_type Adc )
      = 0;

   virtual void processChannel( fr_proc_data_type Proc )
      = 0;
			
   // Close the current frame file
   virtual void writeFrameToStream( );

   template< typename Functor >
   void foreachChannel( Functor& Func ) const
   {
      for ( INT_4U
	       cur = 0,
	       last = GetNumberOfChannels( );
	    cur != last;
	    ++cur )
      {
	 Func( getChannelName( cur ) );
      }
   }

   // Accessors
   static FrameCPP::FrProcData*
   getProcChannel(  const std::string& name, 
		    FrameCPP::FrameH::procData_type* const proc );      
   
   // Accessors
   static const std::string& getHistoryName( );
   
   // Get some pieces out of the frame
   FrameCPP::FrameH::procData_type* getResultProcData( );

   //: Result frame
   frame_h_type		mResultFrame;

   //: Frame writer object.
   //
   // It is necessary to keep this one open while frames are being processed
   //   to minimize memory usage by writing frames as soon as they are
   //   available.
   // std::unique_ptr< FileWriter > mWriter;
   LDASTools::AL::SharedPtr< RDSStream > m_stream;

   virtual void createHistory();   

   const std::string& getChannelName( INT_4U Offset ) const;
   
#if 0
   //--------------------------------------------------------------------
   // Routine to generate the name of the output file. If the file
   //   already exists, then throw an exception.
   //--------------------------------------------------------------------
   std::string	generateOutputName( INT_4U FileStart,
				      REAL_8 FileDeltaT ) const;
#endif /* 0 */

private:
   class private_type;

   //--------------------------------------------------------------------
   // Holder for private information
   //--------------------------------------------------------------------

   std::string	m_ifo_list;

   private_type*	m_p;

   //: No default constructor 
   RDSFrame();
   
   //: No copy constructor
   RDSFrame( const RDSFrame& );
   
   //: No assignment operator
   RDSFrame& operator=( const RDSFrame& );

#if 0
   //--------------------------------------------------------------------
   // Routine to generate the name of the temporary file. If the file
   //   already exists, then throw an exception.
   //--------------------------------------------------------------------
   std::string	generate_tmp_output_name( const INT_4U FrameStart ) const;
#endif /* 0 */

   void init( const frame_file_container_type& FrameFiles, 
	      const channel_container_type& Channels );

};

//=======================================================================
// 
//=======================================================================
namespace FrameAPI
{
   namespace RDS
   {
      //=================================================================
      //
      //=================================================================
      inline bool Options::
      AllowShortFrames( ) const
      {
	 return m_output_allow_short_frames;
      }

      inline Options::compression_level_type Options::
      CompressionLevel( ) const
      {
	 return m_compression_level;
      }

      inline Options::compression_method_type Options::
      CompressionMethod( ) const
      {
	 return m_compression_method;
      }

      inline bool Options::
      CreateChecksumPerFrame( ) const
      {
	 return m_output_checksum_per_frame;
      }

      inline bool Options::
      FillMissingDataValidArray( ) const
      {
	 return m_output_fill_data_valid_array;
      }

      inline Options::frames_per_file_type Options::
      FramesPerFile( ) const
      {
	 return m_frames_per_file;
      }

      inline void Options::
      FramesPerFile( frames_per_file_type FPF )
      {
	 m_frames_per_file = FPF;
      }

      inline bool Options::
      GenerateFrameChecksum( ) const
      {
	 return m_output_checksum_per_frame;
      }

      inline bool Options::
      HistoryRecord( ) const
      {
	 return history_record;
      }

      inline Options::end_type Options::
      OutputTimeEnd( ) const
      {
	 return m_output_stop;
      }

      inline Options::start_type Options::
      OutputTimeStart( ) const
      {
	 return m_output_start;
      }

      inline const std::string& Options::
      OutputType( ) const
      {
	 return m_output_type;
      }

      inline Options::rds_level_type Options::
      RDSLevel( ) const
      {
	 return m_rds_level;
      }

      inline void Options::
      RDSLevel( rds_level_type Level )
      {
	 m_rds_level = Level;
      }

      inline Options::seconds_per_frame_type Options::
      SecondsPerFrame( ) const
      {
	 return m_seconds_per_frame;
      }

      inline void Options::
      SecondsPerFrame( seconds_per_frame_type SPF )
      {
	 m_seconds_per_frame = SPF;
      }

      inline bool Options::
      VerifyChecksum( ) const
      {
	 return VerifyChecksumOfStream( );
      }

      inline bool Options::
      VerifyChecksumOfFrame( ) const
      {
	 return m_input_verify_checksum_per_frame;
      }

      inline bool Options::
      VerifyChecksumPerFrame( ) const
      {
	 return VerifyChecksumOfFrame( );
      }

      inline bool Options::
      VerifyChecksumOfStream( ) const
      {
	 return m_input_verify_checksum;
      }

      inline bool Options::
      VerifyDataValid( ) const
      {
	 return m_input_verify_data_valid;
      }

      inline bool Options::
      VerifyFilenameMetadata( ) const
      {
	 return m_input_verify_filename_metadata;
      }

      inline bool Options::
      VerifyTimeRange( ) const
      {
	 return m_input_verify_time_range;
      }

      //=================================================================
      //=================================================================
      inline void Options::
      AllowShortFrames( bool Value )
      {
      }

      inline void Options::
      CompressionLevel( compression_level_type Value )
      {
	 m_compression_level = Value;
      }

      inline void Options::
      CreateChecksumPerFrame( bool Value )
      {
	 m_output_checksum_per_frame = Value;
      }

      inline void Options::
      FillMissingDataValidArray( bool Value )
      {
	 m_output_fill_data_valid_array = Value;
      }

      inline void Options::
      GenerateFrameChecksum( bool Value )
      {
	 m_output_checksum_per_frame = Value;
      }

      inline void Options::
      HistoryRecord( bool Value )
      {
	 history_record = Value;
      }

      inline void Options::
      OutputTimeEnd( end_type Value )
      {
	 m_output_stop = Value;
      }

      inline void Options::
      OutputTimeStart( start_type Value )
      {
	 m_output_start = Value;
      }

      inline void Options::
      OutputType( const std::string& Value )
      {
	 m_output_type = Value;
      }

      inline void Options::
      VerifyChecksumOfStream( bool Value )
      {
	 m_input_verify_checksum = Value;
      }

      inline void Options::
      VerifyChecksumPerFrame( bool Value )
      {
	 m_input_verify_checksum_per_frame = Value;
      }

      inline void Options::
      VerifyDataValid( bool Value )
      {
	 m_input_verify_data_valid = Value;
      }

      inline void Options::
      VerifyFilenameMetadata( bool Value )
      {
	 m_input_verify_filename_metadata = Value;
      }

      inline void Options::
      VerifyTimeRange( bool Value )
      {
	 m_input_verify_time_range = Value;
      }

      //=================================================================
      //
      //=================================================================
      inline const std::string& FileOptions::
      DirectoryOutputMD5Sum( ) const
      {
	 return m_md5sum_output_directory;
      }

      inline const std::string& FileOptions::
      DirectoryOutputFrames( ) const
      {
	 return m_output_directory;
      }

      inline const std::string& FileOptions::
      MD5SumOutputDirectory( ) const
      {
	 return DirectoryOutputMD5Sum( );
      }

      inline const std::string& FileOptions::
      OutputDirectory( ) const
      {
	 return DirectoryOutputFrames( );
      }

      inline void FileOptions::
      DirectoryOutputFrames( const std::string& Value )
      {
	 m_output_directory = Value;
      }

      inline void FileOptions::
      DirectoryOutputMD5Sum( const std::string& Value )
      {
	 m_md5sum_output_directory = Value;
      }

   } // namespace - RDS
} // namespace - FrameAPI

//=======================================================================
// Inline methods for class RDSFrame
//=======================================================================
inline FrameCPP::FrameH::procData_type* RDSFrame::
getResultProcData( )
{
   return &( mResultFrame->RefProcData( ) );
}
#endif /* RDS_FRAME_HH */

