/* -*- mode: C++; c-basic-offset: 2; -*- */
#if HAVE_CONFIG_H
#include <ldas_tools_config.h>
#endif /* HAVE_CONFIG_H */

#include <string.h>
#include <unistd.h>

#include <iostream>

#include "ldastoolsal/AtExit.hh"
#include "ldastoolsal/CommandLineOptions.hh"

#include "framecpp/Common/FrameSpec.hh"
#include "framecpp/Common/IOStream.hh"
#include "framecpp/Common/FrameBuffer.hh"
#include "framecpp/Common/FrameStream.hh"
#include "framecpp/Common/FrTOC.hh"

#include "framecpp/Version4/FrTOC.hh"
#include "framecpp/Version6/FrTOC.hh"

#include "framecpp/FrameCPP.hh"

#include "toc.hh"

using LDASTools::AL::AtExit;

using namespace FrameCPP::Common;

typedef LDASTools::AL::CommandLineOptions CommandLineOptions;
typedef CommandLineOptions::Option Option;
typedef CommandLineOptions::OptionSet OptionSet;

using std::filebuf;

inline void
depart( int ExitCode )
{
  AtExit::Cleanup( );
  exit( ExitCode );
}

//-----------------------------------------------------------------------
/// \brief Class to handle command line options for this application
//-----------------------------------------------------------------------
class CommandLine
  : protected CommandLineOptions
{
public:
  typedef std::list< std::string >	channel_container_type;

  CommandLine( int ArgC, char** ArgV );

  inline bool
  BadOption( ) const
  {
    bool	retval = false;

    for ( const_iterator
	    cur = begin( ),
	    last = end( );
	  cur != last;
	  ++cur )
    {
      if ( (*cur)[0] == '-' )
      {
	std::cerr << "ABORT: Bad option: " << *cur
		  << std::endl
	  ;
	retval = true;
      }
    }
    return retval;
  }

  inline void
  Usage( int ExitValue ) const
  {
    std::cout << "Usage: "
	      << ProgramName( )
	      << m_options
	      << std::endl;
    depart( ExitValue );
  }

  using CommandLineOptions::Pop;
  using CommandLineOptions::empty;
  using CommandLineOptions::size;

private:
  enum option_types {
    OPT_HELP
  };

  OptionSet	m_options;
};

CommandLine::
CommandLine( int ArgC, char** ArgV )
  : CommandLineOptions( ArgC, ArgV )
{
  m_options.
    Synopsis( "[options] <file> [<file> ...]" );

  m_options.
    Summary( "This command will dump the table of contents for each"
	     " file specified on the command line."
	      );

  //:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
  m_options.
    Add( Option( OPT_HELP,
		 "help",
		 Option::ARG_NONE,
		 "Display this message" ) );
  //---------------------------------------------------------------------
  // Parse the options specified on the command line
  //---------------------------------------------------------------------

  try
  {
    std::string	arg_name;
    std::string arg_value;
    bool	parsing = true;

    while( parsing )
    {
      const int cmd_id( Parse( m_options, arg_name, arg_value ) );

      switch( cmd_id )
      {
      case CommandLineOptions::OPT_END_OF_OPTIONS:
	parsing = false;
	break;
      case OPT_HELP:
	{
	  Usage( 0 );
	}
	break;
      }
    }
  }
  catch( ... )
  {
  }
}

int
main(int ArgC, char* ArgV[])
try
{
  CommandLine	cl( ArgC, ArgV );

  if ( cl.empty( )
       || cl.BadOption( ) )
  {
    cl.Usage( 1 );
  }

  //---------------------------------------------------------------------
  // Process files
  //---------------------------------------------------------------------
  FrameCPP::Initialize( );

  while( cl.empty( ) == false )
  {
    std::string filename( cl.Pop( ) );

    //-------------------------------------------------------------------
    // Creation of the frame structure by reading of frame file
    //-------------------------------------------------------------------
    FrameBuffer< filebuf >*
      ibuf( new FrameBuffer< filebuf >( std::ios::in ) );
    
    ibuf->open( filename.c_str( ), std::ios::in | std::ios::binary );

    IFrameStream	ifs( ibuf, 0 );

    std::cout << "Frame file: " << filename << std::endl
	      << "Frame Specification: " << ifs.Version( ) << std::endl
      ;
    const FrTOC* toc( ifs.GetTOC( ) );

    dump( toc );
  }
  //---------------------------------------------------------------------
  // Exit
  //---------------------------------------------------------------------
  AtExit::Cleanup( );
  depart( 0 );
}
catch( ... )
{
  AtExit::Cleanup( );
  depart( 1 );
}
