/* -*- mode: c++; c-basic-offset: 2; -*- */
#ifndef FILTERS_RESAMPLE_HH
#define FILTERS_RESAMPLE_HH

// $Id: Resample.hh,v 1.20 2006/10/13 19:32:40 emaros Exp $

#include <vector>
#include <valarray>

namespace Filters {

  // Forward declarations
  template<class TCoeffs, class TIn> class LinFilt;

  class ResampleBase {
  public:
    //
    // Since this is purely a base class, there are no constructors
    // available except to derived classes
    //

    //: Destructor is virtual so that RTTI can be used
    virtual ~ResampleBase();
    
    //: Gets the resample numerator
    //!return: upsample factor
    int getP() const;

    //: Gets the resample numerator
    //!return: downsample factor
    int getQ() const;
    
    //: Gets anti-aliasing filter parameters
    //!return: filter length param
    int getN() const;

    //: Gets anti-aliasing filter parameters
    //!return: filter order
    int getNOrder() const;

    //: Gets anti-aliasing filter parameters
    //!return:  Kaiser Window beta parameter
    double getBeta() const;
    
    //: Gets the delay due to the filtering action
    //!return: returns the group delay of the impulse due to filtering
    double getDelay() const;

    //: Pure virtual constructor
    //!return: pointer to new ResampleBase object
    virtual ResampleBase* Clone() const = 0;

    //
    //: Comparison operator
    //
    //!param: const Resample& rhs - instance to be assigned from
    //!return: bool - true if data is the same, false otherwise
    bool operator==(const ResampleBase& rhs);

    //: Reset the internal state to its initial condition
    //
    // This function sets the internal state information back to
    // zero, exactly as it is when the Resample object is first created.
    // This means that when the resampling is next applied, data "before" the
    // start of the series will be treated as if it were zero.
    virtual void reset() = 0;

  protected:
    //
    //: General purpose constructor.  
    //
    //!param: int p - upsampling ratio
    //!param: int q - downsampling ratio
    //!param: int n - filter order: 2*n*max(p, q) is the length of the
    //+ filter
    //!param: double beta - the beta parameter of the Kaiser Window
    //
    //!exc: invalid_argument - if p or q or n < 1 or beta < 0.0
    //
    ResampleBase(const int p, const int q, const int n, const double beta);

    //
    //: Special purpose constructor.  
    //
    //!param: int p - upsampling ratio
    //!param: int q - downsampling ratio
    //
    //!exc: invalid_argument - if p or q < 1
    //
    ResampleBase(const int p, const int q);

  private:
    //: Private default constructor, left undefined
    ResampleBase();
    
    //: upsample factor
    int m_p;

    //: downsample factor
    int m_q;

    //: filter order parameter (not actual filter order)
    int m_n;

    //: anti-aliasing filter order
    int m_nOrder;

    //: Kaiser window beta
    double m_beta;

    //: anti-aliasing filter group delay
    double m_delay;

  }; // end ResampleBase

  //-------------------------------------------------------------------------

  template<class TIn>
  class ResampleTraits {
    public:
    typedef TIn OutType;
  };

  template<>
  class ResampleTraits<short int> {
    public:
    typedef float OutType;
  };

  template<>
  class ResampleTraits<int> {
    public:
    typedef float OutType;
  };

  template<class TIn>
  class Resample : public ResampleBase {

  public:

    typedef typename ResampleTraits<TIn>::OutType TOut;

    //
    //: General purpose constructor.  
    //
    //!param: int p - upsampling ratio
    //!param: int q - downsampling ratio
    //!param: int n = 10 - filter order: 2*n*max(p, q) is the length of the
    //+ filter
    //!param: double beta = 5 - the beta parameter of the Kaiser Window
    //
    //!exc: invalid_argument - if p or q or n < 1 or beta < 0.0
    //
    Resample(const int p, const int q,
             const int n = 10, const double beta = 5.0L);
    
    //
    //: Special purpose constructor.  
    //
    //!param: int p - upsampling ratio
    //!param: int q - downsampling ratio
    //!param: std::valarray<TOut>& b - filter to use
    //
    //!exc: invalid_argument - b.size() == 0
    //
    Resample(const int p, const int q,
             const std::valarray<double>& b);
    
    //
    //: Copy constructor
    //
    //!param: const Resample& rhs - instance to be copied
    Resample(const Resample& rhs);

    //
    //: Virtual destructor
    //
    virtual ~Resample();
    
    //
    //: Assignment, needed because class has pointer members
    //
    //!param: const Resample& rhs - instance to be assigned from
    //!return: reference to *this
    const Resample& operator=(const Resample& rhs);

    //
    //: Comparison operator
    //
    //!param: const Resample& rhs - instance to be assigned from
    //!return: bool - true if data is the same, false otherwise
    bool operator==(const Resample& rhs);

    //
    //: Virtual constructor
    //
    //!return: pointer to new Resample object
    virtual Resample* Clone() const;

    //
    //: Filter coefficents accessor
    //
    //!param: std::valarray<double>& b - FIR filter coefficents array
    void getB(std::valarray<double>& b) const;

    //: Reset the internal state to its initial condition
    //
    // This function sets the internal state information back to
    // zero, exactly as it is when the Resample object is first created.
    // This means that when the resampling is next applied, data "before" the
    // start of the series will be treated as if it were zero.
    virtual void reset();

    //
    //: Resamples "in" by the factor p/q and returns the result in "out".
    //
    // The first call to apply initialises the internal state of Resample
    // so that the input data is treated as if it were preceded by an
    // infinite number of zeroes. After the first apply(), the Resampler is
    // in a non-default state and it will assume that the next call to
    // apply() is on a segment of data that is contiguous with the preceding
    // input. In this way, a long data set can be broken up into smaller
    // segments which can be resampled individually. The recombined data
    // is identical resampling the original long data set.
    //
    // This function is not const to remind users that its use modifies
    // the internal state of Resample.
    //
    //!param: out - pointer to result of resampling the "in" data
    //!param: in - pointer to data to be resampled
    //!param: inlen - length of in
    //
    //!exc: invalid_argument - Input length must be a multiple of q
    void apply(TOut* const out, const TIn* const in, const size_t inlen);

    //
    //: Resamples "in" by the factor p/q and returns the result in "out".
    //
    // The first call to apply initialises the internal state of Resample
    // so that the input data is treated as if it were preceded by an
    // infinite number of zeroes. After the first apply(), the Resampler is
    // in a non-default state and it will assume that the next call to
    // apply() is on a segment of data that is contiguous with the preceding
    // input. In this way, a long data set can be broken up into smaller
    // segments which can be resampled individually. The recombined data
    // is identical resampling the original long data set.
    //
    // This function is not const to remind users that its use modifies
    // the internal state of Resample.
    //
    //!param: std::valarray<T>& out - result of resampling the "in" data
    //!param: const std::valarray<T>& in - data to be resampled
    //
    //!exc: invalid_argument - Input length must be a multiple of q
    void apply(std::valarray<TOut>& out, const std::valarray<TIn>& in);

  private:
    typedef LinFilt< double, TOut > lft_type;
    typedef lft_type*		    lfd_type;
    typedef std::vector< lfd_type > lf_type;


    //: Initialise the filter bank
    void initFilters(const std::valarray<double>& b);

    //: Obtain the filter at position (p, q)
    LinFilt<double, TOut>& getFilter(const int p, const int q) const;

    //: Pointer to partial sum saved for split input purposes
    std::valarray<TOut> m_orphan;

    //: Storage for anti-aliasing filters
    lf_type m_lf;

  }; // end Resample

  inline
  int ResampleBase::getP() const
  {
    return m_p;
  }

  inline
  int ResampleBase::getQ() const
  {
    return m_q;
  }

} // end Filters

#endif // FILTERS_RESAMPLE_HH


