#ifndef FIRLP_HH
#define FIRLP_HH

#include <memory>

#include "HannWindow.hh"

namespace Filters {
  
    class FIRLP {
    
    public:     
    
	// Constructors
    
	//: "Provide everything" constructor
        //
	//!param: const double fc - pass-band edge (Nyquist frequency units)
	//!param: const int order - filter order
	//!param: const Window& win - window for filter design
	//!exc: invalid_argument - if fc <= 0 or fc >= 1
	//!exc: invalid_argument - if t_order <= 0
	FIRLP(const double fc = 0.5, 
	      const int order = 10,  
	      const Window& win = HannWindow());
    
	//: Copy constructor
        //
	//!param: const FIRLP& firlp - FIRLP to duplicate
	FIRLP(const FIRLP& firlp);
    
	//: Destructor
	~FIRLP();
    
	//: Assignment operator
        //
	//!param: const FIRLP& rhs - instance to be assigned
	const FIRLP& operator=(const FIRLP& rhs);
    
	// Accessors
    
	//: Return pass-band edge
        //
	//!return: double - pass band edge as fraction of Nyquist frequency
	double getFc() const;
    
	//: Return filter order
        //
	//!return: int - filter order (length - 1)
	int getOrder() const;
    
	// Mutators
    
	//: Set order
        //
	//!param int - new filter order
	//!exc: std::invalid_argument - if order <= 0
	void setOrder(const int order);
    
	//: Set pass-band edge
        //
	//!param double - pass-band edge as fraction of Nyquist frequncy
	//!exc: std::invalid_argument - if fc <= 0 or fc >= 1
	void setFc(const double fc);
    
	//: Design filter
        //
	//!param: std::valarray<T>& tmp - transfer function coefficients
	template<class T>
	void apply(std::valarray<T>& tmp) const;
    
    private:
	size_t m_length;               // length of window (order + 1)
	double m_fc;                   // ratio of cutoff freq to Nyquist freq
    
	std::auto_ptr<Window> m_win;   // window
    };
  
    inline 
    double FIRLP::getFc() const { return m_fc; }
  
    inline 
    int FIRLP::getOrder() const { return (m_length - 1); }
  
} // namespace Filters

#endif // FIRLP_HH
