#include "general/config.h"

#include <cctype>
#include <iostream>   
#include <sstream>
#include <stdexcept>
#include <typeinfo>

#include "util.hh"

#include "general/types.hh"

using namespace std;

//-----------------------------------------------------------------------------
/// \brief  Case-Insensitve String comparison
///
/// Compares the strings S1 and S2 and returns an integer less than, equal to,
/// greater than zero.
///
/// \param S1 Left hand side of comparison
/// \param S2 Right hand side of comparison
///
int General::
cmp_nocase( const std::string& S1, const std::string& S2 )
{
  std::string::const_iterator	p1 = S1.begin();
  std::string::const_iterator	p2 = S2.begin();

  int N = std::min( S1.size( ), S2.size( ) );

  while( N-- &&
	 ( ( *p1 == *p2 )
	   || ( toupper( *p1 ) == toupper( *p2 ) ) ))
  {
    p1++;
    p2++;
  }
  if (N < 0)
  {
    return ( S2.size() == S1.size() )
      ? 0
      : ( ( S1.size() < S2.size() )
	  ? -1
	  : 1 );
  }
  return 
    (toupper( *p1 ) < toupper( *p2 ))
    ? -1
    : 1; 
}

std::string slower( const std::string &s )
{
   std::string	str;

   for( std::string::const_iterator i = s.begin();
        i != s.end();
        ++i )
   {
      str += tolower( *i );
   }

   return str;
}
   
   
//-----------------------------------------------------------------------------
/// \brief  Convert to Lower-case
///
///!param std::string& s
///
void string2lower( std::string& s )
{
    std::string::iterator iter = s.begin();
    for ( std::string::size_type size = s.size(); size != 0; --size )
    {
        *iter = tolower( *iter );
        ++iter;
    }
}


//-------------------------------------------------------------------------------
/// \brief  Is character a space?
///
/// \param c Character to test.
///
/// \return bool True if argument is whitespace, false otherwise.
///   
bool is_space( const char c )
{
   return isspace( c );
}

   
//-------------------------------------------------------------------------------
/// \brief  Is character a not space?
///
/// \param c Character to test.
///
/// \return bool True if argument is not a whitespace, false otherwise.
///      
bool not_space( const char c )
{
   return !isspace( c );
}   

//-----------------------------------------------------------------------
/// \brief  Rethrow a std::exception adding information to the front
///
/// \param Prefix
///	Text to be prepended to the exception text.
/// \param Exception
///	Origional exception that was thrown and used as type
///     of exception to throw.
//-----------------------------------------------------------------------
void General::
Rethrow( const std::string& Prefix, const std::exception& Exception )
{
  std::ostringstream	err;
  err << Prefix << ": " << Exception.what( );
  try
  {
    throw( Exception );
  }
  /// logic_error: begin
  catch( const std::length_error& e )
  {
    throw std::length_error( err.str( ) );
  }
  catch( const std::domain_error& e )
  {
    throw std::domain_error( err.str( ) );
  }
  catch( const std::out_of_range& e )
  {
    throw std::out_of_range( err.str( ) );
  }
  catch( const std::invalid_argument& e )
  {
    throw std::invalid_argument( err.str( ) );
  }
  catch( const std::logic_error& e )
  {
    throw std::logic_error( err.str( ) );
  }
  /// logic_error: end
  /// runtime_error: begin
  catch( const std::range_error& e )
  {
    throw std::range_error( err.str( ) );
  }
  catch( const std::overflow_error& e )
  {
    throw std::overflow_error( err.str( ) );
  }
  catch( const std::underflow_error& e )
  {
    throw std::underflow_error( err.str( ) );
  }
  catch( const std::runtime_error& e )
  {
    throw std::runtime_error( err.str( ) );
  }
  /// runtime_error: end
  /// others: begin
  catch( const std::bad_alloc& e )
  {
    throw std::bad_alloc( e );
  }
  catch( const std::bad_exception& e )
  {
    throw std::bad_exception( e );
  }
  catch( const std::bad_cast& e )
  {
    throw std::bad_cast( e );
  }
  catch( const std::bad_typeid& e )
  {
    throw std::bad_typeid( e );
  }
  catch( const std::ios_base::failure& e )
  {
    throw std::ios_base::failure( err.str( ) );
  }
  /// others: end
  catch( const std::exception& e )
  {
    throw std::runtime_error( err.str( ) );
  }
}
