#ifndef GENERAL_AUTO_ARRAY_H
#define GENERAL_AUTO_ARRAY_H

#if ! defined(SWIGIMPORTED)
#include <stdexcept>

#include "general/config.h"
#include "general/Memory.hh"


#ifdef HAVE_UNIQUE_PTR
namespace General
{
#if HAVE_CXX_TEMPLATE_ALIASES
template<class T>
    using AutoArray = std::unique_ptr< T[] >;
#else /* HAVE_CXX_TEMPLATE_ALIASES */
  template< typename T >
  class AutoArray
    : public std::unique_ptr< T[] >
  {
  public:
    explicit AutoArray( T* ptr_to_delete = 0 )
      : std::unique_ptr< T[] >( ptr_to_delete )
    {
    }
  };
#endif /* HAVE_CXX_TEMPLATE_ALIASES */
}
#else /* HAVE_UNIQUE_PTR */
namespace General
{
  //---------------------------------------------------------------------
  /// \brief  Gives reference like symantics to AutoArray
  //---------------------------------------------------------------------
  template< typename T>
  struct AutoArrayRef
  {
    /// Array pointer
    T*	m_ptr;
    /// \brief  Constructor
    explicit	AutoArrayRef( T* P )
      : m_ptr( P )
    {
    }
  };

  //---------------------------------------------------------------------
  /// \brief  Auto_ptr for arrays.
  //---------------------------------------------------------------------
  template< typename T >
  class AutoArray
  {
  public:
    typedef T element_type;

    /// Constructor
    explicit AutoArray( T* ptr_to_delete = 0 );

    //-------------------------------------------------------------------
    /// \brief  Copy Constructor
    ///	
    /// Construct a new AutoArray. This object takes ownership from the previous
    /// object
    //-------------------------------------------------------------------
    AutoArray( AutoArray<T>& Data )
      : mArray( Data.release( ) )
    {
    }

    //-------------------------------------------------------------------
    /// \brief  Destructor
    //-------------------------------------------------------------------
    ~AutoArray();

    //-------------------------------------------------------------------
    /// \brief  Array syntax
    //-------------------------------------------------------------------
    T& operator[]( const size_t index ) const;
    //-------------------------------------------------------------------
    /// \brief  Pointer syntax
    //-------------------------------------------------------------------
    T* operator->() const;
    //-------------------------------------------------------------------
    /// \brief  Assignment operator
    //-------------------------------------------------------------------
    AutoArray& operator=( AutoArray< T >& A );
    //-------------------------------------------------------------------
    /// \brief  Obtain array pointer
    //-------------------------------------------------------------------
    T* get() const;
    //-------------------------------------------------------------------
    /// \brief  Release 
    //-------------------------------------------------------------------
    T* release();
    void reset( T* p = 0 );

    //-------------------------------------------------------------------
    /// The following methods give reference like symantics to AutoArray
    //-------------------------------------------------------------------
    AutoArray( AutoArrayRef< T > Ref );
  
    AutoArray& operator=( AutoArrayRef< T > Ref )
    {
      reset( Ref.m_ptr );
      return *this;
    }

    template < typename T1 > operator AutoArrayRef< T1 >( )
    {
      return AutoArrayRef< T1 >( this->release( ) );
    }

    template < typename T1 > operator AutoArray< T1 >( )
    {
      return AutoArray< T1 >( this->release( ) );
    }

  private:

    T* mArray;

  };


  //---------------------------------------------------------------------
  /// \brief  Constructor
  ///
  /// \param ptr_to_delete A pointer to the dynamically
  ///+       allocated array. Default is NULL.
  ///
  template< typename T > AutoArray< T >::
  AutoArray( T* ptr_to_delete )
    : mArray( ptr_to_delete )
  {
  }


  //---------------------------------------------------------------------
  /// \brief  Copy Constructor
  //---------------------------------------------------------------------
  template< typename T > AutoArray< T >::
  AutoArray( AutoArrayRef< T > Ref )
    : mArray( Ref.m_ptr )
  {
  }

  //---------------------------------------------------------------------
  /// \brief  Destructor
  //---------------------------------------------------------------------
  template< typename T > AutoArray< T >::~AutoArray()
  {
    delete[] mArray;
  }

  //---------------------------------------------------------------------
  /// \brief  Overloaded = operator.
  ///
  /// This allows the reassignment of pointer ownership
  ///
  /// \param A The source array.
  ///
  /// \return A reference to this
  //---------------------------------------------------------------------
  template< typename T > AutoArray<T>& AutoArray< T >::
  operator=( AutoArray<T>& A )
  {
    reset( A.release( ) );
    return *this;
  }

  //---------------------------------------------------------------------
  /// \brief  Overloaded [] operator.
  ///
  /// \param index Index into array.
  ///
  /// \return T& - A reference to the requested array element.
  //---------------------------------------------------------------------
  template< typename T > T& AutoArray< T >::operator[]( const size_t index ) const
  {
    if( mArray == 0 )
    {
      throw std::runtime_error( "General::AutoArray::operator[]: array is NULL" );
    }   

    return *( mArray + index );
  }


  //---------------------------------------------------------------------
  /// \brief  Overloaded -> operator.
  ///
  /// \return T& - A reference to the requested array element.
  //---------------------------------------------------------------------
  template< typename T > T* AutoArray< T >::operator->() const
  {
    return mArray;
  }

  //---------------------------------------------------------------------
  /// \brief  Get array buffer.
  ///
  /// \return T* - A pointer to the array.
  //---------------------------------------------------------------------
  template< typename T > T* AutoArray< T >::get() const
  {
    return mArray;
  }


  //---------------------------------------------------------------------
  /// \brief  Release array buffer.
  ///
  /// \return T* - A pointer to the array.
  //---------------------------------------------------------------------
  template< typename T > T* AutoArray< T >::release()
  {
    T* data( mArray );
    mArray = 0;

    return data;
  }


  //---------------------------------------------------------------------
  /// \brief  Reset array buffer.
  ///
  ///!param:: T* - A pointer to new array buffer. Default is NULL.
  //---------------------------------------------------------------------
  template< typename T > void AutoArray< T >::reset( T* ptr_to_delete )
  {
    if( mArray != ptr_to_delete )
    {
      delete[] mArray;
      mArray = ptr_to_delete;
    }
   
    return;
  }

#if 0
  template < typename T, T1 > operator
  AutoArray<T>::AutoArrayRef< T1 >( )
  {
    return AutoArrayRef< T1 >( this->release( ) );
  }

  template < typename T, T1 > operator
  AutoArray<T>::
  AutoArray< T1 >( )
  {
    return AutoArray< T1 >( this->release( ) )
  }
#endif /* 0 */

}
#endif /* HAVE_UNIQUE_PTR */

#endif /* ! defined(SWIGIMPORTED) */

#endif
