#if HAVE_CONFIG_H
#include "framecpp/config.h"
#endif /* HAVE_CONFIG_H */

#include "general/fstream.hh"

#include "framecpp/Common/FrameBuffer.hh"
#include "framecpp/Common/IOStream.hh"
#include "framecpp/Common/MD5SumFilter.hh"

#include "general/unittest.h"
#include "general/autoarray.hh"

General::UnitTest	Test;

#define BUFFER_SIZE 2048

static const char* FILENAME_SOURCE =
  "Z-R_std_test_frame_ver6-600000000-1.gwf";

static const char* FILENAME_SINK =
  "/dev/null"
  ;

using namespace FrameCPP::Common;
using std::ios;

MD5Sum				tmd5sum;

template< typename BT >
inline void
OpenException( const std::string& Header,
	       const std::string& StreamLabel,
	       std::ios::openmode Mode )
{
  try
  {
    using std::filebuf;

    FrameBuffer< filebuf >	b( Mode );
    b.open( StreamLabel, Mode );
    Test.Check( false )
      << Header
      << "No exception thown"
      << std::endl
      ;
  }
  catch( const std::runtime_error& Except )
  {
    Test.Check( true )
      << Header
      << "Exception: " << Except.what( )
      << std::endl
      ;
  }
  catch( const std::exception& Except )
  {
    Test.Check( false )
      << Header
      << "Exception: " << Except.what( )
      << std::endl
      ;
  }
  catch( ... )
  {
    Test.Check( false )
      << Header
      << "Exception: Unknown"
      << std::endl
      ;
  }
}

void
TestExceptions( )
{
  std::ostringstream	hdr;
  const char bad_file[] = "/bad_directory/bad_file";

  //---------------------------------------------------------------------
  // Test opening using std::filebuf
  //---------------------------------------------------------------------
  hdr.str( "std::filebuf::open: " );
  OpenException< std::filebuf >( hdr.str( ), bad_file, std::ios::in );
  OpenException< std::filebuf >( hdr.str( ), bad_file, std::ios::out );
  //---------------------------------------------------------------------
  // Test opening using General::filebuf
  //---------------------------------------------------------------------
  hdr.str( "General::filebuf::open: " );
  OpenException< General::filebuf >( hdr.str( ), bad_file, std::ios::in );
  OpenException< General::filebuf >( hdr.str( ), bad_file, std::ios::out );
}

template< class BT >
void
filetest( const std::string& BufferType, INT_4U BufferSize )
{
  std::ostringstream	lead;
  typename BT::char_type ibuffer[ BUFFER_SIZE ];
  typename BT::char_type obuffer[ BUFFER_SIZE ];

  lead << "filetest:"
       << " BufferType: " << BufferType
       << " BufferSize: " << BufferSize;

  try
  {
    MD5SumFilter			imd5sum;
    MD5SumFilter			omd5sum;

    imd5sum.Reset( );
    omd5sum.Reset( );

    FrameBuffer< BT >*	ibuf( new FrameBuffer< BT >( std::ios::in ) );
    FrameBuffer< BT >*	obuf( new FrameBuffer< BT >( std::ios::out ) );

    IStream	ifs( ibuf );
    OStream	ofs( obuf );

    ibuf->pubsetbuf( ibuffer, sizeof( ibuffer ) / sizeof( *ibuffer ) );
    obuf->pubsetbuf( obuffer, sizeof( obuffer ) / sizeof( *obuffer ) );

    ibuf->FilterAdd( &imd5sum );
    obuf->FilterAdd( &omd5sum );

    ibuf->open( FILENAME_SOURCE, std::ios::in ); // input
    if ( ! ibuf->is_open( ) )
    {
      std::ostringstream	msg;

      msg << "Unable to open the file: " << FILENAME_SOURCE;
      throw std::runtime_error( msg.str( ) );
    }
    obuf->open( FILENAME_SINK,
			std::ios::out | std::ios::binary ); // output

    General::AutoArray< typename BT::char_type >
      buffer( new char[ BufferSize ] );

    ifs.seekg( 0 );
    ifs.clear( );
    while ( ( ifs.eof( ) == false ) )
    {
      MD5Sum m;

      ifs.read( buffer.get( ), BufferSize );
      ofs.write( buffer.get( ), ifs.gcount( ) );
    }

    obuf->close( );
    ibuf->close( );

    imd5sum.Finalize( );
    omd5sum.Finalize( );

    Test.Message( )
      << " imd5sum: " << imd5sum
      << " omd5sum: " << omd5sum
      << std::endl;
    Test.Check( imd5sum == tmd5sum )
      << lead.str( )
      << " Verifying the reader md5sum with control md5sum value"
      << std::endl;
    Test.Check( omd5sum == tmd5sum )
      << lead.str( )
      << " Verifying the writer md5sum with control md5sum value"
      << std::endl;
  }
  catch( const std::exception& e )
  {
  }
  catch( ... )
  {
  }
}

template< class BT >
void
filetest2( const std::string& BufferType, INT_4U BufferSize )
{
#if 0
  std::ostringstream	lead;
  typename BT::char_type ibuffer[ 2048 ];
  typename BT::char_type obuffer[ 2048 ];

  lead << "filetest2:"
       << " BufferType: " << BufferType
       << " BufferSize: " << BufferSize;

  try
  {
    MD5SumFilter			imd5sum;
    MD5SumFilter			omd5sum;

    imd5sum.Reset( );
    omd5sum.Reset( );

    IStream< BT >	ifs( std::ios::in );
    OStream< BT >	ofs( std::ios::out );

    ifs.rdbuf( )->pubsetbuf( ibuffer, sizeof( ibuffer ) / sizeof( *ibuffer ) );
    ofs.rdbuf( )->pubsetbuf( obuffer, sizeof( obuffer ) / sizeof( *obuffer ) );

    ifs.rdbuf( )->open( FILENAME_SOURCE, std::ios::in ); // input
    if ( ! ifs.rdbuf( )->is_open( ) )
    {
      std::ostringstream	msg;

      msg << "Unable to open the file: " << FILENAME_SOURCE;
      throw std::runtime_error( msg.str( ) );
    }
    ofs.rdbuf( )->open( FILENAME_SINK,
			std::ios::out | std::ios::binary ); // output

    General::AutoArray< typename BT::char_type >
      buffer( new char[ BufferSize ] );

    ifs.seekg( 0 );
    ifs.clear( );
    while ( ( ifs.eof( ) == false ) )
    {
      MD5Sum m;

      ifs.rdbuf( )->FilterAdd( &imd5sum );
      ofs.rdbuf( )->FilterAdd( &omd5sum );

      ifs.read( buffer.get( ), BufferSize );
      ofs.write( buffer.get( ), ifs.gcount( ) );

      ifs.rdbuf( )->FilterRemove( &imd5sum );
      ofs.rdbuf( )->FilterRemove( &omd5sum );
    }

    ofs.rdbuf( )->close( );
    ifs.rdbuf( )->close( );

    imd5sum.Finalize( );
    omd5sum.Finalize( );

    Test.Message( )
      << " imd5sum: " << imd5sum
      << " omd5sum: " << omd5sum
      << std::endl;
    Test.Check( imd5sum == tmd5sum )
      << lead.str( )
      << " Verifying the reader md5sum with control md5sum value"
      << std::endl;
    Test.Check( omd5sum == tmd5sum )
      << lead.str( )
      << " Verifying the writer md5sum with control md5sum value"
      << std::endl;
  }
  catch( const std::exception& e )
  {
  }
  catch( ... )
  {
  }
#endif /* 0 */
}

int
main(int argc, char* argv[])
{
  Test.Init( argc, argv );

  //---------------------------------------------------------------------
  // Ensure that exceptions cases are handled properly
  //---------------------------------------------------------------------
  TestExceptions( );
  //---------------------------------------------------------------------
  // Calculate the md5sum of VERSION_6_SAMPLE to use to validate
  //   streaming implementation.
  //---------------------------------------------------------------------
  char buffer[ 2048 ];

  std::ifstream f( FILENAME_SOURCE, std::ios::in );
  if ( ! f.is_open( ) )
  {
    std::ostringstream	msg;

    msg << "Unable to open the file: " << FILENAME_SOURCE;
    Test.Check( false )
      << "Aborting all tests because the base md5sum cannot be calculated: "
      << msg.str( )
      << std::endl;
    Test.Exit( );
  }

  while ( ( f.eof( ) == false ) )
  {
    f.read( &( buffer[ 0 ] ), sizeof( buffer ) );
    tmd5sum.Update( reinterpret_cast< unsigned char* >( &( buffer[ 0 ] ) ),
		    INT_4U( f.gcount( ) ) );
  }
  f.close( );

  tmd5sum.Finalize( );

  //---------------------------------------------------------------------
  // Start to perform tests
  //---------------------------------------------------------------------
  try
  {
#define GENERIC_TEST(f,a,s)			\
    f< a >( #a, s )
#define STD_TEST(f,s) \
    GENERIC_TEST(f,std::filebuf,s)
#define GENERAL_TEST(f,s) \
    GENERIC_TEST(f,General::filebuf,s)

#define TEST(s)				\
    GENERAL_TEST(filetest,s); \
    GENERAL_TEST(filetest2,s); \
    STD_TEST(filetest,s); \
    STD_TEST(filetest2,s)

    //-------------------------------------------------------------------
    // Testing of corner cases
    //-------------------------------------------------------------------
    TEST(BUFFER_SIZE - 3);
    TEST(BUFFER_SIZE - 2);
    TEST(BUFFER_SIZE - 1);
    TEST(BUFFER_SIZE);
    TEST(BUFFER_SIZE + 1);
    TEST(BUFFER_SIZE + 2);
    TEST(BUFFER_SIZE + 3);
    TEST(BUFFER_SIZE / 2 );

    //-------------------------------------------------------------------
    // Testing of other size buffers
    //-------------------------------------------------------------------
    TEST(512);
    TEST(1024 * 1);
    TEST(1024 * 8);
    TEST(1024 * 20);
    TEST(1024 * 2000);
  }
  catch (...)
  {
    Test.Check( false ) << "Caught an exception" << std::endl;
  }
  Test.Exit( );
}
