#include "framecpp/config.h"

#include <iostream>

#include "framecpp/Common/CheckSumFilter.hh"
#include "framecpp/Common/IOStream.hh"
#include "framecpp/Common/Description.hh"
#include "framecpp/Common/Verify.hh"

#include "framecpp/Version3/FrEndOfFile.hh"
#include "framecpp/Version3/FrSE.hh"
#include "framecpp/Version3/FrSH.hh"

#include "framecpp/Version3/FrameSpec.hh"

using namespace FrameCPP::Version_3;

using FrameCPP::Common::CheckSum;
using FrameCPP::Common::CheckSumFilter;
using FrameCPP::Common::Description;
using FrameCPP::Common::FrameSpec;


FrEndOfFile::
FrEndOfFile( )
  : Common::FrEndOfFile( StructDescription( ) ),
    m_nFrames( 0 ),
    m_nBytes( 0 ),
    m_chkFlag( 0 ),
    m_chkSum( 0 )
{
}

FrEndOfFile::
FrEndOfFile( istream_type& Stream )
  : Common::FrEndOfFile( StructDescription( ) )
{
  //---------------------------------------------------------------------
  // Read the data from the stream
  //---------------------------------------------------------------------
  Stream >> m_nFrames
	 >> m_nBytes
    ;
  //---------------------------------------------------------------------
  // Check for checksum calculations
  //---------------------------------------------------------------------
  CheckSumFilter*
    crc = Stream.GetCheckSumFile( );
  if ( crc )
  {
    //-------------------------------------------------------------------
    // Finish the checksum calculation for the file
    //-------------------------------------------------------------------
    m_chkFlag = crc->Type( );
    m_chkSum = crc->Value( );				// Stop calculating
  }
  else
  {
    m_chkFlag = CheckSum::NONE;
    m_chkSum = 0;
  }
  Stream >> m_chkFlag
	 >> m_chkSum;
}

FrEndOfFile::
~FrEndOfFile( )
{
}

FrameSpec::size_type FrEndOfFile::
Bytes( const Common::StreamBase& Stream ) const
{
  return
    sizeof( m_nFrames )
    + sizeof( m_nBytes )
    + sizeof( m_chkFlag )
    + sizeof( m_chkSum )
    ;
}

FrEndOfFile* FrEndOfFile::
Clone( ) const
{
  return new FrEndOfFile( *this );
}

FrEndOfFile* FrEndOfFile::
Create( istream_type& Stream ) const
{
  return new FrEndOfFile( Stream );
}

FrEndOfFile::chkSum_cmn_type FrEndOfFile::
Filter( const Common::IStream& Stream,
	Common::CheckSum& Filt,
	chkType_cmn_type& Type,
	void* Buffer,
	INT_8U Size ) const
{
  chkFlag_type	flag;
  chkSum_type	value;
  INT_8U	offset( Size - sizeof( flag ) - sizeof( value ) );

  if ( Bytes( Stream ) != Size )
  {
    std::ostringstream	msg;
    msg << "FrEndOfFile::Filter: "
	<< " Expected a buffer of size: " << Bytes( Stream )
	<< " but received a buffer of size: " << Size
      ;
    throw std::length_error( msg.str( ) );
  }
  Filt.calc( Buffer,
	     sizeof( nFrames_type)
	     + sizeof( nBytes_type ) );

  std::copy( (const char*)Buffer + offset,
	     (const char*)Buffer + offset + sizeof( flag ),
	     (char*)&flag );
  offset += sizeof( flag );
  std::copy( (const char*)Buffer + offset,
	     (const char*)Buffer + offset + sizeof( value ),
	     (char*)&value );

  if ( Stream.ByteSwapping( ) )
  {
    reverse< sizeof( flag ) >( &flag, 1 );
    reverse< sizeof( value ) >( &value, 1 );
  }
	  
  Type = flag;
  return value;
}

FrEndOfFile::nBytes_cmn_type FrEndOfFile::
NBytes( ) const
{
  return m_nBytes;
}

const char* FrEndOfFile::
ObjectStructName( ) const
{
  return StructName( );
}

void FrEndOfFile::
VerifyObject( Common::Verify& Verifier,
	      Common::IFrameStream& Stream ) const
{
  Verifier.SeenFrEndOfFile( true );
  Verifier.ExamineFrEndOfFileChecksum( Stream, m_chkFlag, m_chkSum );
}
		  
void FrEndOfFile::
Write( ostream_type& Stream ) const
{
  //---------------------------------------------------------------------
  // Get local information
  //---------------------------------------------------------------------

  const nFrames_type	nFrames( Stream.GetNumberOfFrames( ) );
  const nBytes_type	nBytes( Stream.tellp( )
				+ std::streampos( Bytes( Stream ) ) );

  chkFlag_type		chkFlag = CheckSum::NONE;
  chkSum_type		chkSum = 0;

  //---------------------------------------------------------------------
  // Calculate checksum
  //---------------------------------------------------------------------
  CheckSumFilter*
    crc = Stream.GetCheckSumFile( );

  if ( crc )
  {
    chkFlag = crc->Type( );

    crc->Filter( &nFrames, sizeof( nFrames ) );
    crc->Filter( &nBytes, sizeof( nBytes ) );
    
    chkSum = crc->Value( );
    
    Stream.SetCheckSumFile( CheckSum::NONE );
  }

  //---------------------------------------------------------------------
  // Write out to the stream
  //---------------------------------------------------------------------
  Stream << nFrames
	 << nBytes
	 << chkFlag
	 << chkSum
    ;
}

bool FrEndOfFile::
operator==( const FrEndOfFile& RHS ) const
{
  return ( ( m_nFrames == RHS.m_nFrames )
	   && ( m_nBytes == RHS.m_nBytes )
	   && ( m_chkFlag == RHS.m_chkFlag )
	   && ( m_chkSum == RHS.m_chkSum ) );
}

bool FrEndOfFile::
operator==( const Common::FrameSpec::Object& Obj ) const
{
  return ( *this == Obj );
}

const Description* FrEndOfFile::
StructDescription( )
{
  static Description ret;
  if ( ret.size( ) == 0 )
  {
    ret( FrSH( "FrEndOfFile", FrEndOfFile::s_object_id,
	       "End of File Data Structure" ) );
    ret( FrSE( "nFrames", "INT_4U",
	       "Number of frames in this file") );
    ret( FrSE( "nBytes", "INT_4U",
	       "Total number of bytes in this file"
	       " ( 0 if NOT computed )") );
    ret( FrSE( "chkFlag", "INT_4U",
	       "Checksum schemes: 0=none, 1=CRC, >2 presently unsued" ) );
    ret( FrSE ( "chkSum", "INT_4U", "File checksum." ) );
  }

  return &ret;
}

void FrEndOfFile::
assign( assign_stream_type& Stream )
{
  //---------------------------------------------------------------------
  // Read the data from the stream
  //---------------------------------------------------------------------
  Stream >> m_nFrames
	 >> m_nBytes
	 >> m_chkFlag
	 >> m_chkSum
    ;
}

FrEndOfFile::demote_ret_type FrEndOfFile::
demote( INT_2U Target,
	demote_arg_type Obj,
	istream_type* Stream ) const
{
  throw
    Unimplemented( "Object* FrEndOfFile::demote( Object* Obj ) const",
		   DATA_FORMAT_VERSION, __FILE__, __LINE__ );
}

FrEndOfFile::promote_ret_type FrEndOfFile::
promote( INT_2U Target,
	 promote_arg_type Obj,
	 istream_type* Stream ) const
{
  throw
    Unimplemented( "Object* FrEndOfFile::promote( Object* Obj ) const",
		   DATA_FORMAT_VERSION, __FILE__, __LINE__ );
}
