#include "filters/config.h"

#include <stdexcept>

#include "KaiserWindow.hh"

namespace Filters {

    KaiserWindow::KaiserWindow(const double beta)
    {
	set_beta(beta);
    }

    void KaiserWindow::set_beta(const double& beta)
    {
	if (beta < 0)
	{
	    throw std::invalid_argument("invalid Kaiser window beta parameter");
	}

	// If beta changes we need to re-populate ourself
	m_beta = beta;
	m_I0beta = I0(m_beta);
	populate();
    }

    std::string KaiserWindow::name() const
    {
	return std::string("Kaiser Window");
    }

    double KaiserWindow::param() const
    {
	return m_beta;
    }

    double KaiserWindow::element(const size_t i) const
    {
	const size_t n = size();

	if (n == 1)
	{
	    return 1.0;
	}
	else
	{
	    double x = (i - (n-1)/2.0)*2.0/(n-1);
	    return I0(m_beta*std::sqrt(1 - x*x))/m_I0beta;
	}
    }

    //-------------------------------------------------------------------------
    //
    //: Calculate I0(x)
    //
    // Calculates the value of the 0th order, modified Bessel function of the 
    // first kind using a power series expansion. (See "Handbook of Math.
    // Functions," eds. Abramowitz and Stegun, 9.6.12 for details.)
    //
    // NOTE: the accuracy of the expansion is chosen to be 2e-9
    //
    //!param: const double& x
    //
    //!return: double - I0(x)
    //
    double KaiserWindow::I0(const double& x) const
    { 
	double ds(1.0); 
	double d(0.0); 
	double s(1.0); 

	do 
	{ 
	    d  += 2.0; 
	    ds *= x*x/(d*d); 
	    s  += ds; 
	} 
	while (ds > 2e-9); 
  
	return s; 
    } 

    KaiserWindow* KaiserWindow::Clone() const
    {
	return new KaiserWindow(*this);
    }

} // namespace Filters
