/* -*- mode: c++; c-basic-offset: 2; -*- */
#ifndef KFIRLP_HH
#define KFIRLP_HH

#include <stdexcept>

#include "KaiserWindow.hh"

namespace Filters {

  class KFIRLP {
  public:
    
    //: Constructor
    //!param: const double& t_fc - freq. of mid-transition band (Nyquist units)
    //!param: const double& t_alpha - is stop-band depth in dB (positive) 
    //!param: const double& t_df - is transition band width (Nyquist units)
    //!exc: domain_error - Thrown if t_fc <= 0 or t_fc >= 1 or
    //+the abs(2*df-1) is not less than one
    //!exc: invalid_argument - Thrown if t_alpha<=0
    KFIRLP(const double& t_fc, 
           const double& t_alpha, 
           const double& t_df)
      throw(std::domain_error, std::invalid_argument);
    
    //: set mid-point of transition band
    //!param: double t_fc - freq. of mid-transition band (Nyquist units)
    //!exc: domain_error - if t_fc <= 0 or t_fc >= 1
    void setFc(double t_fc)
      throw(std::domain_error);
    
    //: get mid-point of transition band
    //!return: Returns the current mid-point of the transition band    
    double getFc() const;
    
    //: set stop-band depth
    //!param: double t_alpha - sets the stop-band attenuation
    //!exc: invalid_argument - if t_alpha <= 0
    void setAlpha(double t_alpha)
      throw(std::domain_error);
    
    //: get stop-band depth
    //!return: Returns the current stop-band attenuation
    double getAlpha() const;
    
    //: set transition band width
    //!param: double t_df - is transition band width (Nyquist frequency units)
    //!exc: domain_error - if the abs(2*df-1) is not less than one
    void setDf(const double t_df)
      throw(std::domain_error);
    
    //: get transition band width
    //!return: Returns the current width of the transition band
    double getDf() const;
    
    //: get transfer function
    //!param: valarray<double>& b -  filter coefficients in the container
    template<class T>
    void apply( std::valarray<T>& b);
    
    //: filter order
    //!return: int - order (length-1) of the filter
    int getOrder() const;
    
  private:
    
    //: Kaiser Window beta parameter
    //!return: double - the beta parameter corresponding to the current
    //+  order, alpha
    double getBeta() const;
    
    // data members
    
    double fc;
    double df;             // width of the transition zone of the filter
    double alpha;          // side lobe height
    KaiserWindow kwin;     // Kaiser Window
  };

  inline double KFIRLP::getFc() const { return fc; }
  
  inline double KFIRLP::getAlpha() const { return alpha; }
  
  inline double KFIRLP::getDf() const { return df; }
  
} // namespace Filters

#endif // KFIRLP_HH
