#include <algorithm>
#include <sstream>

#include "general/AtExit.hh"
#include "general/ErrorLog.hh"

#include "genericAPI/StatFork.hh"
#include "genericAPI/StatDirect.hh"

#include "genericAPI/StatPool.hh"

using General::StdErrLog;

namespace {
  typedef	GenericAPI::StatDirect stat_type;
  // typedef	GenericAPI::StatFork stat_type;
  typedef std::list< GenericAPI::StatPool::info_type > pool_type;
  
  static stat_type	lstat_base;

  inline static pool_type&
  available( )
  {
    static pool_type		m_available;

    return m_available;
  }

  inline static pool_type&
  in_use( )
  {
    static pool_type		m_in_use;

    return m_in_use;
    
  }

}

static bool initialize( );
namespace GenericAPI
{
  // Instantiate generic to Singleton class methods:   
  // see "general" library   
  SINGLETON_TS_INST( StatPool );

  MutexLock::lock_type StatPool::m_pool_lock = PTHREAD_MUTEX_INITIALIZER;
  StatPool::user_init_cb StatPool::m_user_init_cb = NULL;
  bool StatPool::m_initialized = initialize( );

  StatPool::
  StatPool( )
  {
    if ( StdErrLog.IsOpen( ) )
    {
      std::ostringstream	msg;
      
      msg << "StatPool::StatPool";
      StdErrLog( General::ErrorLog::DEBUG,
		 __FILE__, __LINE__,
		 msg.str( ) );
    }
  }

  void StatPool::
  Cleanup( )
  {
    static bool has_been_cleaned = false;
    if ( has_been_cleaned == false )
    {
      MutexLock	lock( m_pool_lock );

      if ( has_been_cleaned == false )
      {
	if ( StdErrLog.IsOpen( ) )
	{
	  std::ostringstream	msg;
      
	  msg << "StatPool::~StatPool";
	  StdErrLog( General::ErrorLog::DEBUG,
		     __FILE__, __LINE__,
		     msg.str( ) );
	}
	for (  pool_type::const_iterator
		 cur = available( ).begin( ),
		 last = available( ).end( );
	       cur != last;
	       ++cur )
	{
	  delete *cur;
	}
	available( ).erase( available( ).begin( ), available( ).end( ) );
#if 0
	for (  pool_type::const_iterator
		 cur = in_use( ).begin( ),
		 last = in_use( ).end( );
	       cur != last;
	       ++cur )
	{
	  delete *cur;
	}
#endif /* 0 */
	in_use( ).erase( in_use( ).begin( ), in_use( ).end( ) );
	has_been_cleaned = true;
      } // if ( has_been_cleaned == false ) (after Mutex lock)
    } // if ( has_been_cleaned == false )
  }

  StatPool::info_type StatPool::
  StatType( )
  {
    return &lstat_base;
  }

  void StatPool::
  Destroy( StatPool::info_type Key )
  {
    MutexLock	lock( m_pool_lock );

    if ( StdErrLog.IsOpen( ) )
    {
      std::ostringstream	msg;
      
      msg << "StatPool::destroy: Key: " << (void*)Key;
      if ( Key )
      {
	msg << Key->Debug( );
      }
      StdErrLog( General::ErrorLog::DEBUG,
		 __FILE__, __LINE__,
		 msg.str( ) );
    }
    //-------------------------------------------------------------------
    // Take out of m_in_use list
    //-------------------------------------------------------------------
    pool_type::size_type	size = in_use( ).size( );

    /* erase the element */
    in_use( ).remove( Key );
    if ( size != in_use( ).size( ) )
    {
      //-----------------------------------------------------------------
      // Destroy the process
      //-----------------------------------------------------------------
      if ( StdErrLog.IsOpen( ) )
      {
	std::ostringstream	msg;
      
	msg << "StatPool::release: deleting: Key: " << (void*)Key;
	if ( Key )
	{
	  msg << Key->Debug( );
	}
	StdErrLog( General::ErrorLog::DEBUG,
		   __FILE__, __LINE__,
		   msg.str( ) );
      }
      delete Key;
    }
  }

  void StatPool::
  Release( StatPool::info_type Key )
  {
    MutexLock	lock( m_pool_lock );

    if ( StdErrLog.IsOpen( ) )
    {
      std::ostringstream	msg;
      
      msg << "StatPool::release: " << Key->Debug( );
      StdErrLog( General::ErrorLog::DEBUG,
		 __FILE__, __LINE__,
		 msg.str( ) );
    }
    //-------------------------------------------------------------------
    // Take out of in_use( ) list
    //-------------------------------------------------------------------
    pool_type::size_type size = in_use( ).size( );

    /* erase the element */
    in_use( ).remove( Key );
    if ( size != in_use( ).size( ) )
    {
      //-----------------------------------------------------------------
      // Put into available( ) queue
      //-----------------------------------------------------------------
      if ( StdErrLog.IsOpen( ) )
      {
	std::ostringstream	msg;
      
	msg << "StatPool::release: returned to a available pool: "
	    << Key->Debug( );
	StdErrLog( General::ErrorLog::DEBUG,
		   __FILE__, __LINE__,
		   msg.str( ) );
      }
      if ( General::AtExit::IsExiting( ) )
      {
	delete Key;
      }
      else
      {
	available( ).push_back( Key );
      }
    }
  }

  StatPool::info_type StatPool::
  Request( )
  {
    MutexLock	lock( m_pool_lock );
    info_type retval = NULL;

    if ( available( ).size( ) == 0 )
    {
      //-----------------------------------------------------------------
      // Create new thread
      //-----------------------------------------------------------------
      retval = lstat_base.vnew( );
      if ( m_user_init_cb )
      {
	(*m_user_init_cb)( retval );
      }
      retval->Init( );
      if ( StdErrLog.IsOpen( ) )
      {
	std::ostringstream	msg;
	
	msg << "StatPool::request: Created new " << retval->Debug( );
	StdErrLog( General::ErrorLog::DEBUG,
		   __FILE__, __LINE__,
		   msg.str( ) );
      }
    }
    else
    {
      //-----------------------------------------------------------------
      // Take one from the available( ) list
      //-----------------------------------------------------------------
      retval = available( ).front( );
      available( ).pop_front( );
      if ( StdErrLog.IsOpen( ) )
      {
	std::ostringstream	msg;
	
	msg << "StatPool::request: Reusing " << retval->Debug( );
	StdErrLog( General::ErrorLog::DEBUG,
		   __FILE__, __LINE__,
		   msg.str( ) );
      }
    }
    in_use( ).push_back( retval );
    return retval;
  }
}

static bool
initialize( )
{
  General::AtExit::Append( GenericAPI::StatPool::Cleanup,
			   "GenericAPI::StatPool::Cleanup",
			   120 );
  return true;
}
