#ifndef GENERIC_API__MOUNT_POINT_STATUS_HH
#define GENERIC_API__MOUNT_POINT_STATUS_HH

#include <algorithm>
#include <list>
#include <map>
#include <string>
#include <vector>

#include "general/Directory.hh"
#include "general/mutexlock.hh"
#include "general/ReadWriteLock.hh"
#include "general/Thread.hh"

namespace GenericAPI
{

  //=====================================================================
  /// \brief Monitors a list of directories.
  ///
  /// The main focus of this class is to provide methods which maintain
  /// information about a group of directories.
  //=====================================================================
  class MountPointStatus
  {
  public:
    enum state_type {
      STATE_OFFLINE,
      STATE_ONLINE
    };

    //-------------------------------------------------------------------
    /// \brief Container for a set of mount points.
    ///
    /// This type is used by several routines to pass a set of
    /// directories on which to operate.
    //-------------------------------------------------------------------
    typedef std::list< std::string >	mount_point_list_type;

    //-------------------------------------------------------------------
    /// \brief Initialize the class
    ///
    /// \return
    ///     True if initialization completes successfully,
    ///     false otherwise.
    //-------------------------------------------------------------------
    static bool Initialize( );

    //-------------------------------------------------------------------
    /// \brief Adds directories to set of managed directories
    ///
    /// \param[in] MountPoint
    ///	    A group of directories to be added.
    ///
    /// \return
    ///     Nothing
    //-------------------------------------------------------------------
    static void Add( const mount_point_list_type& MountPoint );

    //-------------------------------------------------------------------
    /// \brief Checks if directory is being managed
    ///
    /// \param[in] Dir
    ///     The name of a file or directory on the file system
    ///
    /// \return
    ///     If their is an exact match, then true is returned,
    ///     false otherwise.
    //-------------------------------------------------------------------
    static bool IsDirManaged( const std::string& Dir );

    //-------------------------------------------------------------------
    /// \brief Removes directories from the set of managed directories.
    ///
    /// \param[in] MountPoint
    ///	    A group of directories to be removed.
    ///
    /// \return
    ///     Nothing
    //-------------------------------------------------------------------
    static void Remove( const mount_point_list_type& MountPoint );

    //-------------------------------------------------------------------
    /// \brief Resets the set of directories to be managed.
    ///
    /// \param[in] MountPoint
    ///	    A group of directories to be managed.
    ///
    /// \return
    ///     Nothing
    //-------------------------------------------------------------------
    static void Set( const mount_point_list_type& MountPoint );

    //-------------------------------------------------------------------
    /// \brief Resets the set of directories to be managed.
    ///
    /// \param[in] MountPoint
    ///	    A group of directories to be managed.
    ///
    /// \return
    ///     Nothing
    //-------------------------------------------------------------------
    static void Set( const std::vector< std::string >& MountPoint );

    //-------------------------------------------------------------------
    /// \brief Returns the on-line status of the file system
    ///
    /// \param[in] Path
    ///	    The Path to a file or directory on the file system for
    ///     which the on-line status is being queried.
    ///
    /// \return
    ///     The on-line status of the file system.
    //-------------------------------------------------------------------
    static state_type Status( const std::string& Path );

    static void Offline( const std::string& Path );
  }; /* class MountPointStatus */

  //=====================================================================
  // MountPointStatus
  //=====================================================================

  inline void MountPointStatus::
  Set( const std::vector< std::string >& MountPoint )
  {
    mount_point_list_type    mp;

    std::copy( MountPoint.begin( ), MountPoint.end( ),
	       std::back_inserter( mp ) );
    Set( mp );
  }
} /* namespace - GenericAPI */

#endif /* GENERIC_API__MOUNT_POINT_STATUS_HH */
