#include "genericAPI/config.h"

#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/types.h>

#include <fcntl.h>
#include <netdb.h>
#include <pthread.h>
#include <unistd.h>

#include <cctype>
#include <cstdio>
#include <ctime>

#include <algorithm>

#include "general/autoarray.hh"
#include "general/unordered_map.hh"
#include "general/gpstime.hh"
#include "general/System.hh"

#include "genericAPI/LogHTML.hh"
#include "genericAPI/Logging.hh"
#include "genericAPI/Stat.hh"
#include "genericAPI/Symbols.hh"
#if 0
#include "genericAPI/TclSymbolTable.hh"
#endif /* 0 */

#ifndef AI_NUMERICSERV
#define AI_NUMERICSERV AI_NUMERICHOST
#endif
#ifndef AI_ADDRCONFIG
#define AI_ADDRCONFIG 0
#endif

using GenericAPI::LogHTML;
using General::GPSTime;

typedef General::unordered_map< std::string, const char* > site_container_type;
typedef struct addrinfo* addr_info_type;

static addr_info_type manager_addr_info = (addr_info_type)NULL;
static const pthread_t main_thread = pthread_self( );

static site_container_type& site_info( );
static std::string site_info_lookup( const std::string& LDASSystemName );
static void getaddrinfo_for_manager( );

//-----------------------------------------------------------------------
/// \brief Format gif image appropriate for log entry
//-----------------------------------------------------------------------
static const std::string gif_html( LogHTML::message_type MT,
				   const General::GPSTime& TimeStamp );
//-----------------------------------------------------------------------
/// \brief Format gif information for legend section of web pages.
//-----------------------------------------------------------------------
static const std::string gif_legend( LogHTML::message_type MT );

//-----------------------------------------------------------------------
/// \brief Format time value for GMT time
//-----------------------------------------------------------------------
static const std::string gm_time( const General::GPSTime& Time );

//-----------------------------------------------------------------------
/// \brief Fromat time value for local time
//-----------------------------------------------------------------------
static const std::string local_time( const GPSTime& Time );

static void manager_command( const std::string& Command );
static int manager_connect( );

namespace GenericAPI
{

  LogHTML::
  LogHTML( const std::string& BaseName )
    : General::Log( BaseName )
  {
  }

  void LogHTML::
  Message( message_type MessageType, level_type Level,
	   const std::string& Caller, const std::string& JobInfo,
	   const std::string& Message )
  {
    getaddrinfo_for_manager( );
    //-------------------------------------------------------------------
    // Verify that the message should be logged
    //-------------------------------------------------------------------
    //    if ( ShouldLog( MessageType, Level ) )
    {
      //-----------------------------------------------------------------
      // OK, this message should be logged.
      // Format the message for logging and then get it into the queue
      //-----------------------------------------------------------------
      std::ostringstream	msg;
      General::GPSTime		t;

      t.Now( );

      msg
	//---------------------------------------------------------------
	// GIF image
	//---------------------------------------------------------------
	<< gif_html( MessageType, t )
	//---------------------------------------------------------------
	// Time stamp
	//---------------------------------------------------------------
	<< " <b>" << t.GetSeconds( ) << "</b>"
	;
      if ( *(JobInfo.begin( )) == '<' )
      {
	//---------------------------------------------------------------
	// JobInfo has already been html formatted
	//---------------------------------------------------------------
	msg << " " << JobInfo;
      }
      else
      {
	//---------------------------------------------------------------
	// JobInfo has not been formatted for HTML.
	// Do only the very basic
	//---------------------------------------------------------------
	msg << " <b><i><font color=\"green\">" << JobInfo << "</font></i></b>";
      }
      //-----------------------------------------------------------------
      msg
	//---------------------------------------------------------------
	// Caller
	//---------------------------------------------------------------
	<< " <b><i><font color=\"red\">" << Caller << "</font></i></b>"
	//---------------------------------------------------------------
	// Body of message
	//---------------------------------------------------------------
	<< " <tt>"
	;
      if ( Message.length( ) == 0 )
      {
	msg << "no message passed for logging";
      }
      else
      {
	//---------------------------------------------------------------
	// Need to have temporary copy for mutations along the way
	//---------------------------------------------------------------
	std::string	body( Message );
	// std::string	tmp_body;
	//---------------------------------------------------------------
	// As the final step, cleanup of whitespaces and non printable
	// characters within body.
	//---------------------------------------------------------------
	register bool ws = false;

	for ( std::string::const_iterator
		cur = body.begin( ),
		last = body.end( );
	      cur != last;
	      ++cur )
	{
	  if ( isspace( *cur ) && ( ws == false ) )
	  {
	    msg << " ";
	    ws = true;
	    continue;
	  }
	  else if ( ! isprint( *cur ) )
	  {
	    msg << ".";
	  }
	  else
	  {
	    msg << *cur;
	  }
	  ws = false;
	}
	msg << "</tt>"
	    << "<br>"
	    << std::endl
	  ;
      }
      //-----------------------------------------------------------------
      // Send the message to the queue
      //-----------------------------------------------------------------
      Log::Message( MessageType, Level, msg.str( ) );
    }
  }

  void LogHTML::
  TclInitialization( Tcl_Interp* Interp )
  {
    getaddrinfo_for_manager( );
  }

  void LogHTML::
  html_header( )
  {
    //-------------------------------------------------------------------
    // Gather information
    //-------------------------------------------------------------------
    const char* host = ( getenv( "HOST" ) )
      ? getenv( "HOST" )
      : "localhost";

    std::string	ldas_api;
    Symbols::LDAS_SYSTEM::value_type	ldas_system_name;

    APINameGet( ldas_api );
    Symbols::LDAS_SYSTEM::Get( ldas_system_name );

    //-------------------------------------------------------------------
    // 
    //-------------------------------------------------------------------
    std::string site( site_info_lookup( ldas_system_name ) );
    std::string special_log_header( "" );

    std::string	title;

    title = "The ";
    title += ldas_api;
    title += " API on ";
    title += host;
    title += " at ";
    title += site;


    std::ostringstream	preamble;
    preamble
      << "<HTML> "
      << "<HEAD> "
      << "<TITLE>" << title << "</TITLE> "
      << "<BODY BGCOLOR=\"#DDDDDD\" TEXT=\"#000000\"> "
      << "<h3>" << title <<"</h3> "
      << "<b><i>Legend:</i></b><br> "
      << gif_legend( LogHTML::MT_GREEN ) << "<br> "
      << gif_legend( LogHTML::MT_YELLOW ) << "<br> "
      << gif_legend( LogHTML::MT_ORANGE ) << "<br> "
      << gif_legend( LogHTML::MT_RED ) << "<br> "
      << gif_legend( LogHTML::MT_BLUE ) << "<br> "
      << gif_legend( LogHTML::MT_PURPLE ) << "<br> "
      << gif_legend( LogHTML::MT_MAIL ) << "<br> "
      << gif_legend( LogHTML::MT_PHONE )
      << "<p>" << special_log_header
      << "<p><b><i>Link:</i></b> "
      << "<a href=\"APIstatus.html\">API Status Page for " << site << "</a> "
      << "<p>" << std::endl
      ;
    writeDirect( preamble.str( ) );
  }


  void LogHTML::
  html_trailer( )
  {
    std::ostringstream	postamble;

    postamble << "</BODY> </HTML>" << std::endl;
    writeDirect( postamble.str( ) );
  }

  //---------------------------------------------------------------------
  /// Rotate the logs according to the LDAS rules
  //---------------------------------------------------------------------
  void LogHTML::
  rotate( )
  try {
    using namespace GenericAPI::Symbols;
    //-------------------------------------------------------------------
    // Check to see if the stream is currently open
    //-------------------------------------------------------------------
    if ( m_stream.is_open( ) )
    {
      //-----------------------------------------------------------------
      // Write the trailer and close the log
      //-----------------------------------------------------------------
      html_trailer( );
      m_stream.close( );
    }
    //-------------------------------------------------------------------
    // Obtain the output directory
    //-------------------------------------------------------------------
    std::string				ldas_api;
    LDAS_ARCHIVE_DIR::value_type	ldas_arc;
    LDAS_LOG_DIR::value_type		ldas_log_dir;

    APINameGet( ldas_api );
    LDAS_ARCHIVE_DIR::Get( ldas_arc );
    LDAS_LOG_DIR::Get( ldas_log_dir );
    //-------------------------------------------------------------------
    // Generate the filename.
    //-------------------------------------------------------------------
    std::ostringstream	filename;

    filename << ldas_log_dir
	     << "/LDAS" << ldas_api
	     << ".log.html"
      ;
    //-------------------------------------------------------------------
    /// \todo
    /// Rotate the logs
    //-------------------------------------------------------------------
    int	fd = ::open( filename.str( ).c_str( ), O_RDONLY );
    if ( fd < 0 )
    {
      //-----------------------------------------------------------------
      // There is an issue with getting access to the log file.
      //-----------------------------------------------------------------
      std::ostringstream	msg;

      msg << "::open( " << filename.str( ) << ") failed: "
	  << General::System::ErrnoMessage( )
	;

      //-----------------------------------------------------------------
      // Report the error using the logging mechanism
      //-----------------------------------------------------------------
#if 0
      GenericAPI::queueLogEntry( msg.str( ),
				 MT_ERROR,
				 0,
				 "GenericAPI::LogHTML::rotate",
				 "IDLE" );
#else
      std::cerr << "INFO: " << msg.str( ) << std::endl;
#endif /* 0 */
    }
    else
    try {
      ::close( fd );
      struct stat stat_buf;

      if ( stat( filename.str( ).c_str( ), &stat_buf ) != 0 )
      {
	//---------------------------------------------------------------
	// There is an issue with getting access to the log file.
	//---------------------------------------------------------------
	std::ostringstream	msg;

	msg << "Stat( " << filename.str( ) << ", stat_buf ) failed: "
	    << General::System::ErrnoMessage( )
	  ;
	throw std::runtime_error( msg.str( ) );

      }
      //-----------------------------------------------------------------
      // Generate the archive name
      //-----------------------------------------------------------------
      std::ostringstream	archive;
      {
	
	General::GPSTime		log_time( stat_buf.st_mtime, 0,
						  General::GPSTime::UTC );

	archive << ldas_arc
		<< "/" << ldas_api << "API"
		<< "/LDAS" << ldas_api << "." << log_time.GetSeconds( )
	  ;
      }
      //-----------------------------------------------------------------
      // Archive the file
      //-----------------------------------------------------------------
      if ( ::rename( filename.str( ).c_str( ),
		     archive.str( ).c_str( ) ) != 0 )
      {
	//---------------------------------------------------------------
	// There is an issue with renaming the file
	//---------------------------------------------------------------
	std::ostringstream	msg;

	msg << "::rename( " << filename.str( )
	    << ", " << archive.str( ) << " ) failed: "
	    << General::System::ErrnoMessage( )
	  ;
	throw std::runtime_error( msg.str( ) );
      }
      //-----------------------------------------------------------------
      // Notifiy the manager to update index
      //-----------------------------------------------------------------
      try
      {
	std::string	command( "::archiveIndex " );

	command += archive.str( );
	manager_command( command );
      }
      catch( const std::exception& Error )
      {
	//---------------------------------------------------------------
	// Record the error in the current log
	//---------------------------------------------------------------
#if 0
	GenericAPI::queueLogEntry( Error.what( ),
				   MT_ERROR,
				   0,
				   "GenericAPI::LogHTML::rotate",
				   "IDLE" );
#else
	std::cerr << "INFO:" << Error.what( ) << std::endl;
#endif /* 0 */
      }
      //-----------------------------------------------------------------
      // Setup for fixing symbolic links
      //-----------------------------------------------------------------
      std::ostringstream base;
      base << ldas_log_dir << "/LDAS" << ldas_api;
      std::string file1( base.str( ) );
      std::string file2( base.str( ) );
      file1 += ".2.html";
      file2 += ".1.html";
      //-----------------------------------------------------------------
      // Roll the logs
      // 1) Remove oldest log
      // 2) Move n-2 logs up one position.
      // 3) Create new symbolic link for the file just archived.
      //-----------------------------------------------------------------
      ::unlink( file1.c_str( ) );
      ::rename( file2.c_str( ), file1.c_str( ) ); /* b.1.html => b.2.html */
      if ( ::symlink( archive.str( ).c_str( ), file2.c_str( ) ) != 0 )
      {
	//---------------------------------------------------------------
	/// \todo
	/// Report error if creation of symlink failed
	//---------------------------------------------------------------
      }

    }
    catch( const std::exception& Error )
    {
      //-----------------------------------------------------------------
      // Report the error using the logging mechanism
      //-----------------------------------------------------------------
#if 0
      GenericAPI::queueLogEntry( Error.what( ),
				 MT_ERROR,
				 0,
				 "GenericAPI::LogHTML::rotate",
				 "IDLE" );
#else
      std::cerr << "INFO: " << Error.what( ) << std::endl;
#endif /* 0 */
    }
    //-------------------------------------------------------------------
    // Open the new log
    //-------------------------------------------------------------------
    m_stream.open( filename.str( ).c_str( ), std::ios::out );
    //-------------------------------------------------------------------
    // Write the preamble
    //-------------------------------------------------------------------
    html_header( );
  }
  catch( ... )
  {
    // Ignore any problems that make it out this far.
  }
}

//=======================================================================
// Local members
//=======================================================================
struct gif_info
{
  enum alt_type {
    ALT_TIME,
    ALT_EMAIL
  };

  const char*		s_name;
  const char*		s_alt_desc;
  const char*		s_description;
  const int 		s_width;
  const int		s_height;
  const alt_type	s_alt_info;
};

//-----------------------------------------------------------------------
// 
//-----------------------------------------------------------------------
const gif_info&
gif_lookup( LogHTML::message_type MT )
{
  //---------------------------------------------------------------------
  // 
  //---------------------------------------------------------------------
  typedef General::unordered_map< int, const gif_info* > gif_container_type;
  static gif_container_type	gifs;

  static bool initialized = false;
  if ( initialized == false )
  {
    static MutexLock::lock_type	gif_baton = MutexLock::Initialize( );
    
    MutexLock	l( gif_baton );
    if ( initialized == false )
    {
      static const gif_info gif_info_table[] = {
	//---------------------------------------------------------------
	// 0
	//---------------------------------------------------------------
	{ "ball_green.gif",
	  "green ball",
	  "Normal status or debugging message",
	  14, 12,
	  gif_info::ALT_TIME
	},
	//---------------------------------------------------------------
	// 1
	//---------------------------------------------------------------
	{ "ball_yellow.gif",
	  "yellow ball",
	  "Notable condition which may be a non-fatal error",
	  14, 12,
	  gif_info::ALT_TIME
	},
	//---------------------------------------------------------------
	// 2
	//---------------------------------------------------------------
	{ "ball_red.gif",
	  "red ball",
	  "Error condition which fatal to job",
	  14, 12,
	  gif_info::ALT_TIME
	},
	//---------------------------------------------------------------
	// 3
	//---------------------------------------------------------------
	{ "mail.gif",
	  "email",
	  "Condition requires email notification of the responsible administrator of this API",
	  27, 16,
	  gif_info::ALT_EMAIL
	},
	//---------------------------------------------------------------
	// 4
	//---------------------------------------------------------------
	{ "mail.gif",
	  "email",
	  "Condition requires email notification of the responsible administrator of this API",
	  27, 16,
	  gif_info::ALT_EMAIL
	},
	//---------------------------------------------------------------
	// 5
	//---------------------------------------------------------------
	{ "telephone.gif",
	  "telephone",
	  "Condition requires phone notification of the responsible administrator of this API",
	  27, 27,
	  gif_info::ALT_TIME
	},
	//---------------------------------------------------------------
	// 6
	//---------------------------------------------------------------
	{ "ball_blue.gif",
	  "blue ball",
	  "Notable condition which is not an error",
	  14, 12,
	  gif_info::ALT_TIME
	},
	//---------------------------------------------------------------
	// 7
	//---------------------------------------------------------------
	{ "ball_purple.gif",
	  "purple ball",
	  "Currently undefined",
	  14, 12,
	  gif_info::ALT_TIME
	},
	//---------------------------------------------------------------
	// 8
	//---------------------------------------------------------------
	{ "ball_orange.gif",
	  "orange ball",
	  "Error condition not fatal to job",
	  14, 12,
	  gif_info::ALT_TIME
	}
      };

      gifs[ LogHTML::MT_OK ] = &gif_info_table[ 0 ];
      gifs[ LogHTML::MT_WARN ] = &gif_info_table[ 1 ];
      gifs[ LogHTML::MT_ERROR ] = &gif_info_table[ 2 ];
      gifs[ LogHTML::MT_EMAIL ] = &gif_info_table[ 3 ];
      gifs[ LogHTML::MT_CERTMAIL ] = &gif_info_table[ 4 ];
      gifs[ LogHTML::MT_PHONE ] = &gif_info_table[ 5 ];
      gifs[ LogHTML::MT_DEBUG ] = &gif_info_table[ 6 ];
      gifs[ LogHTML::MT_NOTE ] = &gif_info_table[ 7 ];
      gifs[ LogHTML::MT_ORANGE ] = &gif_info_table[ 8 ];

      initialized = true;
    }
  }
  gif_container_type::iterator i = gifs.find( MT );
  if ( i == gifs.end( ) )
  {
    throw std::range_error( "No gif info" );
  }
  return *(i->second);
}

//-----------------------------------------------------------------------
//-----------------------------------------------------------------------
static const std::string
gif_legend( LogHTML::message_type MT )
{
  std::ostringstream	desc;

  try
  {
    const gif_info& gi = gif_lookup( MT );

    desc
      << "<img src=\"" << gi.s_name << "\""
      << " width=\"" << gi.s_width << "\" height=\"" << gi.s_height << "\""
      << " alt=\"" << gi.s_alt_desc << "\" title=\"" << gi.s_alt_desc << "\""
      << "> "
      << gi.s_description
      ;
  }
  catch( ... )
  {
  }

  return desc.str( );
}

static const std::string
gif_html( LogHTML::message_type MT,
	  const General::GPSTime& TimeStamp )
{
  std::ostringstream	gif;
  try
  {
    const gif_info& gi = gif_lookup( MT );

    std::ostringstream		alt;

    switch( gi.s_alt_info )
    {
    case gif_info::ALT_EMAIL:
      {
	std::string api;
	std::string notify;

	GenericAPI::APINameGet( api );
	GenericAPI::EMailNotifyGet( api, notify );
	alt << notify;
      }
      break;
    case gif_info::ALT_TIME:
      // Fall through to default case
    default:
      alt << local_time( TimeStamp ) << " &#013;" << gm_time( TimeStamp );
      break;
    };

    gif << "<img src=\"" << gi.s_name << "\""
	<< " width=\"" << gi.s_width << "\" height=\"" << gi.s_height << "\""
	<< " alt=\"" << alt.str( ) << "\" title=\"" << alt.str( ) << "\""
	<< ">";
  }
  catch( ... )
  {
  }
 
  return gif.str( );
}

static const std::string
gm_time( const GPSTime& Time )
{
  std::string	retval;
  time_t 	t = Time.GetSeconds( GPSTime::UTC );
  struct tm	ti;

  if ( gmtime_r( &t, &ti ) == &ti )
  {
    char	buffer[ 256 ];

    if ( strftime( buffer, sizeof( buffer ), "%x-%X GMT", &ti )  )
    {
      retval = buffer;
    }
  }
  return retval;
}

static const std::string
local_time( const GPSTime& Time )
{
  std::string	retval;
  time_t 	t = Time.GetSeconds( GPSTime::UTC );
  struct tm	ti;

  if ( localtime_r( &t, &ti ) == &ti )
  {
    static const int buffer_size = 1024;

    General::AutoArray< char >	buffer( new char[ buffer_size ] );

    if ( strftime( buffer.get( ), buffer_size, "%x-%X %Z", &ti )  )
    {
      retval = buffer.get( );
    }
  }
  return retval;
}

static site_container_type&
site_info( )
{
  static site_container_type site_container;

  if ( site_container.size( ) == 0 )
  {
    //-------------------------------------------------------------------
    // Make thread safe
    //-------------------------------------------------------------------
    static MutexLock::lock_type	site_baton = MutexLock::Initialize( );
    
    MutexLock	l( site_baton );
    if ( site_container.size( ) == 0 )
    {
      site_container[ "wa" ] = "Hanford";
      site_container[ "la" ] = "Livingston";
      site_container[ "mit" ] = "LDAS_MIT";
      site_container[ "dev" ] = "Caltech-Dev";
      site_container[ "test" ] = "Caltech-Test";
      site_container[ "cit" ] = "Caltech-CIT";
      site_container[ "uwm" ] = "LDAS_UWM";
    }
  }
  return site_container;
}

static std::string
site_info_lookup( const std::string& LDASSystemName )
{
  size_t pos = LDASSystemName.find_last_of( "-" );
  if ( pos != std::string::npos )
  {
    site_container_type::const_iterator p
      = site_info( ).find( LDASSystemName.substr( pos + 1 ) );
    if ( p != site_info( ).end( ) )
    {
      return p->second;
    }
  }
  return LDASSystemName;
}


static void
manager_command( const std::string& Command )
{
  using namespace GenericAPI;
  using namespace GenericAPI::Symbols;

  LDAS_MANAGER_KEY::value_type	manager_key;

  LDAS_MANAGER_KEY::Get( manager_key );

  if ( manager_key.empty( ) )
  {
    return;
  }
  //---------------------------------------------------------------------
  // Establish a connection with the manager
  //---------------------------------------------------------------------

  int s = manager_connect( );

  if ( s < 0 )
  {
    return;
  }
  try {
    //-------------------------------------------------------------------
    // Format and send the request
    //-------------------------------------------------------------------
    std::ostringstream		request;
      
    request << manager_key << " NULL NULL"
	    << " eval " << Command
	    << std::endl;
    int b = ::write( s,
		     request.str( ).c_str( ), request.str( ).length( ) );
    //-------------------------------------------------------------------
    // Read the responce
    //-------------------------------------------------------------------
    static const int RESULT_BUFFER_SIZE = 1024;
    char result[ RESULT_BUFFER_SIZE ];

    while ( ( b = ::read( s,
			  &result[ 0 ], RESULT_BUFFER_SIZE - 1 ) )
	    == ( RESULT_BUFFER_SIZE - 1 ) )
    {
      //-----------------------------------------------------------------
      // Loop back over to read more from the manager.
      //-----------------------------------------------------------------
    }
    //-------------------------------------------------------------------
    // Close the connection
    //-------------------------------------------------------------------
    close( s );
  }
  catch( ... )
  {
    //-------------------------------------------------------------------
    // Something bad has happened.
    // Do some cleanup
    //-------------------------------------------------------------------
    if ( s >= 0 )
    {
      close( s );
    }
    //-------------------------------------------------------------------
    // Continue with error
    //-------------------------------------------------------------------
    throw;
  }
}

static int
manager_connect( )
{
  int s = -1;

  getaddrinfo_for_manager( );
  if ( manager_addr_info != (addr_info_type)NULL )
  {
    for (struct addrinfo* cur = manager_addr_info;
	 cur != ( struct addrinfo* )NULL;
	 cur = cur->ai_next )
    {
      s = ::socket( cur->ai_family,
		    cur->ai_socktype,
		    cur->ai_protocol );
      if ( s < 0 )
      {
	continue;
      }

      if ( ::connect( s,
		      cur->ai_addr,
		      cur->ai_addrlen ) < 0 )
      {
	close( s );
	s = -1;
	continue;
      }

      break;  /* okay we got one */
    }
  }

  return s;
}

//-----------------------------------------------------------------------
/// Retrieve the address information that is needed to connect to the
/// manager's emergency port.
///
/// \todo
///     Make this function callable outside of the main thread.
///     It has been noted not to work outside of the main thread
///     on CentOS 5.2.
//-----------------------------------------------------------------------
static void
getaddrinfo_for_manager( )
{
  using namespace GenericAPI;
  using namespace GenericAPI::Symbols;

  if ( manager_addr_info == (addr_info_type)NULL )
  {
    static MutexLock::lock_type	baton = MutexLock::Initialize( );

    MutexLock	l( baton );
    if ( manager_addr_info != (addr_info_type)NULL )
    {
      // Has been set by another thread.
      return;
    }
    
    LDAS_MANAGER_PORT_EMERGENCY::value_type	manager_port;
    LDAS_MANAGER_HOST::value_type		manager_host;

    LDAS_MANAGER_PORT_EMERGENCY::Get( manager_port );
    LDAS_MANAGER_HOST::Get( manager_host );

    if ( manager_host.empty( )
	 || ( manager_port == 0 ) )
    {
      return;
    }

    std::ostringstream	manager_port_string;
    manager_port_string << manager_port;

    //-------------------------------------------------------------------
    // Establish a connection with the manager
    //-------------------------------------------------------------------
    
    addr_info_type     		manager;
    struct addrinfo  		hints;

    ::memset( &hints, 0, sizeof(hints) );
    hints.ai_flags = (AI_NUMERICSERV | AI_ADDRCONFIG );
    hints.ai_family = AF_UNSPEC;
    hints.ai_socktype = SOCK_STREAM;
    
    manager_addr_info = (addr_info_type)NULL;
    if ( ::getaddrinfo( manager_host.c_str( ), manager_port_string.str( ).c_str( ),
			&hints,
			&manager ) == 0 )
    {
      manager_addr_info = manager;
    }
  }
}
