#ifndef DISKCACHE_API__META_COMMANDS_HH
#define DISKCACHE_API__META_COMMANDS_HH

#include <iomanip>

#include "general/CommandLineOptions.hh"
#include "general/Thread.hh"
#include "general/Singleton.hh"
#include "general/SharedPtr.hh"

#include "genericAPI/SocketStream.hh"

#include "diskcacheAPI/Cache/SDGTx.hh"

#include "MountPointManagerSingleton.hh"

namespace diskCache
{
  namespace MetaCommand
  {
    typedef General::CommandLineOptions 	CommandLineOptions;
    typedef CommandLineOptions::Option 		Option;
    typedef CommandLineOptions::OptionSet	OptionSet;
    typedef INT_4U				time_type;
    typedef General::SharedPtr< GenericAPI::Net::ServerSocketStream > server_type;
    typedef server_type::element_type::accept_stream_type	server_responce_type;
    typedef General::SharedPtr< GenericAPI::Net::SocketStream > client_type;
    typedef diskCache::Streams::Interface::version_type	version_type;

    enum {
      VERSION_DEFAULT_BINARY = diskCache::Streams::OBinary::VERSION_DEFAULT
    };
    enum {
      VERSION_DEFAULT_ASCII = diskCache::Streams::OASCII::VERSION_DEFAULT
    };

    class transfer_helper
    {
    public:
      typedef INT_4U	size_type;
      typedef CHAR_U	bool_type;

      template <typename Type>
      inline std::istream& Input( std::istream& Stream, Type& Data )
      {
	Stream.read( reinterpret_cast< char* >( &Data ),
		     sizeof( Type ) );
	return Stream;
      }
    
      template <typename Type>
      inline std::ostream& Output( std::ostream& Stream, Type Data )
      {
	Stream.write( reinterpret_cast< const char* >( &Data ),
		      sizeof( Data ) );
	return Stream;
      }
    
      //---------------------------------------------------------------------
      /// \brief Read a raw block of information
      //---------------------------------------------------------------------
      inline std::istream&
      Blob( std::istream& Stream, bool& Available, std::iostream& Buffer )
      {
	bool_type	a;

	Input( Stream, a );
	Available = a;
	if ( Available )
	{
	  size_type	length;

	  Input( Stream, length );
	  if ( length > 0 )
	  {
	    std::string	b;

	    std::streamsize offset( 0 );

	    b.resize( length );
	    do {
	      Stream.read( &(b[offset]),
			   (std::streamsize)(length ) - offset );
	      offset += Stream.gcount( );
	    } while ( offset < length );
	    Buffer.write( &(b[0]), length );
	    std::flush( Buffer );
	  }
	}
    
	return Stream;
      }

      //---------------------------------------------------------------------
      /// \brief Write a raw block of information
      //---------------------------------------------------------------------
      inline std::ostream&
      Blob( std::ostream& Stream, bool Available, std::iostream& Buffer )
      {
	bool_type	a( Available );
	size_type	len;

	Buffer.seekg( 0, std::ios_base::end );
	len = Buffer.tellg( );
	Buffer.seekg( 0, std::ios_base::beg );

	Output( Stream, a );
	if ( Available )
	{
	  Output( Stream, len );
	  Stream << Buffer.rdbuf( );
	}
	std::flush( Stream );
	return Stream;
      }
    };

    class ClientServerInterface
    {
    public:
      //-----------------------------------------------------------------
      // Server information
      //-----------------------------------------------------------------
      class ServerInfo
      {
      public:
	typedef int port_type;

	ServerInfo( );

	void Hostname( const std::string& Value );

	const std::string& Hostname( ) const;

	void Port( port_type Value );

	port_type Port( ) const;

      private:
	std::string		hostname;
	port_type		port;
      };

      typedef ServerInfo::port_type port_type;

      ClientServerInterface( const ServerInfo& Server );

      virtual ~ClientServerInterface( );

      void ClientHandle( server_responce_type Value );

      const ServerInfo& Server( ) const;

      const std::string& ServerHostname( ) const;

      port_type ServerPort( ) const;

      void ServerRequest( const std::string& Command );

      void operator()( );

    protected:
      server_responce_type clientHandle( );

      void eval( );

      virtual void evalClient( ) = 0;

      virtual void evalServer( ) = 0;
      
      virtual void evalStandalone( ) = 0;

      client_type serverRequestHandle( );

    private:
      const ServerInfo&		server;

      server_responce_type	handle_client;

      client_type		server_request_handle;
    };
    
    inline void ClientServerInterface::ServerInfo::
    Hostname( const std::string& Value )
    {
      hostname = Value;
    }

    inline const std::string& ClientServerInterface::ServerInfo::
    Hostname( ) const
    {
      return hostname;
    }

    inline void ClientServerInterface::ServerInfo::
    Port( port_type Value )
    {
      port = Value;
    }

    inline ClientServerInterface::ServerInfo::port_type ClientServerInterface::ServerInfo::
    Port( ) const
    {
      return port;
    }

    inline ClientServerInterface::
    ClientServerInterface( const ServerInfo& Server )
      : server( Server )
    {
    }

    inline const ClientServerInterface::ServerInfo& ClientServerInterface::
    Server( ) const
    {
      return server;
    }

    inline const std::string& ClientServerInterface::
    ServerHostname( ) const
    {
      return server.Hostname( );
    }

    inline ClientServerInterface::port_type ClientServerInterface::
    ServerPort( ) const
    {
      return server.Port( );
    }

    inline void ClientServerInterface::
    ServerRequest( const std::string& Command )
    {
      if ( ! server_request_handle )
      {
	server_request_handle = client_type( new client_type::element_type( ) );
	server_request_handle->open( ServerHostname( ), ServerPort( ) );
      }
      (*server_request_handle) << INT_4U( Command.size( ) );
      server_request_handle->write( &(Command[0]), Command.size( ) );
      server_request_handle->flush( );
    }

    inline void ClientServerInterface::
    operator()( )
    {
      eval( );
    }

    inline void ClientServerInterface::
    ClientHandle( server_responce_type Value )
    {
      handle_client = Value;
    }

    inline server_responce_type ClientServerInterface::
    clientHandle( )
    {
      return handle_client;
    }

    inline void ClientServerInterface::
    eval( )
    {
      if ( clientHandle( ) )
      {
	evalServer( );
      }
      else if ( ServerPort( ) >= 0 )
      {
	evalClient( );
      }
      else
      {
	evalStandalone( );
      }
    }

    inline client_type ClientServerInterface::
    serverRequestHandle( )
    {
      return server_request_handle;
    }
    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    class CommandTable
    {
      SINGLETON_TS_DECL( CommandTable );

    public:
      enum command_type {
	CMD_UNKNOWN,
	CMD_DAEMON,
	CMD_DUMP,
	CMD_FILENAMES,
	CMD_FILENAMES_RDS,
	CMD_INTERVALS,
	CMD_MOUNT_POINT_STATS,
	CMD_QUIT,
	CMD_SCAN
      };

      //-----------------------------------------------------------------
      /// \brief Destructor to release system resources
      //-----------------------------------------------------------------
      ~CommandTable( );

      //-----------------------------------------------------------------
      /// \brief Translate from ascii to enumerated type
      ///
      /// \param[in] Name
      ///     The ascii form of the command
      ///
      /// \return
      ///     The appropriate value from the command_type enumeration.
      ///     If Name cannot be mapped, then the value CMD_UNKNOWN is
      ///     returned.
      //-----------------------------------------------------------------
      static command_type Lookup( const std::string& Name );

      static const char* Lookup( command_type Command );

    private:
      typedef General::unordered_map< std::string, command_type >
      command_table_type;

      command_table_type	command_table;

      command_type lookup( const std::string& Name );

      const char* lookup( command_type Command );
    };

    class Daemon
      : public General::Thread
    {
    public:
      Daemon( CommandLineOptions& Args, const ClientServerInterface::ServerInfo& Server );

      void operator()( );

      static const OptionSet& Options( );

    private:
      enum {
	OPT_CONCURRENCY,
	OPT_EXCLUDED_DIRECTORIES,
	OPT_EXTENSIONS,
	OPT_LOG,
	OPT_MOUNT_POINTS,
	OPT_OUTPUT_ASCII,
	OPT_OUTPUT_BINARY,
	OPT_VERSION_ASCII,
	OPT_VERSION_BINARY
      };
      static bool	daemon_mode;
      static OptionSet&	m_options;

      static OptionSet& init_options( );

      //-----------------------------------------------------------------
      /// \brief Handle client side requests
      //-----------------------------------------------------------------
      void action( );

      void do_client_request( );

      bool process_cmd( CommandLineOptions& Options );

      bool read_command( char* Buffer, size_t BufferSize );

      CommandLineOptions	m_args;

      typedef diskCache::Cache::SDGTx::file_extension_container_type
      extensions_type;
      typedef MountPointManagerSingleton::mount_point_name_container_type
      mount_points_type;

      INT_4U		m_concurrency;
      extensions_type	m_extensions;
      std::string	m_log;
      mount_points_type	m_mount_points;
      std::string	output_ascii;
      std::string	output_binary;
      server_type	server;
      version_type	version_ascii;
      version_type	version_binary;
      //-----------------------------------------------------------------
      /// \brief
      ///
      /// \todo
      ///     This needs to be made thread safe by having a separate one
      ///     for each thread.
      //-----------------------------------------------------------------
      server_responce_type	client;
      bool			finished;

      const ClientServerInterface::ServerInfo&		server_info;
    };

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    class Dump
      : public ClientServerInterface
    {
    public:
      Dump( CommandLineOptions& Args,
	    const ClientServerInterface::ServerInfo& Server );

      static const OptionSet& Options( );

    protected:
      virtual void evalClient( );

      virtual void evalServer( );
      
      virtual void evalStandalone( );
      
    private:
      enum {
	OPT_IFO,
	OPT_OUTPUT_ASCII,
	OPT_OUTPUT_BINARY,
	OPT_TYPE,
	OPT_VERSION_ASCII,
	OPT_VERSION_BINARY
      };

      static OptionSet&	m_options;

      CommandLineOptions	m_args;

      std::string		m_ifo;
      std::string		m_output_ascii;
      std::string		m_output_binary;
      std::string		m_type;
      version_type		m_version_ascii;
      version_type		m_version_binary;

      static OptionSet& init_options( );
    };

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    class Filenames
      : public ClientServerInterface
    {
    public:
      Filenames( CommandLineOptions& Args,
		 const ClientServerInterface::ServerInfo& Server );

      static const OptionSet& Options( );

    protected:
      virtual void evalClient( );

      virtual void evalServer( );
      
      virtual void evalStandalone( );
      
      void process( std::ostream& Stream );

    private:
      enum {
	OPT_EXTENSION,
	OPT_IFO_TYPE_LIST,
	OPT_START_TIME,
	OPT_END_TIME
      };
      static OptionSet&	m_options;

      static OptionSet& init_options( );

      CommandLineOptions	m_args;

      std::string		m_extension;
      std::string		m_ifo_type_list;
      time_type			m_time_start;
      time_type			m_time_stop;
    };

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    class FilenamesRDS
      : public ClientServerInterface
    {
    public:
      typedef std::list< std::string > query_results_type;

      FilenamesRDS( CommandLineOptions& Args,
		    const ClientServerInterface::ServerInfo& Server );

      const query_results_type& Results( ) const;

      static const OptionSet& Options( );

    protected:
      virtual void evalClient( );

      virtual void evalServer( );
      
      virtual void evalStandalone( );
      
      void process( std::ostream& Stream );

    private:
      enum {
	OPT_EXTENSION,
	OPT_IFO,
	OPT_TYPE,
	OPT_START_TIME,
	OPT_END_TIME,
	OPT_RESAMPLE
      };
      static OptionSet&	m_options;

      static OptionSet& init_options( );

      CommandLineOptions	m_args;

      std::string		m_extension;
      std::string		m_ifo;
      bool			m_resample;
      time_type			m_time_start;
      time_type			m_time_stop;
      std::string		m_type;

      query_results_type	results;
    };

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    class Intervals
      : public ClientServerInterface
    {
    public:
      Intervals( CommandLineOptions& Args,
		 const ClientServerInterface::ServerInfo& Server );

      static const OptionSet& Options( );

    protected:
      virtual void evalClient( );

      virtual void evalServer( );
      
      virtual void evalStandalone( );
      
      void process( std::ostream& Stream );

    private:
      enum {
	OPT_IFO_TYPE_LIST,
	OPT_START_TIME,
	OPT_END_TIME,
	OPT_EXTENSION
      };

      static OptionSet&	m_options;

      static OptionSet& init_options( );

      CommandLineOptions	m_args;

      std::string		m_extension;
      std::string		m_ifo_type_list;
      time_type			m_time_start;
      time_type			m_time_stop;
    };

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    class MountPointStats
    {
    public:
      MountPointStats( CommandLineOptions& Args );

      void operator()( );

      static const OptionSet& Options( );

    private:
      enum {
	OPT_IFO,
	OPT_TYPE
      };
      static OptionSet&	m_options;

      static OptionSet& init_options( );

      CommandLineOptions	m_args;

      std::string				m_ifo;
      std::string				m_type;
    };

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    class Scan
    {
    public:
      Scan( CommandLineOptions& Args );

      void operator()( );

      static const OptionSet& Options( );

    private:
      enum {
	OPT_CONCURRENCY,
	OPT_EXTENSIONS,
	OPT_MOUNT_POINTS,
	OPT_OUTPUT_ASCII,
	OPT_OUTPUT_BINARY,
	OPT_TYPE,
	OPT_VERSION_ASCII,
	OPT_VERSION_BINARY
      };

      typedef diskCache::Cache::SDGTx::file_extension_container_type
      extensions_type;
      typedef MountPointManagerSingleton::mount_point_name_container_type
      mount_points_type;

      static OptionSet&	m_options;

      static OptionSet& init_options( );

      CommandLineOptions	m_args;

      extensions_type		m_extensions;
      mount_points_type		m_mount_points;
      std::string		m_output_ascii;
      std::string		m_output_binary;
      std::string		m_type;
      version_type		m_version_ascii;
      version_type		m_version_binary;
    };

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    inline CommandTable::command_type CommandTable::
    Lookup( const std::string& Name )
    {
      return Instance( ).lookup( Name );
    }

    inline const char* CommandTable::
    Lookup( command_type Command )
    {
      return Instance( ).lookup( Command );
    }

    inline CommandTable::command_type CommandTable::
    lookup( const std::string& Name )
    {
      command_table_type::const_iterator	pos( command_table.find( Name ) );

      //-----------------------------------------------------------------
      // Check if the command exists in the command table
      //-----------------------------------------------------------------
      if ( pos != command_table.end( ) )
      {
	//---------------------------------------------------------------
	// It does; return the appropriate enumuration.
	//---------------------------------------------------------------
	return pos->second;
      }
      //-----------------------------------------------------------------
      // It does not so return unknown command.
      //-----------------------------------------------------------------
      return CMD_UNKNOWN;
    }

    inline const char* CommandTable::
    lookup( command_type Command )
    {
      static const char*	not_found = "";

      command_table_type::const_iterator
	last( command_table.end( ) );
      for ( command_table_type::const_iterator
	      cur( command_table.begin( ) );
	    cur != last;
	    ++cur )
      {
	if (cur->second == Command )
	{
	  return cur->first.c_str( );
	}
      }
      return not_found;
    }

    //===================================================================
    //===================================================================
    inline const FilenamesRDS::query_results_type& FilenamesRDS::
    Results( ) const
    {
      return results;
    }

  }
} // namespace - diskCache

#endif /* DISKCACHE_API__META_COMMANDS_HH */
