#if HAVE_CONFIG_H
#include "general/config.h"
#endif /* HAVE_CONFIG_H */

#include <setjmp.h>
#include <signal.h>

#include <new>
#include <typeinfo>

#include <general/config.h>
#include <general/unittest.h>

#include "unexpected_exception.hh"

// Only need to include this file so we can turn off logging
#include "LDASUnexpected.hh"

#if HAVE_RETHROW_WORKING_IN_BAD_EXCEPTION && ! HAVE_NO_RETURN_FROM_UNEXPECTED_HANDLER
#define	FULL_TEST 1
#endif

General::UnitTest Test;
jmp_buf	jmpbuffer;

extern "C" {
  static void sig_abort(int);	/* handler for abort signal */
}

// Test with no exception specification
void TestValid001()
{
    throw std::runtime_error("Thrown from TestValid001");
}

// Test with correct exception specification
void TestValid002()
    throw(std::runtime_error)
{
    throw std::runtime_error("Thrown from TestValid002");
}

// Test with correct exception specification but invalid throw
void TestValid003()
    throw(std::invalid_argument, General::unexpected_exception)
{
    throw std::runtime_error("Thrown from TestValid003");
}

// Test with incorrect exception specification
template<class E>
void ThrowUnexpected(const std::string& what)
    throw(General::unexpected_exception)
{
    throw E(what);
}

template<class E>
void ThrowUnexpected()
    throw(General::unexpected_exception)
{
    throw E();
}

template<class E>
void TestValid(const std::string& name, const std::string& what)
{
    using General::unexpected_exception;

    if ( setjmp(jmpbuffer) != 0 )
    {
	Test.Check(false) << "Caught Abort Signal" << std::endl;
    }
    else
    {
      try {
	ThrowUnexpected<E>(what);
      }
      catch(unexpected_exception& e)
      {
	Test.Check(true) << "Caught unexpected_exception(\""
			 << e.what() << "\", \""
			 << e.msg() << "\")"
			 << std::endl;
	
#if defined( FULL_TEST )
	Test.Check(std::string("std::")+std::string(e.what()) == name)
	  << "Names match - " << e.what() << " == " << name << std::endl;
	
	Test.Check(std::string(e.msg()) == what)
	  << "What()s match - " << e.msg() << " == " << what << std::endl;
#endif	/* FULL_TEST */
      }
      catch(...)
      {
	Test.Check(false) << "Caught wrong exception" << std::endl;
      }
    }
}

template<class E>
void TestValid(const std::string& name)
{
    using General::unexpected_exception;

    if ( setjmp(jmpbuffer) != 0 )
    {
	Test.Check(false) << "Caught Abort Signal" << std::endl;
    }
    else
    {
      try {
#if defined( FULL_TEST )
	ThrowUnexpected<E>();
#else /* FULL_TEST */
	//---------------------------------------------------------------
	// This ugly piece of code is needed because of an apparent
	//   problem with gcc 3.0 which generates an ABORT if this is
	//   not done.
	//---------------------------------------------------------------
	try {
	  throw E();
	}
#define	CATCH(EX) \
	catch( EX & e ) \
	{ \
	  ThrowUnexpected<EX>(); \
	}
	CATCH(std::bad_alloc)
	CATCH(std::bad_exception)
	CATCH(std::bad_typeid)
	CATCH(std::bad_cast)
	CATCH(std::exception)
#undef CATCH
	catch(...)
	{
	  Test.Check(false) << "Unknown exception throw: " << name
			    << std::endl;
	}
#endif /* FULL_TEST */
      }
      catch(unexpected_exception& e)
      {
	Test.Check(true) << "Caught unexpected_exception(\""
			 << e.what() << "\", \""
			 << e.msg() << "\")"
			 << std::endl;
	
#if defined( FULL_TEST )
	Test.Check(std::string("std::")+std::string(e.what()) == name)
	  << "Names match - " << e.what() << " == " << name << std::endl;
#endif	/* FULL_TEST */
      }
      catch(...)
      {
        Test.Check(false) << "Caught wrong exception" << std::endl;
      }
    }
}

void TestValid_bad_alloc( void )
{
    const std::string name("std::bad_alloc");
    using General::unexpected_exception;

    if ( setjmp(jmpbuffer) != 0 )
    {
	Test.Check(false) << "Caught Abort Signal" << std::endl;
    }
    else
    {
      try {
	ThrowUnexpected<std::bad_alloc>();
      }
      catch(unexpected_exception& e)
      {
	Test.Check(true) << "Caught unexpected_exception(\""
			 << e.what() << "\", \""
			 << e.msg() << "\")"
			 << std::endl;
	
#if defined( FULL_TEST )
	Test.Check(std::string("std::")+std::string(e.what()) == name)
	  << "Names match - " << e.what() << " == " << name << std::endl;
#endif	/* FULL_TEST */
      }
      catch(...)
      {
        Test.Check(false) << "Caught wrong exception" << std::endl;
      }
    }
}

// Test with incorrect exception specification
void TestInvalid001()
    throw()
{
    throw std::runtime_error("Thrown from TestInvalid001");
}

int main(int argc, char** argv)
{
    Test.Init(argc, argv);

    using General::unexpected_exception;
    using General::LDASUnexpected;

    General::LDASUnexpected::makeAbort( false );
    signal(SIGABRT, sig_abort);

    if (!Test.IsVerbose())
    {
	LDASUnexpected::makeQuiet();
    }

#if ! HAVE_NO_RETURN_FROM_UNEXPECTED_HANDLER
    try {
	TestValid001();
    }
    catch(std::runtime_error& e)
    {
	Test.Check(true) << "Caught runtime_error(\""
			 << e.what() << "\")" << std::endl;
    }
    catch(...)
    {
	Test.Check(false) << "Caught wrong exception" << std::endl;
    }
#endif	/* ! HAVE_NO_RETURN_FROM_UNEXPECTED_HANDLER */

    try {
	TestValid002();
    }
    catch(std::runtime_error& e)
    {
	Test.Check(true) << "Caught runtime_error(\""
			 << e.what() << "\")" << std::endl;
    }
    catch(...)
    {
	Test.Check(false) << "Caught wrong exception" << std::endl;
    }

#if FULL_TEST
    try {
      TestValid003();
    }
    catch(unexpected_exception& e)
    {
	Test.Check(true) << "Caught unexpected_exception(\""
			 << e.what() << "\", \""
			 << e.msg() << "\")"
			 << std::endl;
    }
    catch(...)
    {
	Test.Check(false) << "Caught wrong exception" << std::endl;
    }
#endif /* FULL_TEST */

#if FULL_TEST
#define TEST_VALID(exc)  TestValid<exc>(std::string(#exc), \
                                        std::string("Testing validity of " #exc))

#define TEST_VALID2(exc) TestValid<exc>(std::string(#exc))

    TEST_VALID(std::length_error);
    TEST_VALID(std::domain_error);
    TEST_VALID(std::out_of_range);
    TEST_VALID(std::invalid_argument);
    TEST_VALID(std::underflow_error);
    TEST_VALID(std::overflow_error);
    TEST_VALID(std::range_error);
    TEST_VALID(std::logic_error);
    TEST_VALID(std::runtime_error);

    TEST_VALID2(std::bad_alloc);
    TEST_VALID2(std::bad_exception);
    TEST_VALID2(std::bad_typeid);
    TEST_VALID2(std::bad_cast);
    TEST_VALID2(std::exception);

    TestValid_bad_alloc();
#endif	/* FULL_TEST */

#if ! HAVE_NO_RETURN_FROM_UNEXPECTED_HANDLER
    // This test unavoidably causes an abort, as it should
    if ( setjmp(jmpbuffer) != 0 )
    {
	Test.Check(true) << "Caught Abort Signal" << std::endl;
    }
    else
    {
      try {
	TestInvalid001();
      }
      catch(...)
      {
	// TestInvalid001 should cause an abort - if we catch
	// an exception in this case there is a bug!
	Test.Check(false) << "Caught an exception!"
			 << std::endl;
      }
    }
#endif	/* ! HAVE_NO_RETURN_FROM_UNEXPECTED_HANDLER */

    Test.Exit();

    return 0;
}

static void
sig_abort(int)
{
  signal(SIGABRT, sig_abort);
  longjmp(jmpbuffer, 1);
}
