/* -*- mode: c++; c-basic-offset: 4; -*- */

#ifndef MUTEXLOCK_HH
#define MUTEXLOCK_HH

#if ! defined(SWIGIMPORTED)
#include <assert.h>
#include <errno.h>
#include <string.h>

#include <pthread.h>

#include <iostream>
#include <stdexcept>
#endif /* ! defined(SWIGIMPORTED) */

/// \class MutexLock
///
/// \brief Descibes an object which release a mutex lock
///
/// This class ensures that the given mutex is unlocked whenever an
/// exception is thrown but not caught in a locked section.
///
/// It should be used like this:
///
/// \code
/// {
///     MutexLock lock(pthread_mutex_t& mutex); // mutex is now locked
///     do whatever...
///     ...
/// } // MutexLock has been destructed so mutex is now unlocked
/// \endcode
///
///
/// Creating two MutexLock objects with the same mutex in the same scope
/// will cause a deadlock.
///
class MutexLock {
public:
    /// \brief Alias for the system specific mutex type.
    typedef pthread_mutex_t	lock_type;

    /// \brief  Constructor
    MutexLock(lock_type& mutex,
	      const char* const File = "", const unsigned int Line = 0,
	      bool Logging = true );

    /// \brief  Constructor
    MutexLock(lock_type* const mutex_ptr,
	      const char* const File = "", const unsigned int Line = 0,
	      bool Logging = true );

    /// \brief  Destructor
    ~MutexLock( );

    /// \brief  Release the lock before object is destructed
    void Release( const char* const File = "", const unsigned int Line = 0 );

    ///\brief Releases resources associated with Lock.
    inline static int Destroy( lock_type& Lock )
    {
	return pthread_mutex_destroy( &Lock );
    }

    ///\brief Allocates resources needed for lock.
    inline static lock_type Initialize( )
    {
	static const lock_type retval = PTHREAD_MUTEX_INITIALIZER;

	return retval;
    }

    /// \brief Allocate resources needed for Lock.
    inline static void Initialize( lock_type& Lock )
    {
	pthread_mutex_init( &Lock,
			    static_cast< pthread_mutexattr_t* >( NULL ) );
    }

    /// \brief Request exclusive access to Lock.
    inline static void Lock( lock_type& Lock )
    {
	pthread_mutex_lock( &Lock );
    }

    /// \brief Request exclusive access to Lock.
    inline static void Lock( lock_type* Lock )
    {
	pthread_mutex_lock( Lock );
    }

    /// \brief Relinquishes exclusive access to Lock.
    inline static void UnLock( lock_type& Lock )
    {
	pthread_mutex_unlock( &Lock );
    }

    /// \brief Relinquishes exclusive access to Lock.
    inline static void UnLock( lock_type* Lock )
    {
	pthread_mutex_unlock( Lock );
    }

    //-------------------------------------------------------------------
    /// \brief Handler for thead cancelation
    ///
    /// \param[in] VLock
    ///     The VLock is a pointer to a ReadWriteLock object that holds
    ///     a lock.
    //-------------------------------------------------------------------
    static void ThreadCancel( void* VLock );

private:
    /// \brief  Default Constructor
    ///
    /// Default constructor is private to prevent copying of classes
    /// internal data.
    MutexLock();

    /// \brief  Copy Constructor
    ///
    /// Copy constructor is private to prevent copying of classes
    ///   internal data.
    MutexLock(const MutexLock&);

    /// \brief  Assigment operator
    ///
    /// Assignment operator is privat to prevent copying of classes internal
    /// data
    const MutexLock& operator=(const MutexLock&);

    /// \brief  Object to prevent multiple access to critical sections of code.
    lock_type* m_mutex_ptr;

    /// \brief Keeps track of logging state.
    bool	m_logging;
};

namespace General
{
    template< typename V >
    class MutexLockVariable
    {
    public:
	MutexLockVariable( MutexLock::lock_type& Baton,
			   V& Variable,
			   const char* Filename = "",
			   int Linenum = 0 );

	MutexLockVariable( MutexLock::lock_type& Baton,
			   const V& Variable,
			   const char* Filename = "",
			   int Linenum = 0 );
	MutexLockVariable( const MutexLockVariable& Source,
			   const char* Filename = "",
			   int Linenum = 0 );
			     
	const V& Var( ) const;

	V& Var( );

    private:
	MutexLockVariable( );

	MutexLock::lock_type&	m_baton;
	V&			m_variable;
	mutable MutexLock	m_lock;
	mutable bool		m_is_locked;

	MutexLock::lock_type& release( ) const;
    };

    template< typename V >
    inline MutexLockVariable< V >::
    MutexLockVariable(  MutexLock::lock_type& Baton,
			    V& Variable,
			    const char* Filename,
			    int Linenum )
	: m_baton( Baton ),
	  m_variable( Variable ),
	  m_lock( Baton, Filename, Linenum ),
	  m_is_locked( true )
    {
    
    }
  
    template< typename V >
    inline MutexLockVariable< V >::
    MutexLockVariable(  MutexLock::lock_type& Baton,
			    const V& Variable,
			    const char* Filename,
			    int Linenum )
	: m_baton( Baton ),
	  m_variable( const_cast< V& >( Variable ) ),
	  m_lock( Baton, Filename, Linenum ),
	  m_is_locked( true )
    {
    
    }
  
    template< typename V >
    inline MutexLockVariable< V >::
    MutexLockVariable(  const MutexLockVariable& Source,
			    const char* Filename,
			    int Linenum )
	: m_baton( Source.m_baton ),
	  m_variable( Source.m_variable ),
	  m_lock( Source.release( ), Filename, Linenum ),
	  m_is_locked( true )
    {
    
    }

    template<  typename V >
    inline const V& MutexLockVariable< V >::
    Var( ) const
    {
	if ( m_is_locked == false )
	{
	    throw std::logic_error( "MutexLockVariable: data not protected by lock" );
	}
	return m_variable;
    }

    template< typename V >
    inline V& MutexLockVariable< V >::
    Var( )
    {
	if ( m_is_locked == false )
	{
	    throw std::logic_error( "MutexLockVariable: data not protected by lock" );
	}
	return m_variable;
    }

    template< typename V >
    MutexLock::lock_type& MutexLockVariable< V >::
    release( ) const
    {
	m_lock.Release( );	// Reliquish lock
	m_is_locked = false;
	return m_baton;
    }
}

#endif // MUTEXLOCK_HH
