/* -*- mode: c++; c-basic-offset: 4; -*- */

#include "general/config.h"

#include "general/DeadLockDetector.hh"
#include "general/mutexlock.hh"

#if DEAD_LOCK_DETECTOR_ENABLED
using General::DeadLockDetector;

namespace
{
    /// \brief  Log the thread state
    inline void log_lock( DeadLockDetector::state_type State,
			  MutexLock::lock_type* Lock,
			  const char* File,
			  unsigned int Line,
			  bool Logging )
    {
	if ( Logging )
	{
	    DeadLockDetector::SetState( DeadLockDetector::MUTEX,
					Lock,
					State,
					File, Line );
	}
    }
}

#else /* DEAD_LOCK_DETECTOR_ENABLED */
#define log_lock( State, Lock, File, Line, Logging )
#endif /* DEAD_LOCK_DETECTOR_ENABLED */

///----------------------------------------------------------------------
///
/// This contructor creates an object to ensure the releasing
/// of the mutex lock once the object goes out of scope.
/// The lock held by the object can be released early by calling
/// the Release method.
///
/// \param mutex The mutex to provide lock syncronization.
/// \param File The from where the request was made.
/// \param Line The line number from where the request was made.
/// \param Logging True if this object is to be logged, false otherwise
///
/// \see Release
///----------------------------------------------------------------------
MutexLock::
MutexLock(lock_type& mutex, const char* File, const unsigned int Line,
	  bool Logging )
    : m_mutex_ptr(&mutex),
      m_logging( Logging )
{
    int err;

    log_lock( DeadLockDetector::PENDING, m_mutex_ptr,
	      __FILE__, __LINE__,
	      m_logging );
    if ( ( err = pthread_mutex_lock(m_mutex_ptr) ) != 0 )
    {
	std::cerr << "MutexLock: "<< strerror( err )
		  << " File: " << __FILE__ << " Line: " << __LINE__
		  << std::endl;
	log_lock( DeadLockDetector::RELEASED, m_mutex_ptr,
		  __FILE__, __LINE__,
		  m_logging );
	assert( false );
    }
    else
    {
	log_lock( DeadLockDetector::ACQUIRED, m_mutex_ptr,
		  __FILE__, __LINE__,
		  m_logging );
    }
}

MutexLock::
MutexLock(lock_type* const mutex_ptr,
	  const char* File, const unsigned int Line,
	  bool Logging )
    : m_mutex_ptr(mutex_ptr),
      m_logging( Logging )
      
{
    int err;

    log_lock( DeadLockDetector::PENDING, m_mutex_ptr,
	      __FILE__, __LINE__,
	      m_logging );
    if ( ( err = pthread_mutex_lock(m_mutex_ptr) ) != 0 )
    {
	std::cerr << "MutexLock: " << strerror( err )
		  << " File: " << __FILE__ << " Line: " << __LINE__
		  << std::endl;
	log_lock( DeadLockDetector::RELEASED, m_mutex_ptr,
		  __FILE__, __LINE__,
		  m_logging );
    }
    else
    {
	log_lock( DeadLockDetector::ACQUIRED, m_mutex_ptr,
		  __FILE__, __LINE__,
		  m_logging );
    }
}

MutexLock::
~MutexLock()
{
    Release( __FILE__, __LINE__ );
}

void MutexLock::
Release( const char* File, const unsigned int Line )
{
    if ( m_mutex_ptr )
    {
	int err;

	if ( ( err = pthread_mutex_unlock(m_mutex_ptr) ) != 0 )
	{
	    std::cerr << "MutexLock::Release : "
		      << " File: " << __FILE__ << " Line: " << __LINE__
		      << strerror( err )
		      << std::endl;

	}
	else
	{
	    log_lock( DeadLockDetector::RELEASED, m_mutex_ptr,
		      __FILE__, __LINE__,
		      m_logging );
	}
	m_mutex_ptr = (lock_type*)NULL;
    }
}

//-----------------------------------------------------------------------
/// This routine is used when a thread is canceled and the lock needs to
/// be release.
/// It is currently only written to support pthread_cancel_push().
//-----------------------------------------------------------------------
void MutexLock::
ThreadCancel( void* VLock )
{
  if ( VLock )
  {
    MutexLock*	l ( reinterpret_cast< MutexLock*>( VLock ) );

    l->Release( __FILE__, __LINE__ );
  }
}
