#include "general/config.h"

#include <sys/types.h>
#include <pwd.h>
#include <unistd.h>

#include <cstdlib>

#include "general/AtExit.hh"
#include "general/UserInfoCache.hh"

// Instantiate generic to Singleton class methods:   
// see "general" library   
SINGLETON_TS_INST( General::UserInfoCache );

#if !defined(_SC_GETPW_R_SIZE_MAX)
#define _SC_GETPW_R_SIZE_MAX 1024
#endif /* !defined(_SC_GETPW_R_SIZE_MAX) */

namespace General
{
  //---------------------------------------------------------------------
  /// Initialize anything that is specific to the class.
  //---------------------------------------------------------------------
  UserInfoCache::
  UserInfoCache( )
    : m_cache( ),
      m_names( ),
      m_baton( MutexLock::Initialize( ) )
  {
    init( );

    General::AtExit::Append( singleton_suicide,
			     "General::UserInfoCache::singleton_suicide",
			     10 );
  }

  //---------------------------------------------------------------------
  /// Release resource associated with this object
  ///
  /// \todo
  ///     Release the m_names and m_cache resource.
  ///     Need to be sure to use the baton to insure exclusive rights.
  //---------------------------------------------------------------------
  UserInfoCache::
  ~UserInfoCache( )
  {
  }

  //---------------------------------------------------------------------
  /// Look up a user in the password database and cache the information
  /// to prevent excessive use of system calls.
  /// If the user id cannot be found in the password database,
  /// an entry is created with the user name being the string
  /// representation of the user id.
  /// \note
  ///     This call is internal and is only called if nothing is known
  ///     about the specified user id.
  //---------------------------------------------------------------------
  const UserInfoCache::UserInfo& UserInfoCache::
  create( UserInfo::uid_type Id )
  {
    MutexLock l( m_baton );

    static int BUF_SIZE = sysconf( _SC_GETPW_R_SIZE_MAX );
    static char* pw_buf = (char*)calloc( sizeof( char ), BUF_SIZE );

    struct passwd pwinfo;
    struct passwd* pwptr;

    ::getpwuid_r( uid_t( Id ), &pwinfo, pw_buf, BUF_SIZE, &pwptr );
    if ( pwptr )
    {
      m_cache[ UserInfo::uid_type( pwinfo.pw_uid ) ] =
	UserInfo( pwinfo.pw_name, pwinfo.pw_passwd,
		  UserInfo::uid_type( pwinfo.pw_uid ),
		  UserInfo::gid_type( pwinfo.pw_gid ),
		  pwinfo.pw_gecos, pwinfo.pw_dir, pwinfo.pw_shell );
      m_names[ std::string( pwinfo.pw_name ) ]
	= UserInfo::uid_type( pwinfo.pw_uid );
    }
    else
    {
      std::ostringstream	name;
      name << Id;
      m_cache[ Id ] = UserInfo( name.str( ).c_str( ), "*",
				Id, UserInfo::gid_type( Id ),
				name.str( ).c_str( ), "/", "/bin/sh" );
      m_names[ name.str( ) ] = Id;
    }
    return m_cache[ Id ];
  }

  bool UserInfoCache::
  init( )
  {
    //---------------------------------------------------------------------
    // Loop over all entries in the password file
    //---------------------------------------------------------------------
    setpwent( );	// Rewind to start of file;
    struct passwd* cur = getpwent( );	// Get first entry
    while( cur )
    {
      m_cache[ UserInfo::uid_type( cur->pw_uid ) ] =
	UserInfo( cur->pw_name, cur->pw_passwd,
		  UserInfo::uid_type( cur->pw_uid ),
		  UserInfo::gid_type( cur->pw_gid ),
		  cur->pw_gecos, cur->pw_dir, cur->pw_shell );
      m_names[ std::string( cur->pw_name ) ]
	= UserInfo::uid_type( cur->pw_uid );
      cur = getpwent( );	// Advance to the next entry
    }
    endpwent( );
    return true;
  }

  const UserInfoCache::UserInfo& UserInfoCache::
  uid( UserInfo::uid_type Id )
  {
    {
      MutexLock	l( m_baton );

      cache_type::const_iterator pos = m_cache.find( Id );
      if ( pos != m_cache.end( ) )
      {
	return pos->second;
      }
    }

    return create( Id );
  }

} // namespace General

