/* -*- mode: C++ -*- */
#ifndef UNITTEST_TS_H
#define	UNITTEST_TS_H

#include <stdlib.h>
#include <strings.h>

#include <cstdlib>
#include <cstring>

#include <iostream>
#include <fstream>
#include <sstream>
#include <string>

#include "general/AtExit.hh"
#include "general/mutexlock.hh"

namespace General {
  /// \brief  Provide standard routines for unit testing of code.
  ///
  class UnitTestTS
  {
  public:
    /// \brief  Constructor
    UnitTestTS(void);
    /// \brief  Destructor
    ~UnitTestTS(void);
    /// \brief  Check for error conditions
    std::ostream& Check(bool TestResult);
    /// \brief  Check for error conditions
    void Check(bool TestResult, std::string Message);
    /// \brief  Exit with the appropriate exit code
    void Exit(void);
    /// \brief  Initialize the class with command line arguments
    void Init(int ArgC, char** ArgV);
    /// \brief  Check if in verbose mode
    bool IsVerbose( int Level = 0 ) const
    {
       return ( Level <= m_verbose );
    }   
    void Lock( );
    /// \brief  Output info message
    std::ostream& Message( bool Leader = true ) const;
    /// \brief  Output info message
    std::ostream& Message( int Level, bool Leader = true ) const;
    void UnLock( );

  private:
    /// \brief  Level of verbosity
    int			m_verbose;
    /// \brief  Failure status flag
    bool		m_status;
    /// \brief  Pointer to stream when in non-verbose mode
    mutable std::ostringstream	m_null_stream;
    /// \brief	Lock when used in a mult-threaded environment
    MutexLock::lock_type	m_key;
  }; // class - UnitTest

  inline UnitTestTS::
  UnitTestTS(void)
    : m_verbose( -1 ),
      m_status(true),
      m_key( MutexLock::Initialize( ) )
  {
    if ( getenv("TEST_VERBOSE_MODE") )
    {
      if ( 0 == strcmp( getenv( "TEST_VERBOSE_MODE" ), "true" ) )
      {
	m_verbose = 0;
      }
      else if ( 0 == strcmp( getenv( "TEST_VERBOSE_MODE" ), "false" ) )
      {
	m_verbose = -1;
      }
      else
      {
	m_verbose = atoi( getenv( "TEST_VERBOSE_MODE" ) );
      }
    }
  }
  
  inline UnitTestTS::
  ~UnitTestTS(void)
  {
  }

  inline std::ostream& UnitTestTS::
  Check(bool TestResult)
  {
    m_status = m_status && TestResult;
    if ( IsVerbose() )
    {
      std::cout << "-- " << ((TestResult) ? "PASS" : "FAIL") << ": ";
      return std::cout;
    }
    m_null_stream.str( "" );
    return m_null_stream;
  }

  inline void UnitTestTS::
  Check(bool TestResult, std::string Message)
  {
    MutexLock	l( m_key );

    m_status = m_status && TestResult;
    if ( IsVerbose() )
    {
      std::cout << "-- " << ((TestResult) ? "PASS" : "FAIL") << ": " << Message
		<< std::endl << std::flush;
    }
  }

  inline void UnitTestTS::
  Exit(void)
  {
    General::AtExit::Cleanup( );

    int e;
    {
      MutexLock	l( m_key );

      e = (m_status) ? 0 : 1;
    }
    exit ( e );
  }

  inline void UnitTestTS::
  Init(int ArgC, char** ArgV)
  {
    MutexLock	l( m_key );

    for (int x = 1; x < ArgC; x++)
    {
      if (0 == (strcmp(ArgV[x], "--verbose")))
      {
	m_verbose = 0;
	continue;
      }
    }
  }

  inline void UnitTestTS::
  Lock( )
  {
    MutexLock::Lock( m_key );
  }

  inline std::ostream& UnitTestTS::
  Message( bool Leader ) const
  {
    if ( IsVerbose() )
    {
      if (Leader)
      {
	std::cout << "-- MESG: ";
      }
      return std::cout;
    }
    m_null_stream.str( "" );
    return m_null_stream;
  }

  inline std::ostream& UnitTestTS::
  Message( int Level, bool Leader ) const
  {
    if ( IsVerbose( Level ) )
    {
      return Message( Leader );
    }
    m_null_stream.str( "" );
    return m_null_stream;
  }

  inline void UnitTestTS::
  UnLock( )
  {
    MutexLock::UnLock( m_key );
  }

} // namespace - General

#endif	/* UNITTEST_TS_H */
