#ifndef GENERAL__READ_WRITE_LOCK_HH
#define	GENERAL__READ_WRITE_LOCK_HH

#include "general/config.h"

#include <cassert>

#include <stdexcept>
#include <string>

#include "general/mutexlock.hh"
#include "general/Thread.hh"
#include "general/unordered_map.hh"

namespace General
{
  class ReadWriteLock
  {
  public:
    class BusyError
      : public std::runtime_error
    {
    public:
      BusyError( );
    private:
      std::string msg( );
    };

    class TimedoutError
      : public std::runtime_error
    {
    public:
      TimedoutError( );
    private:
      std::string msg( );
    };

    class Sync
    {
    public:
      enum lock_mode {
	NONE   = 0x0000,
	READ   = 0x0001,
	WRITE  = 0x0002
      };

      enum lock_state {
	STATE_NONE = 0x0000,
	PENDING	   = 0x0001,
	ACTIVE     = 0x0002
      };

#if HAVE_PTHREAD_RW_LOCK
      typedef pthread_rwlock_t	lock_type;
#else /* HAVE_PTHREAD_RW_LOCK */
      typedef pthread_mutex_t	lock_type;
#endif /* HAVE_PTHREAD_RW_LOCK */
      static lock_type	INITIALIZER;

      struct info {
	inline info ()
	  : m_mode( NONE ),
	    m_state( STATE_NONE )
	{
	}

	lock_mode	m_mode;
	lock_state	m_state;
      };

      Sync( bool Verbose = true );
      Sync( lock_type& Source, bool Verbose = false );
      Sync( lock_type* Source, bool Verbose = false );

      int Lock( lock_mode Mode,
		const char* Filename,
		int Linenum );

      int Lock( lock_mode Mode,
		bool TryLock,
		const char* Filename,
		int Linenum );

      int Lock( lock_mode Mode,
		int Timeout,
		const char* Filename,
		int Linenum );

      int Modify( lock_mode Mode,
		  const char* Filename,
		  int Linenum );

      int Unlock( const char* Filename,
		  int Linenum );

      //-----------------------------------------------------------------
      /// \brief Check if the current thread already owns the resource.
      //-----------------------------------------------------------------
      bool HasLock( ) const;

      //-----------------------------------------------------------------
      /// \brief Verify that a lock of the appropriate mode.
      ///
      /// \return
      ///     False if no lock is held.
      ///     True if a lock with sufficient priveledges is held.
      /// \note
      ///     An exception is thrown if a write lock is requested
      ///     and the thread currently holds a read lock.
      //-----------------------------------------------------------------
      bool IsLocked( lock_mode Mode ) const;

    private:

      struct hash_func {
	inline size_t operator()( const void* Key ) const
	{
	  return reinterpret_cast< size_t >( Key );
	}
      };

      void conflict( ) const;

      lock_mode		mode( ) const;

      void set( lock_mode Mode, lock_state State );

      void unset( );

      typedef General::unordered_map< void*, info, hash_func > queue_type;

      const bool	m_verbose_tracking;

      lock_type		m_lock;
      lock_type*	m_lock_ptr;

      queue_type	m_info;

      mutable MutexLock::lock_type	m_baton;

      Sync( const Sync& Source );
      Sync& operator=( const Sync& Source );
    };

    typedef Sync::lock_mode	mode_type;
    typedef Sync::lock_type	lock_type;

    static const mode_type READ = Sync::READ;
    static const mode_type WRITE = Sync::WRITE;

    ReadWriteLock( lock_type& Lock, mode_type Mode,
		   const char* Filename = "",
		   int Linenum = 0 );

    ReadWriteLock( Sync& Lock, mode_type Mode,
		   const char* Filename = "",
		   int Linenum = 0 );

    ReadWriteLock( lock_type& Lock, mode_type Mode,
		   bool TryLock,
		   const char* Filename = "",
		   int Linenum = 0 );

    ReadWriteLock( lock_type& Lock, mode_type Mode, int Timeout,
		   const char* Filename = "",
		   int Linenum = 0 );

    ReadWriteLock( Sync& Lock, mode_type Mode, int Timeout,
		   const char* Filename = "",
		   int Linenum = 0 );

    virtual ~ReadWriteLock( );

    static void Destroy( lock_type& Lock );
    static lock_type Initialize( );
    static void Initialize( lock_type& Lock );

    void Modify( mode_type Mode,
		 const char* Filename = "",
		 int Linenum = 0 );

    void Release( const char* Filename = "",
		  int Linenum = 0 );


    static int Interval( int Value );
    static int Timeout( int Value );

    //-------------------------------------------------------------------
    /// \brief Handler for thead cancelation
    ///
    /// \param[in] VLock
    ///     The VLock is a pointer to a ReadWriteLock object that holds
    ///     a lock.
    //-------------------------------------------------------------------
    static void ThreadCancel( void* VLock );

  private:
    ReadWriteLock( const ReadWriteLock& Source );
    const ReadWriteLock& operator=( const ReadWriteLock& Source );

    static void thread_cleanup( void* VLock );

    Sync	m_sync_baton_local;
    Sync*	m_sync_baton;
    mode_type	m_mode;
  }; // class ReadWriteLock


#if 0
  inline void ReadWriteLock::Sync::
  active_add( lock_mode Mode )
  {
    MutexLock	l( m_baton );

    switch( Mode )
    {
    case NONE:
      break;
      case 
    case READ:
      m_pending_reads.erase( Thread::Self::Id( ) );
      m_active_reads.insert( Thread::Self::Id( ) );
      break;
    case WRITE:
      m_pending_writes.erase( Thread::Self::Id( ) );
      m_active_writes.insert( Thread::Self::Id( ) );
      break;
    }
  }

  inline void ReadWriteLock::Sync::
  pending_add( lock_mode Mode )
  {
    MutexLock	l( m_baton );

    switch( Mode )
    {
    case NONE:
      break;
    case READ:
      m_pending_reads.insert( Thread::Self::Id( ) );
      break;
    case WRITE:
      m_pending_writes.insert( Thread::Self::Id( ) );
      break;
    }
  }

  inline void ReadWriteLock::Sync::
  pending_remove( lock_mode Mode )
  {
    MutexLock	l( m_baton );

    switch( Mode )
    {
    case NONE:
      break;
    case READ:
      m_pending_reads.erase( Thread::Self::Id( ) );
      break;
    case WRITE:
      m_pending_writes.erase( Thread::Self::Id( ) );
      break;
    }
  }
#endif /* 0 */

  //---------------------------------------------------------------------
  //---------------------------------------------------------------------
  template< ReadWriteLock::mode_type T, typename V >
  class ReadWriteLockVariable
  {
  public:
    typedef ReadWriteLock::lock_type	lock_type;
    typedef V				element_type;

    ReadWriteLockVariable( lock_type& Baton,
			   int Timeout,
			   V& Variable,
			   const char* Filename = "",
			   int Linenum = 0 );

    ReadWriteLockVariable( lock_type& Baton,
			   int Timeout,
			   const V& Variable,
			   const char* Filename = "",
			   int Linenum = 0 );

    ReadWriteLockVariable( const ReadWriteLockVariable& Source,
			   const char* Filename = "",
			   int Linenum = 0 );
			     
    const V& Var( ) const;

    V& Var( );

  private:
    ReadWriteLockVariable( );

    lock_type&			m_baton;
    int				m_timeout;
    V&				m_variable;
    mutable ReadWriteLock	m_lock;
    mutable bool		m_is_locked;
    const bool			m_is_read_only;

    inline lock_type& release( ) const
    {
      m_lock.Release( );	// Reliquish lock
      m_is_locked = false;
      return m_baton;
    }
  };

  template< ReadWriteLock::mode_type T, typename V >
  inline ReadWriteLockVariable< T, V >::
  ReadWriteLockVariable(  lock_type& Baton,
			  int Timeout,
			  V& Variable,
			  const char* Filename,
			  int Linenum )
    : m_baton( Baton ),
      m_timeout( Timeout ),
      m_variable( Variable ),
      m_lock( Baton, T, m_timeout, Filename, Linenum ),
      m_is_locked( true ),
      m_is_read_only( false )
  {
    
  }
  
  template< ReadWriteLock::mode_type T, typename V >
  inline ReadWriteLockVariable< T, V >::
  ReadWriteLockVariable(  lock_type& Baton,
			  int Timeout,
			  const V& Variable,
			  const char* Filename,
			  int Linenum )
    : m_baton( Baton ),
      m_timeout( Timeout ),
      m_variable( const_cast< V& >( Variable ) ),
      m_lock( Baton, T, m_timeout, Filename, Linenum ),
      m_is_locked( true ),
      m_is_read_only( true )
  {
    
  }
  
  template< ReadWriteLock::mode_type T, typename V >
  inline ReadWriteLockVariable< T, V >::
  ReadWriteLockVariable(  const ReadWriteLockVariable& Source,
			  const char* Filename,
			  int Linenum )
    : m_baton( Source.m_baton ),
      m_timeout( Source.m_timeout ),
      m_variable( Source.m_variable ),
      m_lock( Source.release( ), T, m_timeout,
	      Filename, Linenum ),
      m_is_locked( true ),
      m_is_read_only( Source.m_is_read_only )
  {
    
  }

  template< ReadWriteLock::mode_type T, typename V >
  inline const V& ReadWriteLockVariable< T, V >::
  Var( ) const
  {
    if ( m_is_locked == false )
    {
      throw std::logic_error( "ReadWriteLockVariable: data not protected by lock" );
    }
    return m_variable;
  }

  template< ReadWriteLock::mode_type T, typename V >
  inline V& ReadWriteLockVariable< T, V >::
  Var( )
  {
    if ( T == ReadWriteLock::READ )
    {
      assert( 0 );
      throw std::logic_error( "ReadWriteLockVariable: Cannot access variable for writing while only holding a read lock" );
    }
    if ( m_is_read_only )
    {
      assert( 0 );
      throw std::logic_error( "ReadWriteLockVariable: Cannot access constant variable for writing" );
    }
    if ( m_is_locked == false )
    {
      assert( 0 );
      throw std::logic_error( "ReadWriteLockVariable: data not protected by lock" );
    }
    return m_variable;
  }
} // namespace General

#endif	/* GENERAL__READ_WRITE_LOCK_HH */
