#include "general/config.h"

#include <sys/time.h>

#include <errno.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include <cassert>

#include <string>
#include <iostream>
#include <sstream>

#include "general/DeadLockDetector.hh"
#include "general/IOLock.hh"
#include "general/mutexlock.hh"
#include "general/ReadWriteLock.hh"

using namespace General;

#define VERBOSE_RETRY_LOGGING 1

#if DEAD_LOCK_DETECTOR_ENABLED
using General::DeadLockDetector;
#endif /* DEAD_LOCK_DETECTOR_ENABLED */

namespace
{
  static const int retry_count_max = 10;

#if ! HAVE_PTHREAD_RW_LOCK
  inline int
  pthread_rwlock_destroy( General::ReadWriteLock::lock_type* Lock )
  {
    return pthread_mutex_destroy( Lock );
  }

  inline int
  pthread_rwlock_init(  General::ReadWriteLock::lock_type* Lock,
			void* Attr )
  {
    return pthread_mutex_init( Lock,
			       static_cast< pthread_mutexattr_t* >( Attr ) );
  }

  inline int
  pthread_rwlock_unlock( General::ReadWriteLock::lock_type* Lock )
  {
    return pthread_mutex_unlock( Lock );
  }

  inline int
  pthread_rwlock_tryrdlock( General::ReadWriteLock::lock_type* Lock )
  {
    return pthread_mutex_trylock( Lock );
  }

  inline int
  pthread_rwlock_trywrlock( General::ReadWriteLock::lock_type* Lock )
  {
    return pthread_mutex_trylock( Lock );
  }

  inline int
  pthread_rwlock_rdlock( General::ReadWriteLock::lock_type* Lock )
  {
    return pthread_mutex_lock( Lock );
  }

  inline int
  pthread_rwlock_wrlock( General::ReadWriteLock::lock_type* Lock )
  {
    return pthread_mutex_lock( Lock );
  }

  inline int
  pthread_rwlock_timedrdlock( General::ReadWriteLock::lock_type* Lock,
			      struct timespec*	Timeout )
  {
    return pthread_mutex_timedlock( Lock, Timeout );
  }

  inline int
  pthread_rwlock_timedwrlock( General::ReadWriteLock::lock_type* Lock,
			      struct timespec*	Timeout )
  {
    return pthread_mutex_timedlock( Lock, Timeout );
  }

#undef   PTHREAD_RWLOCK_INITIALIZER
#define  PTHREAD_RWLOCK_INITIALIZER PTHREAD_MUTEX_INITIALIZER
#endif /* ! HAVE_PTHREAD_RW_LOCK */

  struct timespec LockAcquisitionInterval = {
    0,
    250000000,
  };

  std::string
  log ( const int Error,
	const std::string& Message,
	const std::string& Level = "ERROR" )
  {
    std::ostringstream	msg;

    msg << Level << ": ReadWriteLock: " << Message
	<< " Error Code: " << Error
	<< " system error: " << strerror( Error );

    MutexLock::lock_type*	io_baton( IOLock::GetKey( std::cerr ) );

    MutexLock::Lock( io_baton );
    std::cerr << msg.str( ) << std::endl;
    MutexLock::UnLock( io_baton );
    return msg.str( );
  }

  inline void
  normalize( timespec& Time )
  {
    static const long	ns = 1000000000;
    if ( Time.tv_nsec > ns )
    {
      Time.tv_sec += Time.tv_sec / ns;	// Add second component
      Time.tv_nsec = Time.tv_nsec % ns;	// truncate seconds
    }
  }

#if defined(_POSIX_TIMEOUTS) && (_POSIX_TIMEOUTS - 200112L) >= 0L
#if defined(_POSIX_THREADS) && (_POSIX_THREADS - 200112L) >= 0L
  inline bool
  operator<( const struct timespec& LHS,
	     const struct timespec& RHS )
  {
    if ( LHS.tv_sec < RHS.tv_sec )
    {
      return true;
    }
    else if ( LHS.tv_sec > RHS.tv_sec )
    {
      return false;
    }
    else if ( LHS.tv_sec < RHS.tv_sec )
    {
      return true;
    }
    else
    {
      return false;
    }
  }

  inline struct timespec
  operator+=( struct timespec& LHS,
	      const struct timespec& RHS )
  {
    // Add two values together
    LHS.tv_sec += RHS.tv_sec;
    LHS.tv_nsec += RHS.tv_nsec;
    normalize( LHS );
    return LHS;
  }
#endif /* defined(_POSIX_THREADS) && (_POSIX_THREADS - 200112L) >= 0L */
#endif /* defined(_POSIX_TIMEOUTS) && (_POSIX_TIMEOUTS - 200112L) >= 0L */

  inline int
  get_lock( General::ReadWriteLock::Sync::lock_type* Lock,
	    General::ReadWriteLock::Sync::lock_mode Mode,
	    bool TryLock )
  {
    int retval = 0;

    switch( Mode )
    {
    case General::ReadWriteLock::Sync::NONE:
      /// \todo
      ///    Should never be trying to set the lock to NONE;
      ///    should throw an exception.
      break;
    case General::ReadWriteLock::READ:
      retval = ( TryLock )
	? pthread_rwlock_tryrdlock( Lock )
	: pthread_rwlock_rdlock( Lock );
	break;
    case General::ReadWriteLock::WRITE:
      retval = ( TryLock )
	? pthread_rwlock_trywrlock( Lock )
	: pthread_rwlock_wrlock( Lock );
	break;
    }
    return retval;
  }

  inline int
  get_lock( General::ReadWriteLock::lock_type* Lock,
	    General::ReadWriteLock::mode_type Mode,
	    const struct timespec& RelTimeout,
	    const struct timespec& AbsStop )
  {
    int retval = 0;
#if defined(_POSIX_TIMEOUTS) && (_POSIX_TIMEOUTS - 200112L) >= 0L
    /* POSIX Timeouts are supported - option group [TMO] */
#if defined(_POSIX_THREADS) && (_POSIX_THREADS - 200112L) >= 0L
    /* POSIX threads are supported - option group [THR] */

    struct timespec	timeout;

#if HAVE_CLOCK_GETTIME
    clock_gettime( CLOCK_REALTIME, &timeout );
#else  /* HAVE_CLOCK-GETTIME */
    {
      struct timeval tv;

      gettimeofday( &tv, (void*)NULL );

      timeout.tv_sec = tv.tv_sec;
      timeout.tv_nsec = tv.tv_usec * 1000;
    }
#endif /* HAVE_CLOCK_GETTIME */

    if ( AbsStop < timeout )
    {
      throw std::runtime_error( "TIMEDOUT" );
    }
    timeout.tv_sec += RelTimeout.tv_sec;
    timeout.tv_nsec += RelTimeout.tv_nsec;
    normalize( timeout );

    switch( Mode )
    {
    case General::ReadWriteLock::Sync::NONE:
      break;
    case General::ReadWriteLock::READ:
      retval = pthread_rwlock_timedrdlock( Lock, &timeout );
      break;
    case General::ReadWriteLock::WRITE:
      retval = pthread_rwlock_timedwrlock( Lock, &timeout );
      break;
    }
#else
    switch( Mode )
    {
    case General::ReadWriteLock::Sync::NONE:
      break;
    case General::ReadWriteLock::READ:
      retval = pthread_rwlock_rdlock( Lock );
      break;
    case General::ReadWriteLock::WRITE:
      retval = pthread_rwlock_wrlock( Lock );
      break;
    }
#endif
#endif
    return retval;
  }

#if DEAD_LOCK_DETECTOR_ENABLED
  // If a pending deadlock is detected, then an exception will be thrown.
  inline void
  log_lock( General::DeadLockDetector::state_info_type& Info )
  {
    General::DeadLockDetector::SetState( Info );
  }

  inline void
  log_lock_extended( General::DeadLockDetector::state_info_type& Info,
		     General::DeadLockDetector::state_type State,
		     int Error )
  {
    Info.s_state = State;
    Info.s_error = Error;

    General::DeadLockDetector::SetState( Info );
  }

#else	/* DEAD_LOCK_DETECTOR_ENABLED */
#define log_lock( Info )
#define log_lock_extended( Info, State, Error );
#endif	/* DEAD_LOCK_DETECTOR_ENABLED */
}

namespace General
{
  //=====================================================================
  // ReadWriteLock::Sync
  //=====================================================================
  inline void ReadWriteLock::Sync::
  conflict( ) const
  {
    MutexLock	l( m_baton );

    queue_type::const_iterator	pos = m_info.find( reinterpret_cast< void* >( Thread::Self::Id( ) ) );
    assert( ( pos == m_info.end( ) )
	    || ( pos->second.m_state != ACTIVE ) );
  }

  inline ReadWriteLock::Sync::lock_mode ReadWriteLock::Sync::
  mode( ) const
  {
    MutexLock	l( m_baton );

    lock_mode retval = NONE;
    queue_type::const_iterator	pos = m_info.find( reinterpret_cast< void* >( Thread::Self::Id( ) ) );
    if ( pos != m_info.end( ) )
    {
      retval = pos->second.m_mode;
    }
    return retval;
  }

  inline void ReadWriteLock::Sync::
  set( lock_mode Mode, lock_state State )
  {
    if ( m_verbose_tracking == true )
    {
      MutexLock	l( m_baton );

      info&	i( m_info[ reinterpret_cast< void *>( Thread::Self::Id( ) ) ] );

      i.m_mode = Mode;
      i.m_state = State;
    }
  }
  
  inline void ReadWriteLock::Sync::
  unset( )
  {
    if ( m_verbose_tracking == true )
    {
      MutexLock	l( m_baton );

      m_info.erase( reinterpret_cast< void* >( Thread::Self::Id( ) ) );
    }
  }

  ReadWriteLock::Sync::lock_type ReadWriteLock::Sync::INITIALIZER
    = PTHREAD_RWLOCK_INITIALIZER;

  ReadWriteLock::Sync::
  Sync( bool Verbose )
    : m_verbose_tracking( Verbose ),
      m_lock( INITIALIZER ),
      m_lock_ptr( &m_lock ),
      m_baton( MutexLock::Initialize( ) )
  {
  }

  ReadWriteLock::Sync::
  Sync(	lock_type& Source,
	bool Verbose )
    : m_verbose_tracking( Verbose ),
      m_lock_ptr( &Source ),
      m_baton( MutexLock::Initialize( ) )
  {
  }

  ReadWriteLock::Sync::
  Sync( lock_type* Source,
	bool Verbose )
    : m_verbose_tracking( Verbose ),
      m_lock_ptr( Source ),
      m_baton( MutexLock::Initialize( ) )
  {
  }

  bool ReadWriteLock::Sync::
  HasLock( ) const
  {
    MutexLock	l( m_baton );

    bool	retval = false;

    queue_type::const_iterator	pos = m_info.find( reinterpret_cast< void* >( Thread::Self::Id( ) ) );
    if ( pos != m_info.end( ) )
    {
      retval = ( ( pos->second.m_mode != NONE )
		 && ( pos->second.m_state == ACTIVE ) );
    }
    return retval;
  }

  bool ReadWriteLock::Sync::
  IsLocked( lock_mode Mode ) const
  {
    if ( Mode == NONE )
    {
      return true;
    }
    MutexLock	l( m_baton );

    bool	retval = false;

    queue_type::const_iterator	pos = m_info.find( reinterpret_cast< void* >( Thread::Self::Id( ) ) );
    if ( pos != m_info.end( ) )
    {
      switch( pos->second.m_mode )
      {
      case READ:
	assert( Mode == READ );
	// :TRICKY: fall trough
      case WRITE:
	retval = true;
	break;
      case NONE:
	retval = false;
	break;
      }
    }
    return retval;
  }

  int ReadWriteLock::Sync::
  Lock( lock_mode Mode,
	const char* Filename,
	int Linenum )
  {
    conflict( );
    set( Mode, PENDING );

    int retry_count = 0;

    try
    {
    retry:
      int retval = get_lock( m_lock_ptr,
			     Mode,
			     false /* Try Lock */ );
      
      switch ( retval )
      {
      case 0:
	log_lock_extended( info, DeadLockDetector::ACQUIRED, retval );
	break;
      case EDEADLK:
	if ( retry_count < retry_count_max )
	{
	  struct timespec		sleep;
	  ++retry_count;
	  sleep.tv_sec = 0;
	  sleep.tv_nsec = 50000;
	  nanosleep( &sleep, &sleep );
	  goto retry;	// :TRICKY: Goto for error recovery
	}
      case EBUSY:
	log_lock_extended( info, DeadLockDetector::ACQUISITION_ERROR, retval );
	throw BusyError( );
	break;
      default:
	{
	  log_lock_extended( info, DeadLockDetector::ACQUISITION_ERROR, retval );
	  std::ostringstream msg;
	  
	  msg << log( retval, "constructor(lock,mode):" )
	      << " (" << Filename << ":" << Linenum << ")"
	    ;
	  assert( 0 );
	  throw std::runtime_error( msg.str( ) );
	}
	break;
      }
      set( Mode, ACTIVE );
      return retval;
    }
    catch( ... )
    {
      //-----------------------------------------------------------------
      // Take ourselves out of the pending queue
      //-----------------------------------------------------------------
      unset( );
      throw;	// rethrow the error.
    }
  }

  int ReadWriteLock::Sync::
  Lock( mode_type Mode,
	bool TryLock,
	const char* Filename,
	int Linenum )
  {
    conflict( );
    set( Mode, PENDING );

    int retval = 0;

    try
    {
#if DEAD_LOCK_DETECTOR_ENABLED
      DeadLockDetector::state_info_type
	info( m_lock,
	      ( Mode == READ )
	      ? DeadLockDetector::RW_READ
	      : DeadLockDetector::RW_WRITE,
	      DeadLockDetector::PENDING,
	      Filename,
	      Linenum );
#endif /* DEAD_LOCK_DETECTOR_ENABLED */

      log_lock( info );

      int retry_count = 0;

    retry:
      retval = get_lock( m_lock_ptr, Mode, TryLock );

      switch ( retval )
      {
      case 0:
	log_lock_extended( info, DeadLockDetector::ACQUIRED, retval );
	break;
      case EDEADLK:
	if ( retry_count < retry_count_max )
	{
	  log_lock_extended( info, DeadLockDetector::ACQUISITION_ERROR, retval );
	  struct timespec		sleep;
	  ++retry_count;
	  sleep.tv_sec = 0;
	  sleep.tv_nsec = 50000;
	  nanosleep( &sleep, &sleep );
	  goto retry;	// :TRICKY: Goto for error recovery
	}
      case EBUSY:
	log_lock_extended( info, DeadLockDetector::ACQUISITION_ERROR, retval );
	throw BusyError( );
	break;
      default:
	{
	  log_lock_extended( info, DeadLockDetector::ACQUISITION_ERROR, retval );
	  std::ostringstream msg;

	  msg << log( retval, "constructor(lock,mode,trylock):" )
	      << " (" << Filename << ":" << Linenum << ")"
	    ;
	  throw std::runtime_error( msg.str( ) );
	}
	break;
      }
    }
    catch( ... )
    {
      unset( );
      throw;
    }
    set( Mode, ACTIVE );
    return retval;
  }

  int ReadWriteLock::Sync::
  Lock( mode_type Mode,
	int TimeoutMS,
	const char* Filename,
	int Linenum )
  {
    conflict( );
    set( Mode, PENDING );
#if DEAD_LOCK_DETECTOR_ENABLED
    DeadLockDetector::state_info_type
      info( m_lock,
	    ( Mode == READ )
	    ? DeadLockDetector::RW_READ
	    : DeadLockDetector::RW_WRITE,
	    DeadLockDetector::PENDING,
	    Filename,
	    Linenum );
#endif /* DEAD_LOCK_DETECTOR_ENABLED */

    int retval = 0;

    if ( TimeoutMS <= 0 )
    {
      log_lock( info );
      retval = get_lock( m_lock_ptr, Mode, false /* TryLock */ );
    }
    else
    {
      //-------------------------------------------------------------------
      // Non-blocking mode
      //-------------------------------------------------------------------
      struct timespec	stop;
#if VERBOSE_RETRY_LOGGING
      int			iteration = 0;
#endif /* VERBOSE_RETRY_LOGGING */

#if HAVE_CLOCK_GETTIME
    clock_gettime( CLOCK_REALTIME, &stop );
#else  /* HAVE_CLOCK-GETTIME */
    {
      struct timeval tv;

      gettimeofday( &tv, (void*)NULL );

      stop.tv_sec = tv.tv_sec;
      stop.tv_nsec = tv.tv_usec * 1000;
    }
#endif /* HAVE_CLOCK_GETTIME */
    stop.tv_sec += TimeoutMS / 1000;
    stop.tv_nsec += ( TimeoutMS % 1000 ) * 1000000;
    normalize( stop );

#if VERBOSE_RETRY_LOGGING
      MutexLock::lock_type*	io_baton( IOLock::GetKey( std::cerr ) );
#endif /* VERBOSE_RETRY_LOGGING */

      while ( true )
      {
	try
	{
	  log_lock_extended( info, DeadLockDetector::PENDING, 0 );
	  retval = get_lock( m_lock_ptr, Mode, LockAcquisitionInterval, stop );
	  if ( retval == 0 )
	  {
#if VERBOSE_RETRY_LOGGING
	    if ( iteration )
	    {
	      MutexLock::Lock( io_baton );
	      std::cerr << "Succeeded in acquiring lock after " << iteration
			<< " tries"
			<< std::endl;
	      MutexLock::UnLock( io_baton );
	    }
#endif /* VERBOSE_RETRY_LOGGING */
	    break;
	  }
	  else
	  {
	    log_lock_extended( info,
			       DeadLockDetector::ACQUISITION_ERROR, retval );
	  }
	  iteration++;
	}
	catch( ... )
	{
#if VERBOSE_RETRY_LOGGING
	  if ( iteration )
	  {
	    MutexLock::Lock( io_baton );
	    std::cerr << "Failed to acquire lock after " << iteration
		      << " tries"
		      << std::endl;
	    MutexLock::UnLock( io_baton );
	  }
#endif /* VERBOSE_RETRY_LOGGING */
	  retval = ETIMEDOUT;
	  break;
	}
      }
    }
    try
    {
      switch ( retval )
      {
      case 0:
	log_lock_extended( info, DeadLockDetector::ACQUIRED, retval );
	break;
      case EBUSY:
	log( retval, "constructor(lock,mode,timeout):" );
	log_lock_extended( info, DeadLockDetector::ACQUISITION_ERROR, retval );
	throw BusyError( );
	break;
      case ETIMEDOUT:
	log( retval, "constructor(lock,mode,timeout): TIMEDOUT" );
	log_lock_extended( info, DeadLockDetector::ACQUISITION_ERROR, retval );
	throw TimedoutError( );
	break;
      default:
	{
	  log_lock_extended( info, DeadLockDetector::ACQUISITION_ERROR, retval );
	  std::ostringstream msg;

	  msg << log( retval, "constructor(lock,mode,timeout):" )
	      << " (" << Filename << ":" << Linenum << ")"
	    ;
	  throw std::runtime_error( msg.str( ) );
	}
	break;
      }
    }
    catch( ... )
    {
      unset( );
      throw;
    }
    set( Mode, ACTIVE );
    return retval;
  }

  int ReadWriteLock::Sync::
  Modify( lock_mode Mode,
	  const char* Filename,
	  int Linenum )
  {
    int retval = 0;

    if ( ( Mode != mode( ) )
	 && ( m_lock_ptr != (lock_type*)NULL ) )
    {
      {
	retval = pthread_rwlock_unlock( m_lock_ptr );
	set( Mode, PENDING );
#if DEAD_LOCK_DETECTOR_ENABLED
	DeadLockDetector::state_info_type
	  info( m_lock,
		( m_mode == READ )
		? DeadLockDetector::RW_READ
		: DeadLockDetector::RW_WRITE,
		DeadLockDetector::RELEASED,
		__FILE__,
		__LINE__ );
#endif /* DEAD_LOCK_DETECTOR_ENABLED */

	if ( retval != 0 )
	{
	  log_lock_extended( info, DeadLockDetector::ACQUISITION_ERROR, retval );
	  log( retval, "Modify: release:" );
	  return retval;
	}
	log_lock( info );
      }
      {
#if DEAD_LOCK_DETECTOR_ENABLED
	DeadLockDetector::state_info_type
	  info( m_lock,
		( Mode == READ )
		? DeadLockDetector::RW_READ
		: DeadLockDetector::RW_WRITE,
		DeadLockDetector::PENDING,
		Filename,
		Linenum );
#endif /* DEAD_LOCK_DETECTOR_ENABLED */

	log_lock( info );

	retval = get_lock( m_lock_ptr, Mode, false /* TryLock */ );

	if ( retval != 0 )
	{
	  log_lock_extended( info, DeadLockDetector::ACQUISITION_ERROR, retval );
	  log( retval, "Modify: lock:" );
	  unset( );
	  return retval;
	}
	set( Mode, ACTIVE );
	log_lock( info );
      }
    }
    return retval;
  }

  int ReadWriteLock::Sync::
  Unlock( const char* Filename,
	  int Linenum )
  {
    int retval = pthread_rwlock_unlock( m_lock_ptr );
    
    if ( retval == 0 )
    {
      unset( );
    }
    return retval;
  }

}

ReadWriteLock::BusyError::
BusyError( )
  : std::runtime_error( msg( ) )
{
}

std::string ReadWriteLock::BusyError::
msg( )
{
  char	buffer[ 1024];
  strerror_r( EBUSY, buffer, sizeof( buffer ) );
  return std::string( buffer );
}

ReadWriteLock::TimedoutError::
TimedoutError( )
  : std::runtime_error( msg( ) )
{
}

std::string ReadWriteLock::TimedoutError::
msg( )
{
  char	buffer[ 1024];
  strerror_r( ETIMEDOUT, buffer, sizeof( buffer ) );
  return std::string( buffer );
}

ReadWriteLock::
ReadWriteLock( lock_type& Lock, mode_type Mode,
	       const char* Filename,
	       int Linenum )
  : m_sync_baton_local( Lock ),
    m_sync_baton( &m_sync_baton_local ),
    m_mode( Mode )
{
  m_sync_baton->Lock( Mode, Filename, Linenum );
}

ReadWriteLock::
ReadWriteLock( Sync& Baton,
	       mode_type Mode,
	       const char* Filename,
	       int Linenum )
  : m_sync_baton( &Baton ),
    m_mode( Mode )
{
  Baton.Lock( Mode, Filename, Linenum );
}

ReadWriteLock::
ReadWriteLock( lock_type& Lock,
	       mode_type Mode,
	       bool TryLock,
	       const char* Filename,
	       int Linenum )
  : m_sync_baton_local( Lock ),
    m_sync_baton( &m_sync_baton_local ),
    m_mode( Mode )
{
  m_sync_baton->Lock( Mode, TryLock, Filename, Linenum );
}

ReadWriteLock::
ReadWriteLock( lock_type& Lock, mode_type Mode, int TimeoutMS,
	       const char* Filename,
	       int Linenum )
  : m_sync_baton_local( Lock ),
    m_sync_baton( &m_sync_baton_local ),
    m_mode( Mode )
{
  m_sync_baton->Lock( Mode, TimeoutMS, Filename, Linenum );
}

ReadWriteLock::
ReadWriteLock( Sync& Baton,
	       mode_type Mode,
	       int TimeoutMS,
	       const char* Filename,
	       int Linenum )
  : m_sync_baton( &Baton ),
    m_mode( Mode )
{
  m_sync_baton->Lock( Mode, TimeoutMS, Filename, Linenum );
}

ReadWriteLock::
~ReadWriteLock( )
{
  Release( __FILE__, __LINE__ );
}


void General::ReadWriteLock::
Destroy( lock_type& Lock )
{
  int retval = 0;
  retval = pthread_rwlock_destroy( &Lock );
  if ( retval != 0 )
  {
    log( retval, "destroy:" );
  }
}

General::ReadWriteLock::lock_type General::ReadWriteLock::
Initialize( )
{
  return ReadWriteLock::Sync::INITIALIZER;
}

void General::ReadWriteLock::
Initialize( lock_type& Lock )
{
  int retval = 0;
  retval = pthread_rwlock_init( &Lock,
				static_cast< pthread_rwlockattr_t* >( NULL ) );
  if ( retval != 0 )
  {
    log( retval, "initialize:" );
  }
}

void ReadWriteLock::
Modify( mode_type Mode,
	const char* Filename,
	int Linenum )
{
#if old
  int retval = 0;

  if ( ( Mode != m_mode )
       && ( m_lock != (lock_type*)NULL ) )
  {
    {
      retval = pthread_rwlock_unlock( m_lock );

#if DEAD_LOCK_DETECTOR_ENABLED
      DeadLockDetector::state_info_type
	info( m_lock,
	      ( m_mode == READ )
	      ? DeadLockDetector::RW_READ
	      : DeadLockDetector::RW_WRITE,
	      DeadLockDetector::RELEASED,
	      __FILE__,
	      __LINE__ );
#endif /* DEAD_LOCK_DETECTOR_ENABLED */

      if ( retval != 0 )
      {
	log_lock_extended( info, DeadLockDetector::ACQUISITION_ERROR, retval );
	log( retval, "Modify: release:" );
	return;
      }
      log_lock( info );
    }
    {
#if DEAD_LOCK_DETECTOR_ENABLED
      DeadLockDetector::state_info_type
	info( m_lock,
	      ( Mode == READ )
	      ? DeadLockDetector::RW_READ
	      : DeadLockDetector::RW_WRITE,
	      DeadLockDetector::PENDING,
	      Filename,
	      Linenum );
#endif /* DEAD_LOCK_DETECTOR_ENABLED */

      log_lock( info );
      m_mode = Mode;

      retval = get_lock( m_lock, m_mode, false /* TryLock */ );

      if ( retval != 0 )
      {
	log_lock_extended( info, DeadLockDetector::ACQUISITION_ERROR, retval );
	log( retval, "Modify: lock:" );
	return;
      }
      log_lock( info );
    }
  }
#else /* old */
  m_sync_baton->Modify( Mode, Filename, Linenum );
#endif /* old */
}

void
ReadWriteLock::
Release( const char* Filename, int Linenum )
{
  int retval = 0;

  if ( m_sync_baton != (Sync*)NULL )
  {
    retval = m_sync_baton->Unlock( Filename, Linenum );
  }

#if DEAD_LOCK_DETECTOR_ENABLED
  DeadLockDetector::state_info_type
    info( m_lock,
	  ( m_mode == READ )
	  ? DeadLockDetector::RW_READ
	  : DeadLockDetector::RW_WRITE,
	  DeadLockDetector::RELEASED,
	  __FILE__,
	  __LINE__ );
#endif /* DEAD_LOCK_DETECTOR_ENABLED */

  if ( retval != 0 )
  {
    log_lock_extended( info, DeadLockDetector::ACQUISITION_ERROR, retval );
    log( retval, "release:" );
  }
  else
  {
    log_lock( info );
    m_sync_baton = (Sync*)NULL;
  }
}

int ReadWriteLock::
Interval( int Value )
{
  int retval;
 
  retval = LockAcquisitionInterval.tv_sec * 1000;
  retval += LockAcquisitionInterval.tv_nsec / 1000000;

  LockAcquisitionInterval.tv_sec = Value / 1000;
  LockAcquisitionInterval.tv_nsec = ( Value % 1000 ) * 1000000;

  return retval;
}

//-----------------------------------------------------------------------
/// This routine is used when a thread is canceled and the lock needs to
/// be release.
/// It is currently only written to support pthread_cancel_push().
//-----------------------------------------------------------------------
void ReadWriteLock::
ThreadCancel( void* VLock )
{
  if ( VLock )
  {
    ReadWriteLock*	l ( reinterpret_cast< ReadWriteLock*>( VLock ) );

    l->Release( __FILE__, __LINE__ );
  }
}

