/* -*- mode: c++; c-basic-offset: 2; -*- */

#include <time.h>
#ifndef CLOCK_REALTIME
#include <sys/time.h>
#endif

#include <iostream>
#include <stdexcept>

#include "general/ConditionalVariable.hh"
#include "general/System.hh"

namespace General
{
  ConditionalVariable::
  ConditionalVariable( )
  {
    pthread_cond_init( &m_var, NULL );
    MutexLock::Initialize( m_lock );
  }

  ConditionalVariable::
  ~ConditionalVariable( )
  {
    MutexLock::Destroy( m_lock );
    pthread_cond_destroy( &m_var );
  }

  void ConditionalVariable::
  Broadcast( )
  {
    pthread_cond_broadcast( &m_var );
  }

  void ConditionalVariable::
  Release( )
  {
    MutexLock::UnLock( Mutex( ) );
  }

  void ConditionalVariable::
  Signal( )
  {
    pthread_cond_signal( &m_var );
  }

  bool ConditionalVariable::
  TimedWait( INT_4U Seconds )
  {
    struct timespec	ts;

#if defined( CLOCK_REALTIME )
    clock_gettime( CLOCK_REALTIME, &ts );
#else
    timeval	tv;

    gettimeofday( &tv, NULL );

    ts.tv_sec = tv.tv_sec;
    ts.tv_nsec = 1000 * tv.tv_usec;
#endif /* defined( CLOCK_REALTIME ) */
    ts.tv_sec += Seconds;
    int rc = pthread_cond_timedwait( &m_var, &m_lock, &ts );
    switch ( rc )
    {
    case ETIMEDOUT:
      return false;
    case 0:
      return true;
    default:
      {
	std::ostringstream	msg;

	msg << "ConditionalVariable::TimedWait: system error: "
	    << General::System::ErrnoMessage( )
	  ;
	std::cerr << "DEBUG: " << msg.str( )
		  << std::endl
	  ;
	throw std::runtime_error( msg.str( ) );
      }
    }
    
  }

  void ConditionalVariable::
  Wait( )
  {
    pthread_cond_wait( &m_var, &m_lock );
  }

} // namespace - General
