#ifndef GENERAL__AT_EXIT_HH
#define GENERAL__AT_EXIT_HH

#include <list>

#include "general/Singleton.hh"

namespace General
{
  ///--------------------------------------------------------------------
  /// \brief  Keep track of cleanup routines at normal program termination
  ///--------------------------------------------------------------------
  class AtExit
  {
    SINGLETON_TS_DECL( AtExit );

  public:
    ///------------------------------------------------------------------
    /// \brief  Type specifier for exit functions.
    ///------------------------------------------------------------------
    typedef void (*ExitFunction)( );

    ///------------------------------------------------------------------
    /// \brief  Add a cleanup function to call when program is exiting.
    ///
    /// \param[in] Function
    ///     The function to be called when exiting
    ///
    /// \param[in] Name
    ///     A descriptive name of the action being taken
    ///
    /// \param[in] Ring
    ///     The level at which to call.
    ///     The ring structure is inverted in that rings of higher value
    ///     are called prior to rings of lesser values.
    ///------------------------------------------------------------------
    static void Append( ExitFunction Function,
			const std::string& Name,
			int Ring );

    static void Cleanup( );

    static void Init( );

    static bool IsExiting( );

  private:
    ///------------------------------------------------------------------
    /// \brief Stack to keep track of exit functions to call.
    ///
    /// Cannot trust any of the stl objects to be available once exit()
    /// has been called. This structure keeps track of exit routines
    /// that need to be called.
    ///------------------------------------------------------------------
    struct exit_function_node {
      ///----------------------------------------------------------------
      /// Function to be called
      ///----------------------------------------------------------------
      ExitFunction		m_function;

      char*			m_name;

      ///----------------------------------------------------------------
      /// Link to the previous node in the list. If the value of this
      /// object is NULL, then the object is the bottom of the stack.
      ///----------------------------------------------------------------
      exit_function_node*	m_next;

      ///----------------------------------------------------------------
      /// \brief Constructor
      ///
      /// The constructor creates a node which holds the function
      /// to be called and a link to the previous node.
      ///
      /// \param Function The function to called to release resources.
      /// \param Name The name associated with the exit function.
      /// \param Next The address of the next node, NULL otherwise.
      ///----------------------------------------------------------------
      exit_function_node( ExitFunction Function,
			  const std::string& Name,
			  exit_function_node* Next );
    };
    ///------------------------------------------------------------------
    /// \brief Stack to keep track of exit rings.
    ///
    ///------------------------------------------------------------------
    struct exit_ring_node {
      int 			m_ring;
      exit_ring_node*		m_next;
      exit_function_node*	m_exit_function_list;

      ///----------------------------------------------------------------
      /// \brief Constructor
      ///
      /// The constructor creates a node which holds the function
      /// to be called and a link to the previous node.
      ///
      /// \param[in] Ring
      ///     The numeric ring number of this ring.
      ///
      /// \param[in] Next
      ///     The address of the next node, NULL otherwise.
      ///----------------------------------------------------------------
      exit_ring_node( int Ring,
		      exit_ring_node* Next )
	: m_ring( Ring ),
	  m_next( Next ),
	  m_exit_function_list( (exit_function_node*)NULL )
      {
      }
    };

    static bool				m_is_exiting;

    exit_ring_node*			m_head;
    mutable MutexLock::lock_type	m_is_exiting_baton;

    //-------------------------------------------------------------------
    /// \brief  Destructor
    //-------------------------------------------------------------------
    ~AtExit( );

    //-------------------------------------------------------------------
    /// \brief Add one function to call when exiting.
    ///
    /// \param[in] Function
    ///     The function to add to the list of functions to be
    ///     called when the program terminates.
    ///
    /// \param[in] Name
    ///     A descriptive name of the action being taken
    ///
    /// \param[in] Ring
    ///     The level at which to call.
    ///     The ring structure is inverted in that rings of higher value
    ///     are called prior to rings of lesser values.
    //---------------------------------------------------------------------
    void append( ExitFunction Function,
		 const std::string& Name,
		 int Ring );

    //---------------------------------------------------------------------
    /// \brief Cleanup resources that have accumulated
    //---------------------------------------------------------------------
    void cleanup( );

    //---------------------------------------------------------------------
    /// \brief Return if the system is currently exiting.
    //---------------------------------------------------------------------
    bool is_exiting( ) const;

    //-------------------------------------------------------------------
    // Do not allow copy constructor
    //-------------------------------------------------------------------
    const AtExit& operator=( const General::AtExit& );
  };

  inline void AtExit::
  Append( ExitFunction Function,
	  const std::string& Name,
	  int Ring )
  {
    SSelf( ).append( Function, Name, Ring );
  }

  inline void AtExit::
  Cleanup( )
  {
    singleton_suicide( );
  }

  inline void AtExit::
  Init( )
  {
    //-------------------------------------------------------------------
    // Want the side effect of creating space
    //-------------------------------------------------------------------
    SSelf( );
  }

  ///--------------------------------------------------------------------
  /// Return true if called while exiting, false otherwise.
  ///--------------------------------------------------------------------
  inline bool AtExit::
  IsExiting( )
  {
    try
    {
      return SSelf( false ).is_exiting( );
    }
    catch( ... )
    {
    }
    return xxyyzz_singleton_created;
  }

  inline bool AtExit::
  is_exiting( ) const
  {
    MutexLock	l( m_is_exiting_baton );

    return m_is_exiting;
  }
} // namespace - General

#endif /* GENERAL__AT_EXIT_HH */
