#include "general/AtExit.hh"
#include "general/Exec.hh"
#include "general/unittest.h"

General::UnitTest	Test;
using			General::Exec;

static const char*	CMD_FRAMECPP_VERIFY = "./framecpp_verify";

void
test_pr3430( )
{
  const char* test_name = "PR3430";
  try
  {
    Exec	cmd( CMD_FRAMECPP_VERIFY,
		     "--verbose",
		     Exec::COMMAND_END );

    cmd.Spawn( );

    //-------------------------------------------------------------------
    // Look at what was written to standard out.
    //-------------------------------------------------------------------
  }
  catch( ... )
  {
    Test.Check( false )
      << test_name << ": failed because of exception being thrown"
      << std::endl
      ;
  }
}

void
test_pr3437( )
{
  //---------------------------------------------------------------------
  // Testing of PR 3437 - framecpp_verify fails on E1 data
  //---------------------------------------------------------------------
  const char* TEST_NAME = "PR3437";
  const char* TEST_GWF_FILENAME
    = "/archive/frames/E1/L0/LHO/H-R-63891/H-R-638911999-1.gwf";

  try
  {
    Exec	cmd( CMD_FRAMECPP_VERIFY,
		     "--verbose",
		     "--force",
		     "--no-metadata-check",
		     "--relax-eof-checksum",
		     TEST_GWF_FILENAME,
		     Exec::COMMAND_END );

    cmd.Spawn( );

    Test.Check( cmd.IfExited( ) && ( cmd.ExitCode( ) == 0 ) )
      << TEST_NAME << ": framecpp_verify reads E1 data"
      << " ["
      << " exit code: " << cmd.ExitCode( )
      << " signal: " << cmd.Signal( )
      << " core dump: " << cmd.CoreDump( )
      << " ]"
      << std::endl
      ;
  }
  catch( ... )
  {
    Test.Check( false )
      << TEST_NAME << ": failed because of exception being thrown"
      << std::endl
      ;
  }
}

void
test_framecpp_dump_objects( )
{
  static const char* cmd = "framecpp_dump_objects";

  {
    Exec	e( cmd,
		   "--silent-data",
		   "--verbose",
		   cmd,
		   Exec::COMMAND_END );

    e.Spawn( );
    Test.Check( e.IfExited( ) && ( e.ExitCode( ) != 0 ) )
      << "PR3439 - processing non-frame files"
      << " ["
      << " exit code: " << e.ExitCode( )
      << " signal: " << e.Signal( )
      << " core dump: " << e.CoreDump( )
      << " ]"
      << std::endl
      ;
  }
}

void
test_time_trial( )
{
  //---------------------------------------------------------------------
  // Verify that programs complete in a timely mannor.
  //---------------------------------------------------------------------
  const char*	gwf = "Z-R_std_test_frame_ver8-600000000-1.gwf";
#define V 0
#define VC 0
#define V2 1
#if V
  Exec		verify( "./framecpp_verify",
			"--checksum-file-only",
			gwf,
			Exec::COMMAND_END );
#endif /* 0 */
#if VC
  Exec		verify_checksum( "./framecpp_verify_checksums",
				 "--no-checksum-structure",
				 "--no-checksum-frame",
				 gwf,
				 Exec::COMMAND_END );
#endif /* 0 */
#if V2
  Exec		verify2( "./framecpp_verify",
			 "--checksum-file-only",
			 gwf,
			 Exec::COMMAND_END );
#endif /* 0 */

#if V
  verify.Spawn( );
#endif /* 0 */
#if VC
  verify_checksum.Spawn( );
#endif /* 0 */
#if V2
  verify2.Spawn( );
#endif /* 0 */
  
  Test.Message( )
#if V
    << verify.Command( )
#endif /* 0 */
#if VC
    << " vs " << verify_checksum.Command( )
#endif /* 0 */
#if V
    << " verify: " << verify.TimeCPU( ) << " seconds"
#endif /* 0 */
#if VC
    << " verify_checksum: " << verify_checksum.TimeCPU( ) << " seconds"
#endif /* 0 */
#if V2
    << " verify2: " << verify2.TimeCPU( ) << " seconds"
#endif /* 0 */
    << std::endl
    ;
}

int
main( int ArgC, char** ArgV )
{
  //---------------------------------------------------------------------
  // Initialize the testing framework
  //---------------------------------------------------------------------
  Test.Init( ArgC, ArgV );

  try
  {
    //-------------------------------------------------------------------
    // Time to run some tests
    //-------------------------------------------------------------------
    test_framecpp_dump_objects( );
#if 1
    test_time_trial( );
#endif /* 0 */
    //-------------------------------------------------------------------
    // Test specific problem reports
    //-------------------------------------------------------------------
    test_pr3430( );
#if WORKING
    test_pr3437( );
#endif /* WORKING */
  }
  catch ( const std::exception& E )
  {
    Test.Check( false ) << "Caught an exception: " << E.what( ) << std::endl;
  }
  catch ( ... )
  {
    Test.Check( false ) << "Caught an unknown exception" << std::endl;
  }
  try
  {
    //-------------------------------------------------------------------
    // Call cleanup code to ensure all of that is also working
    //-------------------------------------------------------------------
    General::AtExit::Cleanup( );
  }
  catch( ... )
  {
  }

  //---------------------------------------------------------------------
  // Testing is complete. Report back to the user
  //---------------------------------------------------------------------
  Test.Exit();
}
