#if HAVE_CONFIG_H
#include "framecpp/config.h"
#endif /* HAVE_CONFIG_H */

#include <algorithm>
#include <memory>
#include <sstream>

#include "general/autoarray.hh"
#include "general/unittest.h"

#include "framecpp/Dimension.hh"
#include "framecpp/FrAdcData.hh"
#include "framecpp/FrVect.hh"

#include "FrStruct.hh"
#include "FrStruct3.tcc"
#include "FrStruct4.tcc"
#include "FrStruct6.tcc"
#include "FrStruct7.tcc"
#include "FrStruct8.tcc"

typedef FrameCPP::Common::FrameSpec::Info::frame_object_types class_id_type;

General::UnitTest	Test;

using namespace FrameCPP;
using FrameCPP::Common::FrameSpec;

static const INT_2U SAMPLES = 256;

static void
verify_downconvert( std::ostringstream& Leader,
		    const int Version,
		    frame_object_type FrameObj,
		    const std::string& FrameObjName,
		    General::UnitTest& Test );

static void
verify_upconvert( std::ostringstream& Leader,
		  const int Version,
		  frame_object_type FrameObj,
		  const std::string& FrameObjName,
		  General::UnitTest& Test );

static inline void
verify_convert( std::ostringstream& Leader,
		const int Version,
		frame_object_type FrameObj,
		const std::string& FrameObjName,
		General::UnitTest& Test )
{
  std::ostringstream	leader;
  leader.str( "" );
  leader << Leader.str( );
  verify_downconvert( leader, Version, FrameObj, FrameObjName, Test );
  leader.str( "" );
  leader << Leader.str( );
  verify_upconvert( leader, Version, FrameObj, FrameObjName, Test );
}

namespace
{
  template <class T>
  class Riser
  {
  public:
    Riser( )
      : m_counter( 0 )
    {
    }

    T operator()( )
    {
      return m_counter++;
    }

  private:
    T	m_counter;
  };
}

void
fr_adc_data( )
{
  //---------------------------------------------------------------------
  // Create a Single Dimensional Vector
  //---------------------------------------------------------------------
  General::AutoArray< INT_2U >	data_2u( new INT_2U[ SAMPLES ] );
  Dimension				dim( SAMPLES );
  Riser< INT_2U >			riser;	

  std::generate( &(data_2u[0]), &(data_2u[ SAMPLES ]), riser );
  FrVect::data_type
    data( reinterpret_cast< FrVect::data_type::element_type * >( data_2u.release( ) ) );
  General::SharedPtr< FrVect >
    v( new FrVect( "fr_vect",
		   FrVect::RAW,			/* Compress */
		   FrVect::FR_VECT_2U,		/* Type */
		   1,				/* nDim */
		   &dim,			/* dims */
		   SAMPLES,			/* NData */
		   SAMPLES * sizeof( INT_2U ),	/* NBytes */
		   data,			/* Data */
		   "unitY"			/* unitY */
		   ) );

  //---------------------------------------------------------------------
  // Create the adc
  //---------------------------------------------------------------------
  std::unique_ptr< FrAdcData >
    a( new FrAdcData( "adc_data", INT_4U( 0 ), INT_4U( 0 ), INT_4U( 0 ),
		      REAL_8( SAMPLES ),
		      FrameCPP::Version::FrAdcData::DEFAULT_BIAS,
		      FrameCPP::Version::FrAdcData::DEFAULT_SLOPE,
		      FrameCPP::Version::FrAdcData::DEFAULT_UNITS( ),
		      FrameCPP::Version::FrAdcData::DEFAULT_FSHIFT,
		      FrameCPP::Version::FrAdcData::DEFAULT_TIME_OFFSET,
		      FrameCPP::Version::FrAdcData::DEFAULT_DATA_VALID,
		      FrameCPP::Version::FrAdcData::DEFAULT_PHASE ) );
  a->RefData( ).append( v );

  //---------------------------------------------------------------------
  // Check the creation of sub FrAdcData
  //---------------------------------------------------------------------
  FrAdcData::subset_ret_type	sub_adc( a->Subset( 0, 1 ) );
  Test.Check( *a == *sub_adc )
    << "Creation of subadc of equal length"
    << std::endl;
}

void
fr_vect( )
{
  //---------------------------------------------------------------------
  // Create a Single Dimensional Vector
  //---------------------------------------------------------------------
  General::AutoArray< INT_2U >	data_2u( new INT_2U[ SAMPLES ] );
  Dimension			dim( SAMPLES );
  Riser< INT_2U >		riser;	

  std::generate( &(data_2u[0]), &(data_2u[ SAMPLES ]), riser );
  FrVect::data_type
    data( reinterpret_cast< FrVect::data_type::element_type * >( data_2u.release( ) ) );
  General::SharedPtr< FrVect >
    v( new FrVect( "fr_vect",
		   FrVect::RAW,			/* Compress */
		   FrVect::FR_VECT_2U,		/* Type */
		   1,				/* nDim */
		   &dim,			/* dims */
		   SAMPLES,			/* NData */
		   SAMPLES * sizeof( INT_2U ),	/* NBytes */
		   data,			/* Data */
		   "unitY"			/* unitY */
		   ) );
  //---------------------------------------------------------------------
  // Check the creation of sub vectors
  //---------------------------------------------------------------------
  FrVect::subfrvect_type sub_vect( v->SubFrVect( 0, SAMPLES ) );
  Test.Check( ( *v == *sub_vect ) )
    << "Creation of subvector of equal length"
    << std::endl;

}

int
main(int argc, char* argv[])
try
{
  //---------------------------------------------------------------------
  // Prepare for testing
  //---------------------------------------------------------------------
  Test.Init( argc, argv );

  //---------------------------------------------------------------------
  // Testing of basic data types
  //---------------------------------------------------------------------
  fr_vect( );		// FrVect
  fr_adc_data( );	// FrAdcData

  //---------------------------------------------------------------------
  // Test upconverting of frame structures.
  //---------------------------------------------------------------------
  for( int x = FRAME_SPEC_MIN;
       x <=  FRAME_SPEC_MAX;
       ++x )
  {
    if ( x == 5 )
    {
      //-----------------------------------------------------------------
      // These versions are not supported.
      //-----------------------------------------------------------------
      continue;
    }
    //-------------------------------------------------------------------
    //
    //-------------------------------------------------------------------

    std::ostringstream		leader;
    mk_frame_object_ret_type	obj;

    //-------------------------------------------------------------------
    // FrameH
    //-------------------------------------------------------------------
    leader.str( "frame_h:" );
    obj = mk_frame_object( x, FrameSpec::Info::FSI_FRAME_H );
    verify_convert( leader, x, obj, "FrameH", Test );
    //-------------------------------------------------------------------
    // FrAdcData
    //-------------------------------------------------------------------
    leader.str( "fr_adc_data:" );
    obj = mk_frame_object( x, FrameSpec::Info::FSI_FR_ADC_DATA );
    verify_convert( leader, x, obj, "FrAdcData", Test );
    //-------------------------------------------------------------------
    // FrDetector
    //-------------------------------------------------------------------
    leader.str( "fr_detector:" );
    obj = mk_frame_object( x, FrameSpec::Info::FSI_FR_DETECTOR );
    verify_convert( leader, x, obj, "FrDetector", Test );
    //-------------------------------------------------------------------
    // FrEvent
    //-------------------------------------------------------------------
    leader.str( "fr_event:" );
    obj = mk_frame_object( x, FrameSpec::Info::FSI_FR_EVENT );
    verify_convert( leader, x, obj, "FrEvent;", Test );
    //-------------------------------------------------------------------
    // FrHistory
    //-------------------------------------------------------------------
    leader.str( "fr_history:" );
    obj = mk_frame_object( x, FrameSpec::Info::FSI_FR_HISTORY );
    verify_convert( leader, x, obj, "FrHistory", Test );
    //-------------------------------------------------------------------
    // FrMsg
    //-------------------------------------------------------------------
    leader.str( "fr_msg:" );
    obj = mk_frame_object( x, FrameSpec::Info::FSI_FR_MSG );
    verify_convert( leader, x, obj, "FrMsg", Test );
    //-------------------------------------------------------------------
    // FrProcData
    //-------------------------------------------------------------------
    leader.str( "fr_proc_data:" );
    obj = mk_frame_object( x, FrameSpec::Info::FSI_FR_PROC_DATA );
    verify_convert( leader, x, obj, "FrProcData", Test );
    //-------------------------------------------------------------------
    // FrRawData
    //-------------------------------------------------------------------
    leader.str( "fr_raw_data:" );
    obj = mk_frame_object( x, FrameSpec::Info::FSI_FR_RAW_DATA );
    verify_convert( leader, x, obj, "FrRawData", Test );
    //-------------------------------------------------------------------
    // FrSerData
    //-------------------------------------------------------------------
    leader.str( "fr_ser_data:" );
    obj = mk_frame_object( x, FrameSpec::Info::FSI_FR_SER_DATA );
    verify_convert( leader, x, obj, "FrSerData", Test );
    //-------------------------------------------------------------------
    // FrSimData
    //-------------------------------------------------------------------
    leader.str( "fr_sim_data:" );
    obj = mk_frame_object( x, FrameSpec::Info::FSI_FR_SIM_DATA );
    verify_convert( leader, x, obj, "FrSimData", Test );
    //-------------------------------------------------------------------
    // FrSimEvent
    //-------------------------------------------------------------------
    leader.str( "fr_sim_event:" );
    obj = mk_frame_object( x, FrameSpec::Info::FSI_FR_SIM_EVENT );
    verify_convert( leader, x, obj, "FrSimEvent;", Test );
    //-------------------------------------------------------------------
    // FrStatData
    //-------------------------------------------------------------------
    leader.str( "fr_stat_data:" );
    obj = mk_frame_object( x, FrameSpec::Info::FSI_FR_STAT_DATA );
    verify_convert( leader, x, obj, "FrStatData", Test );
    //-------------------------------------------------------------------
    // FrSummary
    //-------------------------------------------------------------------
    leader.str( "fr_summary:" );
    obj = mk_frame_object( x, FrameSpec::Info::FSI_FR_SUMMARY );
    verify_convert( leader, x, obj, "FrSummary", Test );
    //-------------------------------------------------------------------
    // FrTable
    //-------------------------------------------------------------------
    leader.str( "fr_table:" );
    obj = mk_frame_object( x, FrameSpec::Info::FSI_FR_TABLE );
    verify_convert( leader, x, obj, "FrTable", Test );
    //-------------------------------------------------------------------
    // FrVect
    //-------------------------------------------------------------------
    leader.str( "fr_vect:" );
    obj = mk_frame_object( x, FrameSpec::Info::FSI_FR_VECT );
    verify_convert( leader, x, obj, "FrVect", Test );
  }

  //---------------------------------------------------------------------
  // Exit with the proper exit status
  //---------------------------------------------------------------------
  Test.Exit( );
} catch (...) {
  Test.Check( false ) << "Caught an exception" << std::endl;
  Test.Exit( );
}

//=======================================================================
// General routines
//=======================================================================

static void
verify_downconvert( std::ostringstream& Leader,
		    const int Version,
		    frame_object_type FrameObj,
		    const std::string& FrameObjName,
		    General::UnitTest& Test )
{
  Leader << " down convert from version: " << Version
	 << ": "
    ;
  try
  {
    switch ( Version )
    {
    case 3:
      //-----------------------------------------------------------------
      // This is the first supported version of the frame spec
      //-----------------------------------------------------------------
      return;
    case 4:
      verify_downconvert< 4 >( FrameObj, Leader.str( ), Test );
      return;
    case 6:
      verify_downconvert< 6 >( FrameObj, Leader.str( ), Test );
      return;
    case 7:
      verify_downconvert< 7 >( FrameObj, Leader.str( ), Test );
      return;
    case 8:
      verify_downconvert< 8 >( FrameObj, Leader.str( ), Test );
      return;
    default:
      break;
    }
    //-------------------------------------------------------------------
    // Unsupported version passed
    //-------------------------------------------------------------------
    std::ostringstream	msg;

    msg << "verify_downconvert:"
	<< " version: " << Version
	<< " struct: " << FrameObjName
      ;
    throw
      Unimplemented( msg.str( ),
		     0, __FILE__, __LINE__ );
  }
  catch( const std::exception& except )
  {
    Test.Check( false )
      << Leader.str( )
      << "struct: " << FrameObjName
      << " Caught excpetion: " << except.what( )
      << std::endl;
    ;
  }
}

static void
verify_upconvert( std::ostringstream& Leader,
		  const int Version,
		  frame_object_type FrameObj,
		  const std::string& FrameObjName,
		  General::UnitTest& Test )
{
  Leader << " up convert to version: " << Version
	 << ": "
    ;
  try
  {
    switch ( Version )
    {
    case 3:
      //-----------------------------------------------------------------
      // This is the first supported version of the frame spec
      //-----------------------------------------------------------------
      return;
    case 4:
      verify_upconvert< 4 >( FrameObj, Leader.str( ), Test );
      return;
    case 6:
      verify_upconvert< 6 >( FrameObj, Leader.str( ), Test );
      return;
    case 7:
      verify_upconvert< 7 >( FrameObj, Leader.str( ), Test );
      return;
    case 8:
      verify_upconvert< 8 >( FrameObj, Leader.str( ), Test );
      return;
    default:
      break;
    }
    //-------------------------------------------------------------------
    // Unsupported version passed
    //-------------------------------------------------------------------
    std::ostringstream	msg;

    msg << "verify_upconvert:"
	<< " version: " << Version
	<< " struct: " << FrameObjName
      ;
    throw
      Unimplemented( msg.str( ),
		     0, __FILE__, __LINE__ );
  }
  catch( const std::exception& except )
  {
    Test.Check( false )
      << Leader.str( )
      << "struct: " << FrameObjName
      << " Caught excpetion: " << except.what( )
      << std::endl;
    ;
  }
}

//=======================================================================
// Frame Object
//=======================================================================
mk_frame_object_ret_type
mk_frame_object( int SpecVersion, FrameSpec::Info::frame_object_types Type )
{
  try
  {
    switch( SpecVersion )
    {
    case 3:
      return mk_frame_object< 3 >( Type );
    case 4:
      return mk_frame_object< 4 >( Type );
    case 6:
      return mk_frame_object< 6 >( Type );
    case 7:
      return mk_frame_object< 7 >( Type );
    case 8:
      return mk_frame_object< 8 >( Type );
    default:
      {
	std::ostringstream	msg;

	msg << "mk_frame_object: Unsupported version: "
	    << SpecVersion
	  ;
	throw
	  Unimplemented( msg.str( ),
			 0, __FILE__, __LINE__ );
      }
      break;
    }
  }
  catch( const std::exception& except )
  {
    Test.Check( false )
      << " mk_frame_object: "
      << " Caught excpetion: " << except.what( )
      << std::endl;
    ;
  }
  return mk_frame_object_ret_type( );
}
