#ifndef FrameCPP_VERSION_8_FrVect_HH
#define FrameCPP_VERSION_8_FrVect_HH

#if ! defined(SWIGIMPORTED)
#include <memory>
#include <vector>
#include <string>
#endif /* ! defined(SWIGIMPORTED) */

#include "general/types.hh"
#if ! defined(SWIGIMPORTED)
#include "general/SharedArray.hh"
#include "general/SharedPtr.hh"

#include "framecpp/FrameCPP.hh"

#include "framecpp/Common/Compression.hh"
#include "framecpp/Common/FrVect.hh"

#include "framecpp/Version7/FrVect.hh"

#include "framecpp/Version8/FrameSpec.hh"
#include "framecpp/Version8/Dimension.hh"
#include "framecpp/Version8/GPSTime.hh"
#include "framecpp/Version8/STRING.hh"
#endif /* ! defined(SWIGIMPORTED)*/

#undef FR_VECT_COMPRESS_TYPE
#undef FR_VECT_TYPE_TYPE
#undef FR_VECT_NDATA_TYPE
#undef FR_VECT_NBYTES_TYPE
#undef FR_VECT_NDIM_TYPE
#undef FR_VECT_COMPRESSION_LEVEL_TYPE

#define	FR_VECT_COMPRESS_TYPE	INT_2U
#define	FR_VECT_TYPE_TYPE	INT_2U
#define	FR_VECT_NDATA_TYPE	INT_8U
#define	FR_VECT_NBYTES_TYPE	INT_8U
#define	FR_VECT_NDIM_TYPE	INT_4U
#define FR_VECT_COMPRESSION_LEVEL_TYPE int

namespace FrameCPP
{
  namespace Version_8
  {
#if ! defined(SWIG)
    using Previous::FrVectDataTypes;
#endif /* ! defined(SWIG) */

    //===================================================================
    /// \brief Vector Data Structure Definition
    //===================================================================
    class FrVect
#if ! defined(SWIG)
      : public ObjectWithChecksum< Common::FrameSpec::Object >,
        public Common::FrVect,
        public FrVectDataTypes
#endif /* ! defined(SWIG) */
    {
    public:
      typedef FR_VECT_COMPRESSION_LEVEL_TYPE compression_level_type;

      enum compression_scheme_type {
	//---------------------------------------------------------
	//---------------------------------------------------------

#if WORDS_BIGENDIAN
#define NATIVE_DECL(x) x = BIGENDIAN_##x
#else
#define NATIVE_DECL(x) x = LITTLEENDIAN_##x
#endif
#define CST_DECL(x,y) BIGENDIAN_##x = y, LITTLEENDIAN_##x = y + 256, NATIVE_DECL(x)

	CST_DECL(RAW,0),
	CST_DECL(GZIP,1),
        CST_DECL(DIFF_GZIP,3),
	CST_DECL(ZERO_SUPPRESS_WORD_2,5),
	CST_DECL(ZERO_SUPPRESS_WORD_4,8),
	CST_DECL(ZERO_SUPPRESS_WORD_8,10),

#undef CST_DECL
#undef NATIVE_DECL

	//---------------------------------------------------------
	// Declaration of meta modes
	//---------------------------------------------------------
	ZERO_SUPPRESS_OTHERWISE_GZIP = 6,
	BEST_COMPRESSION = 9,

	//---------------------------------------------------------
	// Backwards compatability
	//---------------------------------------------------------
	ZERO_SUPPRESS_SHORT = ZERO_SUPPRESS_WORD_2,
	ZERO_SUPPRESS_INT_FLOAT = ZERO_SUPPRESS_WORD_4,
#undef GENERIC
	NONE = RAW
      };

      enum data_types_type {
	FR_VECT_C = 0,
	FR_VECT_2S = 1,
	FR_VECT_8R = 2,
	FR_VECT_4R = 3,
	FR_VECT_4S = 4,
	FR_VECT_8S = 5,
	FR_VECT_8C = 6,
	FR_VECT_16C = 7,
	FR_VECT_STRING = 8,
	FR_VECT_2U = 9,
	FR_VECT_4U = 10,
	FR_VECT_8U = 11,
	FR_VECT_1U = 12
      };


      typedef FR_VECT_COMPRESS_TYPE		compress_type;
      typedef FR_VECT_TYPE_TYPE			type_type;
      typedef FR_VECT_NDATA_TYPE		nData_type;
      typedef FR_VECT_NBYTES_TYPE		nBytes_type;
      typedef FR_VECT_NDIM_TYPE			nDim_type;
      typedef General::SharedArray< CHAR_U >	data_type;
      typedef data_type::element_type		data_element_type;
      typedef data_element_type*		data_pointer_type;
      typedef const data_element_type*		data_const_pointer_type;
      typedef std::vector< Dimension >		dimension_container_type;
      typedef std::unique_ptr< FrVect >		subfrvect_type;

      static const int DEFAULT_GZIP_LEVEL;

      //-----------------------------------------------------------------
      /// \brief Promote previous version of object to this version
      ///
      /// \param[in] Source
      ///     The frame specification version of the object.
      /// \param[in] Obj
      ///     The previous version of the object.
      /// \param[in] Stream
      ///     Stream from which the origional object was created.
      ///
      /// \return
      ///     A new instance of FrameH based on Obj
      //-----------------------------------------------------------------
      static promote_ret_type
      Promote( INT_2U Source,
	       promote_arg_type Obj,
	       Common::IStream* Stream );

      //-----------------------------------------------------------------
      /// \brief The name structure name of this object.
      ///
      /// \return
      ///     The name of the structure as specified by the frame
      ///     specification.
      //-----------------------------------------------------------------
      static const char* StructName( );

      //-----------------------------------------------------------------
      /// \brief The description of structure
      ///
      /// \return
      ///     A Description object which describes this structure as
      ///     specified by the frame specification.
      //-----------------------------------------------------------------
      static const Common::Description* StructDescription( );

      //-----------------------------------------------------------------
      /// \brief Default constructor
      ///
      /// \return
      ///    A new instance of this object.
      //-----------------------------------------------------------------
      FrVect( );

      //-----------------------------------------------------------------
      /// \brief Copy Constructor
      ///
      /// \param[in] vect
      ///     The object from which to copy the information.
      ///
      /// \return
      ///    A new instance of this object.
      //-----------------------------------------------------------------
      FrVect( const FrVect& vect );


      //-----------------------------------------------------------------
      /// \brief Constructor.
      ///
      /// \param[in] name
      ///     The name of the data.
      /// \param[in] type
      ///     The data type.
      /// \param[in] nDim
      ///     The number of dimensions.
      /// \param[in] dims
      ///     A pointer to 'nDim' Dimension objects containing
      ///     information about the dimensions for this data.
      /// \param[in] byte_order
      ///     Byte order of the data. Default is BYTE_ORDER_HOST
      /// \param[in] data
      ///     A pointer to the data.  Default: 0
      /// \param[in] unitY
      ///     Units for the data.  Default: ""
      ///
      /// \return
      ///    A new instance of this object.
      //-----------------------------------------------------------------
      FrVect( const std::string& name,
	      type_type type,
	      nDim_type nDim,
	      const Dimension* dims,
	      const byte_order_type byte_order = BYTE_ORDER_HOST,
	      const void* data = 0, 
	      const std::string& unitY = "" );

      FrVect( const std::string& name,
	      compress_type Compress,
	      type_type type,
	      nDim_type nDim,
	      const Dimension* dims,
	      nData_type NData,
	      nBytes_type NBytes,
	      data_type Data,
	      const std::string& unitY = "" );

#if 0
      FrVect( const std::string& name,
	      compress_type Compress,
	      type_type type,
	      nDim_type nDim,
	      const Dimension* dims,
	      nData_type NData,
	      nBytes_type NBytes,
	      data_type Data,
	      const std::string& unitY = "" );
#endif /* 0 */

      //-----------------------------------------------------------------
      /// \brief Constructor
      ///
      /// \param[in] Name
      ///     The name of the data.
      /// \param[in] Type
      ///     The data type.
      /// \param[in] NDim
      ///     The number of dimensions.
      /// \param[in] Dims
      ///     A pointer to 'nDim' Dimension objects containing
      ///     information about the dimensions for this data.
      /// \param[in] UnitY
      ///     Units for the data.
      ///
      /// \return
      ///    A new instance of this object.
      //-----------------------------------------------------------------
      FrVect( const std::string& Name,
	      type_type Type,
	      nDim_type NDim,
	      const Dimension* Dims,
	      const std::string& UnitY );

#if ! defined(SWIG)
      template< class T >
      FrVect( const std::string& name, nDim_type nDim = 1,
	      const Dimension* dims = 0,
	      const T* data = 0, const std::string& unitY = "" );

      template< class T >
      FrVect( const std::string& name, nDim_type nDim = 1,
	      const Dimension* dims = 0,
	      T* data = 0, const std::string& unitY = "" );
#endif /* ! defined(SWIG) */

      //-----------------------------------------------------------------
      /// \brief Promotion Constructor
      ///
      /// \param[in] Source
      ///     An FrAdcData structure from a previous frame specification.
      /// \param[in] Stream
      ///     The stream from which the earliest version of the
      ///     FrAdcData structure was read.
      ///
      /// \return
      ///     A new instance of the object.
      //-----------------------------------------------------------------
      FrVect( const Previous::FrVect& Source,
	      Common::IStream* Stream );

      //-----------------------------------------------------------------
      /// \brief Destructor
      //-----------------------------------------------------------------
      virtual ~FrVect( );

      //-----------------------------------------------------------------
      /// \brief Set the current FrVect with the MIME data
      //-----------------------------------------------------------------
      void MimeData( const std::string& MimeType,
		     void* Data, nBytes_type DataSize );

      //-----------------------------------------------------------------
      /// \brief The name structure name of this object.
      ///
      /// \return
      ///     The name of the structure as specified by the frame
      ///     specification.
      //-----------------------------------------------------------------
      virtual const char* ObjectStructName( ) const;

      //-----------------------------------------------------------------
      /// \brief equality operator
      ///
      /// \param[in] RHS
      ///     The FrAdcData object to be compared.
      ///
      /// \return
      ///     The value true is returned if this object is equivelent
      ///     to the RHS instance; false otherwise.
      //-----------------------------------------------------------------
      bool operator==( const FrVect& RHS ) const;

      //-----------------------------------------------------------------
      /// \brief inequality operator
      ///
      /// \param[in] RHS
      ///     The FrAdcData object to be compared.
      ///
      /// \return
      ///     The value true is returned if this object is not equivelent
      ///     to the RHS instance; false otherwise.
      //-----------------------------------------------------------------
      bool operator!=( const FrVect& RHS ) const;

      //-----------------------------------------------------------------
      /// \brief Concatinate with another FrAdcData
      ///
      /// \param[in] RHS
      ///     The source of the information to append to this FrAdcData
      ///     structure.
      ///
      /// \return
      ///     A reference to this object
      //-----------------------------------------------------------------
      FrVect& operator+=( const FrVect& RHS );

      //-----------------------------------------------------------------
      /// \brief equality operator
      ///
      /// \param[in] RHS
      ///     The FrVect object to be compared.
      ///
      /// \return
      ///     The value true is returned if the object is equivelent
      ///     to the RHS instance; false otherwise.
      //-----------------------------------------------------------------
      virtual bool operator==( const Common::FrameSpec::Object& RHS ) const;

      //-----------------------------------------------------------------
      /// \brief Allocate a block of memory suitable for storing data.
      ///
      /// \param[in] Type
      ///     The type of data that is to be stored in the buffer.
      /// \param[in] Dims
      ///     Information concearning each dimension of the array
      ///     of data to be stored in the buffer.
      /// \param[in] NDim
      ///     The number of dimensions of data to be stored in the
      ///     buffer
      ///
      /// \return
      ///     Upon successful allocation, an appropriated sized buffer
      ///     is returned.
      ///     An expection is thrown for all error conditions.
      //-----------------------------------------------------------------
      static data_type DataAlloc( type_type Type,
				  const Dimension* Dims,
				  const nDim_type NDim = 1 );
      //-----------------------------------------------------------------
      /// \brief Retrieve the channel name.
      ///
      /// \return
      ///     The channel name
      //-----------------------------------------------------------------
      const std::string& GetName( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the compression algorithm number.
      ///
      /// \return
      ///     The compression algorithm number.
      //-----------------------------------------------------------------
      compress_type GetCompress( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the vector class.
      ///
      /// \return
      ///     The vector class.
      //-----------------------------------------------------------------
      type_type GetType( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the number of sample elements in data series.
      ///
      /// \return
      ///     The number of sample elements in data series.
      //-----------------------------------------------------------------
      nData_type GetNData( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the number of bytes in the compressed vector.
      ///
      /// \return
      ///     The number of bytes in the compressed vector.
      //-----------------------------------------------------------------
      nBytes_type GetNBytes( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the pointer to the data.
      ///
      /// \return
      ///     The pointer to the data.
      ///
      /// \note
      ///     This will expand the data if it is currently compressed.
      //-----------------------------------------------------------------
      data_type GetData( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the pointer to the compressed data.
      ///
      /// \return
      ///     The pointer to the compressed data.
      //-----------------------------------------------------------------
      data_type GetDataRaw( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the pointer to the compressed data.
      ///
      /// \return
      ///     The pointer to the compressed data.
      //-----------------------------------------------------------------
      data_type GetDataRaw( );

      //-----------------------------------------------------------------
      /// \brief Retrieve the pointer to the uncompressed data.
      ///
      /// \return
      ///     The pointer to the uncompressed data.
      //-----------------------------------------------------------------
      data_type GetDataUncompressed( );

      //-----------------------------------------------------------------
      /// \brief Retrieve the pointer to the uncompressed data.
      ///
      /// \return
      ///     The pointer to the uncompressed data.
      //-----------------------------------------------------------------
      const CHAR_U* GetDataUncompressed( data_type& Expanded ) const;

      template< class T > static type_type GetDataType();

      //-----------------------------------------------------------------
      /// \brief Retrieve the number of dimensions
      ///
      /// \return
      ///     The number of dimensions
      //-----------------------------------------------------------------
      nDim_type GetNDim( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the Nth dimension
      ///
      /// \return
      ///     The Nth dimension
      //-----------------------------------------------------------------
      const Dimension& GetDim( nDim_type Offset ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the Nth dimension
      ///
      /// \return
      ///     The Nth dimension
      //-----------------------------------------------------------------
      Dimension& GetDim( nDim_type Offset );

      //-----------------------------------------------------------------
      /// \brief Retrieve all the dimension information
      ///
      /// \return
      ///     The colletion of dimension data
      //-----------------------------------------------------------------
      dimension_container_type& GetDims( );

      //-----------------------------------------------------------------
      /// \brief Retrieve the description of how to interpret each element
      ///
      /// \return
      ///     The description of how to interpret each element
      //-----------------------------------------------------------------
      const std::string& GetUnitY( ) const;

      void Compress( compression_scheme_type Scheme, int GZipLevel );
      void Uncompress( );

      virtual void CompressData( INT_4U Scheme, INT_2U GZipLevel );

      virtual Compression::compression_base_type
      Compression( ) const;

      virtual Common::FrameSpec::Object*
      CloneCompressed( cmn_compression_scheme_type Scheme,
		       cmn_compression_level_type Level ) const;

      //-----------------------------------------------------------------
      /// \brief Establish the channel name.
      ///
      /// \param[in] Name
      ///     The channel name
      //-----------------------------------------------------------------
      void SetName( const std::string& Name );

      //-----------------------------------------------------------------
      /// \brief Retrieve the description of how to interpret each element
      ///
      /// \return
      ///     The description of how to interpret each element
      //-----------------------------------------------------------------
      void SetUnitY( const std::string& UnitY );

      //-----------------------------------------------------------------
      //-----------------------------------------------------------------
      void SetNData( nData_type NData );

      //-----------------------------------------------------------------
      //-----------------------------------------------------------------
      size_t GetTypeSize( ) const;

      //-----------------------------------------------------------------
      /// \brief Get size of single data point of particular size.
      ///
      /// \param[in] Type
      ///     Data type ID.
      ///
      /// \return
      ///     Data size.
      //-----------------------------------------------------------------
      static size_t GetTypeSize( type_type Type );

      //-----------------------------------------------------------------
      /// \brief Merge with another FrAdcData
      ///
      /// \param[in] RHS
      ///     The source of the information to append to this FrAdcData
      ///     structure.
      ///
      /// \return
      ///     A reference to this object
      //-----------------------------------------------------------------
      FrVect& Merge( const FrVect& RHS );

      //-----------------------------------------------------------------
      /// \brief Request a subset of the data.
      ///
      /// \param[in] Start
      ///     The offset in the data set where to start.
      /// \param[in] Stop
      ///     The offset in the data set where to stop.
      ///
      /// \return
      ///     The subset of data bounded by data[Start, Stop).
      //-----------------------------------------------------------------
      subfrvect_type SubFrVect( INT_4U Start, INT_4U Stop ) const;

    protected:
      //-----------------------------------------------------------------
      /// \brief Verify the integrity of the object
      ///
      /// \param Verifier
      ///     The handle to the Verify object used to store data that
      ///     needs to be accumulated before being processed.
      /// \param Stream
      ///     The input stream from which the object was created.
      ///
      /// \remarks
      ///     Upon failure, this method thows an exception.
      //-----------------------------------------------------------------
      virtual void VerifyObject( Common::Verify& Verifier,
				 Common::IFrameStream& Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Demotes object to previous version of the frame spec
      ///
      /// \param[in] Target
      ///     The version of the frame specification to demote too.
      /// \param[in] Obj
      ///     The version of the object to demote.
      /// \param[in] Stream
      ///     The input stream from which the original object was read.
      ///
      /// \return
      ///     An object of the previous generation.
      //-----------------------------------------------------------------
      virtual demote_ret_type
      demote( INT_2U Target,
	      demote_arg_type Obj,
	      Common::IStream* Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Promotes object to another version of the frame spec
      ///
      /// \param[in] Target
      ///     The version of the promoted frame specification.
      /// \param[in] Obj
      ///     The object to be promoted.
      /// \param[in] Stream
      ///     The input stream from which the original object was read.
      ///
      /// \return
      ///     An object promoted to the next generation.
      //-----------------------------------------------------------------
      virtual promote_ret_type
      promote( INT_2U Target,
	       promote_arg_type Obj,
	       Common::IStream* Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Number of bytes needed to write this structure
      ///
      /// \param[in] Stream
      ///     The stream from which to the object is being read or
      ///     written.
      ///
      /// \return
      ///     The number of bytes need to read or write this object.
      //-----------------------------------------------------------------
      virtual FrameCPP::cmn_streamsize_type
      pBytes( const Common::StreamBase& Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Stream constructor
      ///
      /// \param[in] Stream
      ///     The input stream from where the object is being read.
      //-----------------------------------------------------------------
      virtual FrVect* pCreate( Common::IStream& Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Write the structure to the stream
      ///
      /// \param[in] Stream
      ///     The output stream where the object is to be written.
      //-----------------------------------------------------------------
      virtual void pWrite( Common::OStream& Stream ) const;

      void
      expandToBuffer( data_type& Dest,
		      nBytes_type& DestNBytes ) const;

    private:
      //-----------------------------------------------------------------
      /// \brief Object with checksum data.
      //-----------------------------------------------------------------
      typedef ObjectWithChecksum< Common::FrameSpec::Object > object_type;

      //-----------------------------------------------------------------
      /// \brief Dictionary class id for this structure.
      //-----------------------------------------------------------------
      static const Common::FrameSpec::Info::frame_object_types s_object_id
      = Common::FrameSpec::Info::FSI_FR_VECT;
      
      using Common::FrameSpec::Object::Create;

      struct data_definition_type
      {
      public:

	//---------------------------------------------------------------
	/// Channel name.
	///
	/// \note
	///     Not required to be unique.
	//---------------------------------------------------------------
	STRING				name;
	//---------------------------------------------------------------
	/// Compression algorithm number
	//---------------------------------------------------------------
	compress_type			compress;
	//---------------------------------------------------------------
	/// Vector class
	//---------------------------------------------------------------
	type_type			type;
	//---------------------------------------------------------------
	/// Number of sample elements in data series.
	//---------------------------------------------------------------
	nData_type			nData;
	//---------------------------------------------------------------
	/// Number of bytes in the compressed vector.
	//---------------------------------------------------------------
	nBytes_type			nBytes;
	//---------------------------------------------------------------
	/// Compressed data series.
	//---------------------------------------------------------------
	data_type			data;
	//---------------------------------------------------------------
	/// Dimension data for data series.
	//---------------------------------------------------------------
	dimension_container_type	dimension;
	//---------------------------------------------------------------
	/// String describing how to interpret the value of each
	/// element.
	//---------------------------------------------------------------
	STRING				unitY;

        data_definition_type();
        data_definition_type( const std::string& n, INT_2U t, 
			      const byte_order_type byte_order, nDim_type ndim,
			      const Dimension* dims, const std::string& unit );

	void copy_core( const data_definition_type& Source );

	bool operator==( const data_definition_type& RHS ) const;

      };
 
      data_definition_type	m_data;

      //-----------------------------------------------------------------
      /// \brief Stream Constructor
      ///
      /// \param[in] Stream
      ///     The stream from which the object is being read.
      ///
      /// \return
      ///    A new instance of this object.
      //-----------------------------------------------------------------
      FrVect( Common::IStream& Stream );

      void data_copy( data_const_pointer_type Data, nBytes_type NBytes );

      void data_alloc( nBytes_type NBytes );

    };


    inline Compression::compression_base_type FrVect::
    Compression( ) const
    {
      return ( 0x0FF & m_data.compress );
    }

    inline FrVect::promote_ret_type FrVect::
    Promote( INT_2U Source,
	     promote_arg_type Obj,
	     Common::IStream* Stream )
    {
      return Object::PromoteObject< Previous::FrVect, FrVect >
	( DATA_FORMAT_VERSION, Source, Obj, Stream );
    }

    inline const char* FrVect::
    StructName( )
    {
      static const CHAR* class_name( "FrVect" );
      return class_name;
    }

    //-------------------------------------------------------------------
    /// Return a pointer to the data being managed by the FrVect
    /// structure.
    ///
    /// \see GetDataRaw, GetDataUncompressed
    //-------------------------------------------------------------------
    inline FrVect::data_type FrVect::
    GetData( ) const
    {
      return GetDataRaw( );
    }

    inline FrVect::compress_type FrVect::
    GetCompress( ) const
    {
      return m_data.compress;
    }

    inline FrVect::type_type FrVect::
    GetType( ) const
    {
      return m_data.type;
    }

    inline size_t FrVect::
    GetTypeSize( ) const
    {
      return GetTypeSize( m_data.type );
    }

    inline FrVect::nData_type FrVect::
    GetNData( ) const
    {
      return m_data.nData;
    }

    inline FrVect::nBytes_type FrVect::
    GetNBytes( ) const
    {
      return m_data.nBytes;
    }

    inline FrVect::nDim_type FrVect::GetNDim( ) const
    {
      return m_data.dimension.size( );
    }

    inline const Dimension& FrVect::
    GetDim( nDim_type Offset ) const
    {
      //:TODO: Throw exception if value out of range
      return m_data.dimension[ Offset ];
    }

    inline Dimension& FrVect::
    GetDim( nDim_type Offset )
    {
      //:TODO: Throw exception if value out of range
      return m_data.dimension[ Offset ];
    }

    inline FrVect::dimension_container_type& FrVect::
    GetDims( )
    {
      return m_data.dimension;
    }

    inline const std::string& FrVect::
    GetUnitY( ) const
    {
      return m_data.unitY;
    }

    //-------------------------------------------------------------------
    /// Assign a new value for the channel name.
    //-------------------------------------------------------------------
    inline void FrVect::
    SetName( const std::string& Name )
    {
      m_data.name = Name;
    }

    //-------------------------------------------------------------------
    /// Assign a new value for the channel name.
    //-------------------------------------------------------------------
    inline void FrVect::
    SetUnitY( const std::string& UnitY )
    {
      m_data.unitY = UnitY;
    }

    inline void FrVect::
    SetNData( nData_type NData )
    {
      m_data.nData = NData;
    }
  } // namespace - Version_8
} // namespace FrameCPP

#if defined( __SUNPRO_CC ) && ( __SUNPRO_CC <= 0x550 )
#include "framecpp/Version8/FrVect.icc"
#endif /* */

#endif /* FrameCPP_VERSION_8_FrVect_HH */
