#ifndef FrameCPP_VERSION_8_FrTOCStatData_HH
#define FrameCPP_VERSION_8_FrTOCStatData_HH

#include "framecpp/Version8/STRING.hh"

namespace FrameCPP
{
  namespace Common
  {
    class TOCInfo;
  }

  namespace Version_8
  {
    //===================================================================
    /// \brief Indexed elements of the FrSimData structure.
    //===================================================================
    class FrTOCStatData
    {
    public:
      typedef STRING	name_type;
      typedef INT_4U	nstat_type;
      typedef STRING	detector_type;
      typedef INT_4U	nstat_instance_type;
      typedef INT_4U	tstart_type;
      typedef INT_4U	tend_type;
      typedef INT_4U	version_type;
      typedef INT_8U	positionStat_type;

      struct stat_instance_type {
	//---------------------------------------------------------------
	/// \brief Start time of the static data.
	//---------------------------------------------------------------
	tstart_type		tStart;
	//---------------------------------------------------------------
	/// \brief End time of the static data.
	//---------------------------------------------------------------
	tend_type		tEnd;
	//---------------------------------------------------------------
	/// \brief Version of the static data.
	//---------------------------------------------------------------
	version_type		version;
	//---------------------------------------------------------------
	/// \brief FrStatData offset positions.
	///
	/// FrStatData offset positions, in bytes,
	/// from the beginning of the file.
	//---------------------------------------------------------------
	positionStat_type	positionStat;
      };

      typedef std::vector< stat_instance_type > stat_instance_container_type;

      struct stat_type {
	//---------------------------------------------------------------
	/// \brief FrDetector associated with the static data.
	//---------------------------------------------------------------
	detector_type			detector;
	//---------------------------------------------------------------
	/// \brief Container of static data members
	//---------------------------------------------------------------
	stat_instance_container_type	stat_instances;
      };

      typedef std::map< name_type, stat_type >	stat_container_type;

      //-----------------------------------------------------------------
      /// \brief Default constructor.
      //-----------------------------------------------------------------
      FrTOCStatData( );

      //-----------------------------------------------------------------
      /// \brief Number of bytes needed to write this structure
      ///
      /// \param[in] Stream
      ///     The stream from which to the object is being read or
      ///     written.
      ///
      /// \return
      ///     The number of bytes need to read or write this object.
      //-----------------------------------------------------------------
      Common::FrameSpec::size_type
      Bytes( const Common::StreamBase& Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Return all channels
      ///
      /// \return
      ///     Constant container to the indexed FrStatData elements.
      //-----------------------------------------------------------------
      const stat_container_type& GetStat( ) const;

      //-----------------------------------------------------------------
      /// \brief Return number of FrStatData structures.
      ///
      /// \return
      ///     The number of FrStatData structures.
      //-----------------------------------------------------------------
      INT_4U GetNTotalStat( ) const;

      //-----------------------------------------------------------------
      /// \brief equality operator
      ///
      /// \param[in] RHS
      ///     The FrTOCAdcData object to be compared.
      ///
      /// \return
      ///     The value true is returned if this object is equivelent
      ///     to the RHS instance; false otherwise.
      //-----------------------------------------------------------------
      bool operator==( const FrTOCStatData& RHS ) const;

      //-----------------------------------------------------------------
      /// \brief The description of structure
      ///
      /// \return
      ///     A Description object which describes this structure as
      ///     specified by the frame specification.
      //-----------------------------------------------------------------
      template< typename SE >
      static void Description( Common::Description& Desc );

      //-----------------------------------------------------------------
      /// \brief Gather TOC info for FrStatData being written.
      ///
      /// \param[in] Info
      ///     Information
      ///
      /// \param[in] FrameOffset
      ///     The frame offset of the frame being written.
      ///     The frame offsets start at 0 (zero).
      ///
      /// \param[in] Position
      ///     The byte offset from the start of the file
      ///     where the structure is written.
      //-----------------------------------------------------------------
      void QueryStatData( const Common::TOCInfo& Info,
			  INT_4U FrameOffset,
			  INT_8U Position );

    protected:
      stat_container_type	m_info;

      //-----------------------------------------------------------------
      /// \brief Stream constructor
      ///
      /// \param[in] Stream
      ///     The input stream from where the object is being read.
      //-----------------------------------------------------------------
      FrTOCStatData( Common::IStream& Stream );

      //-----------------------------------------------------------------
      /// \brief Write the structure to the stream
      ///
      /// \param[in] Stream
      ///     The output stream where the object is to be written.
      //-----------------------------------------------------------------
      void write( Common::OStream& Stream ) const;
    };

    inline Common::FrameSpec::size_type FrTOCStatData::
    Bytes( const Common::StreamBase& Stream ) const
    {
      Common::FrameSpec::size_type 
	retval
	= sizeof( nstat_type ) // nStatType
	+ sizeof( nstat_type ) // nTotalStat
	;

      static const INT_4U si =
	sizeof( tstart_type )
	+ sizeof( tend_type )
	+ sizeof( version_type )
	+ sizeof( positionStat_type )
	;


      for ( stat_container_type::const_iterator
	      cur = m_info.begin( ),
	      last = m_info.end( );
	    cur != last;
	    ++cur )
      {
	retval +=
	  cur->first.Bytes( )
	  + cur->second.detector.Bytes( )
	  + sizeof( nstat_instance_type )
	  + ( cur->second.stat_instances.size( ) * si )
	  ;
      }
      return retval;
    }

    template< typename SE >
    void FrTOCStatData::
    Description( Common::Description& Desc )
    {
      Desc( SE( "nStatType", "INT_4U",
		"Number of static data block types in the file." ) );
      Desc( SE( "nameStat", "STRING[nStatType]",
		"Array of FrStatData name (size of nStatType)" ) );
      Desc( SE( "detector", "STRING[nStatType]",
		"Array of Detector name(size of nStatType)" ) );
      Desc( SE( "nStatInstance", "INT_4U[nStatType]",
		"Array of number of instance for each FrStatData(size of nStatType)" ) );
      Desc( SE( "nTotalStat", "INT_4U",
		"Summation of nStatInstance array" ) );
      
      Desc( SE( "tStart", "INT_4U[nTotalStat]",
		"Array of GPS integer start times,"
		" in seconds (size of nTotalStat)" ) );
      Desc( SE( "tEnd", "INT_4U[nTotalStat]",
		"Array of GPS integer end times,"
		" in seconds (size of nTotalStat)" ) );
      Desc( SE( "version", "INT_4U[nTotalStat]",
		"Array of version time"
		" (size of nTotalStat)" ) );
      Desc( SE( "positionStat", "INT_8U[nTotalStat]",
		"Array of FrStatData positions from beginning of file"
		" (size of nTotalStat)" ) );
    }

    inline INT_4U FrTOCStatData::
    GetNTotalStat( ) const
    {
      INT_4U retval = 0;
      for ( stat_container_type::const_iterator
	      cur = m_info.begin( ),
	      last = m_info.end( );
	    cur != last;
	    ++cur )
      {
	retval += cur->second.stat_instances.size( );
      }
      return retval;
    }

    inline const FrTOCStatData::stat_container_type& FrTOCStatData::
    GetStat( ) const
    {
      return m_info;
    }

  } // namespace - Version_8
} // namespace - FrameCPP

#endif /* FrameCPP_VERSION_8_FrTOCStatData_HH */
