#ifndef FrameCPP_VERSION_8_FrTOCEvent_HH
#define FrameCPP_VERSION_8_FrTOCEvent_HH

#include "framecpp/Version7/FrTOCEvent.hh"

#include "framecpp/Version8/STRING.hh"
#include "framecpp/Version8/GPSTime.hh"

namespace FrameCPP
{
  namespace Common
  {
    class TOCInfo;
  }

  namespace Version_8
  {
    //===================================================================
    /// \brief Indexed elements of the FrEvent structure.
    //===================================================================
    class FrTOCEvent
    {
    public:
      typedef INT_4U	nevent_type;
      typedef STRING	name_type;
      typedef INT_4U	nTotalEvent_type;
      typedef INT_4U	gtimesEvent_type;
      typedef INT_4U	gtimenEvent_type;
      typedef REAL_4	amplitudeEvent_type;
      typedef INT_8U	positionEvent_type;

      struct event_type {
	//---------------------------------------------------------------
	/// \brief Start time of event.
	//---------------------------------------------------------------
	GPSTime			GTime;
	//---------------------------------------------------------------
	/// \brief Amplitude of event.
	//---------------------------------------------------------------
	amplitudeEvent_type	amplitudeEvent;
	//---------------------------------------------------------------
	/// \brief FrEvent offset positions.
	///
	/// FrEvent offset positions, in bytes,
	/// from the beginning of the file.
	//---------------------------------------------------------------
	positionEvent_type	positionEvent;

	const event_type& operator=( const Previous::FrTOCEvent::event_type& Source );
      };

      static const int event_type_size
      = sizeof( gtimesEvent_type )
	+ sizeof( gtimenEvent_type )
	+ sizeof( amplitudeEvent_type )
	+ sizeof( positionEvent_type )
	;

      typedef std::vector< event_type > events_container_type;
      /* typedef std::vector< event_container_type > events_container_type; */
      typedef std::map< name_type, events_container_type >
      nameEvent_container_type;

      //-----------------------------------------------------------------
      /// \brief Default constructor
      //-----------------------------------------------------------------
      FrTOCEvent( );

      //-----------------------------------------------------------------
      /// \brief Number of bytes needed to write this structure
      ///
      /// \param[in] Stream
      ///     The stream from which to the object is being read or
      ///     written.
      ///
      /// \return
      ///     The number of bytes need to read or write this object.
      //-----------------------------------------------------------------
      Common::FrameSpec::size_type
      Bytes( const Common::StreamBase& Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Return all events
      ///
      /// \return
      ///     Constant container to the indexed FrEvent elements.
      //-----------------------------------------------------------------
      const nameEvent_container_type& GetEvent( ) const;

      //-----------------------------------------------------------------
      /// \brief Gather TOC info for FrEvent being written.
      ///
      /// \param[in] Info
      ///     Information
      ///
      /// \param[in] FrameOffset
      ///     The frame offset of the frame being written.
      ///     The frame offsets start at 0 (zero).
      ///
      /// \param[in] Position
      ///     The byte offset from the start of the file
      ///     where the structure is written.
      //-----------------------------------------------------------------
      void QueryEvent( const Common::TOCInfo& Info,
		       INT_4U FrameOffset,
		       INT_8U Position );

      //-----------------------------------------------------------------
      /// \brief asignment operator
      ///
      /// \param[in] Source
      ///     The source to be copied.
      //-----------------------------------------------------------------
      const FrTOCEvent& operator=( const Previous::FrTOCEvent& Source );

       //-----------------------------------------------------------------
      /// \brief equality operator
      ///
      /// \param[in] RHS
      ///     The FrTOCAdcData object to be compared.
      ///
      /// \return
      ///     The value true is returned if this object is equivelent
      ///     to the RHS instance; false otherwise.
      //-----------------------------------------------------------------
      bool operator==( const FrTOCEvent& RHS ) const;

      //-----------------------------------------------------------------
      /// \brief The description of structure
      ///
      /// \return
      ///     A Description object which describes this structure as
      ///     specified by the frame specification.
      //-----------------------------------------------------------------
      template< typename SE >
      static void Description( Common::Description& Desc );

    protected:
      nameEvent_container_type		m_info;

      //-----------------------------------------------------------------
      /// \brief Stream constructor
      ///
      /// \param[in] Stream
      ///     The input stream from where the object is being read.
      //-----------------------------------------------------------------
      FrTOCEvent( Common::IStream& Stream );

      //-----------------------------------------------------------------
      /// \brief Locate the position of the event
      ///
      /// \param[in] Frame
      ///     Frame offset of the event.
      ///
      /// \param[in] Event
      ///     The name of the event.
      ///
      /// \return
      ///     The file position of the requested event.
      //-----------------------------------------------------------------
      positionEvent_type
      positionEvent( INT_4U Frame, const std::string& Event ) const;

      //-----------------------------------------------------------------
      /// \brief Write the structure to the stream
      ///
      /// \param[in] Stream
      ///     The output stream where the object is to be written.
      //-----------------------------------------------------------------
      void write( Common::OStream& Stream ) const;
    };

    inline const FrTOCEvent::event_type& FrTOCEvent::event_type::
    operator=( const Previous::FrTOCEvent::event_type& Source )
    {
      GTime = Source.GTime;
      amplitudeEvent = Source.amplitudeEvent;
      positionEvent = Source.positionEvent;

      return *this;
    }

    inline Common::FrameSpec::size_type FrTOCEvent::
    Bytes( const Common::StreamBase& Stream ) const
    {
      Common::FrameSpec::size_type 
	retval = sizeof( nevent_type )
	+ sizeof( nTotalEvent_type );
      //-----------------------------------------------------------------
      // 
      //-----------------------------------------------------------------
      for ( nameEvent_container_type::const_iterator
	      cur = m_info.begin( ),
	      last = m_info.end( );
	    cur != last;
	    ++cur )
      {
	retval +=
	  cur->first.Bytes( )
	  + sizeof( nevent_type )
	  + ( cur->second.size( ) * event_type_size )
	  ;
      }
      return retval;
    }

    template< typename SE >
    void FrTOCEvent::
    Description( Common::Description& Desc )
    {
      Desc( SE( "nEventType", "INT_4U",
		"Number of FrEvent in the file" ) );
      Desc( SE( "nameEvent", "STRING[nEventType]",
		"Array of FrEvent names" ) );
      Desc( SE( "nEvent", "INT_4U[nEventType]",
		"Number of FrEvent for each type of FrEvent"
		" (size of nEventType)" ) );
      Desc( SE( "nTotalEvent", "INT_4U",
		"Total number of FrEvent" ) );
      Desc( SE( "GTimeSEvent", "INT_4U[nTotalEvent]",
		"GPS time in integer seconds" ) );
      Desc( SE( "GTimeNEvent", "INT_4U[nTotalEvent]",
		"Residual GPS time in integer nanoseconds" ) );
      Desc( SE( "amplitudeEvent", "REAL_4[nTotalEvent]",
		"Event amplitude" ) );
      Desc( SE( "positionEvent", "INT_8U[nTotalEvent]",
		"Array of FrEvent positions, in bytes,"
		" from beginning of file" ) );
    }

    inline const FrTOCEvent::nameEvent_container_type& FrTOCEvent::
    GetEvent( ) const
    {
      return m_info;
    }

  } // namespace - Version_8
} // namespace - FrameCPP

#endif /* FrameCPP_VERSION_8_FrTOCEvent_HH */
