#ifndef FrameCPP_VERSION_8_FrSummary_HH
#define FrameCPP_VERSION_8_FrSummary_HH

#include "framecpp/Common/SearchContainer.hh"
#include "framecpp/Common/TOCInfo.hh"

#include "framecpp/Version7/FrSummary.hh"

#include "framecpp/Version8/FrameSpec.hh"
#include "framecpp/Version8/STRING.hh"

#include "framecpp/Version8/FrTable.hh"
#include "framecpp/Version8/FrVect.hh"

namespace FrameCPP
{
  namespace Version_8
  {
    //===================================================================
    /// \brief Summary Data Structure Definition
    //===================================================================
    class FrSummary
      : public ObjectWithChecksum< Common::FrameSpec::Object >,
	public Common::TOCInfo
    {
    public:
      typedef Common::SearchContainer< FrVect, &FrVect::GetName >
      moments_type;
      typedef moments_type::iterator iterator;
      typedef moments_type::const_iterator const_iterator;

      typedef Common::SearchContainer< FrTable, &FrTable::GetNameSlow >
      table_type;
      typedef table_type::iterator table_iterator;
      typedef table_type::const_iterator const_table_iterator;

      //-----------------------------------------------------------------
      /// \brief Default constructor
      ///
      /// \return
      ///    A new instance of this object.
      //-----------------------------------------------------------------
      FrSummary( );

      //-----------------------------------------------------------------
      /// \brief Copy Constructor
      ///
      /// \param[in] Source
      ///     The object from which to copy the information.
      ///
      /// \return
      ///    A new instance of this object.
      //-----------------------------------------------------------------
      FrSummary( const FrSummary& Source );

      //-----------------------------------------------------------------
      /// \brief Constructor
      ///
      /// \param[in] name
      ///     Name of summary statistic.
      /// \param[in] comment
      ///     Comment.
      /// \param[in] test
      ///     Statistical test(s) used on raw data.
      /// \param[in] gtime
      ///     GPS time corresponding to this FrSummary.
      ///
      /// \return
      ///     A new instance of the object.
      //-----------------------------------------------------------------
      FrSummary( const std::string& name,
		 const std::string& comment,
		 const std::string& test,
		 const GPSTime& gtime );

      //-----------------------------------------------------------------
      /// \brief Promotion Constructor
      ///
      /// \param[in] Source
      ///     An FrSummary structure from a previous frame specification.
      /// \param[in] Stream
      ///     The stream from which the earliest version of the
      ///     FrSummary structure was read.
      ///
      /// \return
      ///     A new instance of the object.
      //-----------------------------------------------------------------
      FrSummary( Previous::FrSummary& Source, Common::IStream* Stream );

      //-----------------------------------------------------------------
      /// \brief Promote previous version of object to this version
      ///
      /// \param[in] Source
      ///     The frame specification version of the object.
      /// \param[in] Obj
      ///     The previous version of the object.
      /// \param[in] Stream
      ///     Stream from which the origional object was created.
      ///
      /// \return
      ///     A new instance of FrameH based on Obj
      //-----------------------------------------------------------------
      static promote_ret_type
      Promote( INT_2U Source,
	       promote_arg_type Obj,
	       Common::IStream* Stream );

      //-----------------------------------------------------------------
      /// \brief The name structure name of this object.
      ///
      /// \return
      ///     The name of the structure as specified by the frame
      ///     specification.
      //-----------------------------------------------------------------
      static const char* StructName( );

      //-----------------------------------------------------------------
      /// \brief The description of structure
      ///
      /// \return
      ///     A Description object which describes this structure as
      ///     specified by the frame specification.
      //-----------------------------------------------------------------
      static const Common::Description* StructDescription( );

      //-----------------------------------------------------------------
      /// \brief The name structure name of this object.
      ///
      /// \return
      ///     The name of the structure as specified by the frame
      ///     specification.
      //-----------------------------------------------------------------
      virtual const char* ObjectStructName( ) const;

#if WORKING_VIRTUAL_TOCQUERY
      //-----------------------------------------------------------------
      /// \brief Answer requests for information as needed by TOC
      ///
      /// \param[in] InfoClass
      ///     Class of information being requested.
      //-----------------------------------------------------------------
      virtual void TOCQuery( int InfoClass, ... ) const;
#endif /*  WORKING_VIRTUAL_TOCQUERY */

      //-----------------------------------------------------------------
      /// \brief Return the name of summary statistic.
      ///
      /// \return
      ///     The name of summary statistic.
      //-----------------------------------------------------------------
      const std::string& GetName( ) const;

      //-----------------------------------------------------------------
      /// \brief Return the comment.
      ///
      /// \return
      ///     The comment.
      //-----------------------------------------------------------------
      const std::string& GetComment( ) const;
  
      //-----------------------------------------------------------------
      /// \brief Return the statistical test(s) used on raw data.
      ///
      /// \return
      ///     The the statistical test(s) used on raw data.
      //-----------------------------------------------------------------
      const std::string& GetTest( ) const;
  
      //-----------------------------------------------------------------
      /// \brief Return the GPS time corresponding to this object.
      ///
      /// \return
      ///     The GPS time corresponding to this object.
      //-----------------------------------------------------------------
      const GPSTime& GetGTime() const;
  
      //-----------------------------------------------------------------
      /// \brief Identifier for vector containing statistical descriptors.
      ///
      /// \return
      ///     A constant reference to the statistical descriptors.
      //-----------------------------------------------------------------
      const moments_type& RefMoments( ) const;
  
      //-----------------------------------------------------------------
      /// \brief Identifier for vector containing statistical descriptors.
      ///
      /// \return
      ///     A reference to the statistical descriptors.
      //-----------------------------------------------------------------
      moments_type& RefMoments( );
  
      //-----------------------------------------------------------------
      /// \brief Identifier for table containing additional summary information.
      ///
      /// \return
      ///     A constant reference to the table containing additional
      ///     summary Information.
      //-----------------------------------------------------------------
      const table_type& RefTable( ) const;
  
      //-----------------------------------------------------------------
      /// \brief Identifier for table containing additional summary information.
      ///
      /// \return
      ///     A reference to the table containing additional
      ///     summary Information.
      //-----------------------------------------------------------------
      table_type& RefTable( );
   
      //-----------------------------------------------------------------
      /// \brief Merge with another FrSummary
      ///
      /// \param[in] RHS
      ///     The source of the information to append to this FrSummary
      ///     structure.
      ///
      /// \return
      ///     A reference to this object
      //-----------------------------------------------------------------
      FrSummary& Merge( const FrSummary& RHS );

      //-----------------------------------------------------------------
      /// \brief equality operator
      ///
      /// \param[in] RHS
      ///     The FrSummary object to be compared.
      ///
      /// \return
      ///     The value true is returned if this object is equivelent
      ///     to the RHS instance; false otherwise.
      //-----------------------------------------------------------------
      bool operator==( const FrSummary& RHS ) const;

      //-----------------------------------------------------------------
      /// \brief equality operator for abstract data type
      ///
      /// \param[in] RHS
      ///     The object to be compared.
      ///
      /// \return
      ///     The value true is returned if this object is equivelent
      ///     to the RHS instance; false otherwise.
      //-----------------------------------------------------------------
      virtual bool operator==( const Common::FrameSpec::Object& RHS ) const;

    protected:
      //-----------------------------------------------------------------
      /// \brief Demotes object to previous version of the frame spec
      ///
      /// \param[in] Target
      ///     The version of the frame specification to demote too.
      /// \param[in] Obj
      ///     The version of the object to demote.
      /// \param[in] Stream
      ///     The input stream from which the original object was read.
      ///
      /// \return
      ///     An object of the previous generation.
      //-----------------------------------------------------------------
      virtual demote_ret_type
      demote( INT_2U Target,
	      demote_arg_type Obj,
	      Common::IStream* Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Promotes object to another version of the frame spec
      ///
      /// \param[in] Target
      ///     The version of the promoted frame specification.
      /// \param[in] Obj
      ///     The object to be promoted.
      /// \param[in] Stream
      ///     The input stream from which the original object was read.
      ///
      /// \return
      ///     An object promoted to the next generation.
      //-----------------------------------------------------------------
      virtual promote_ret_type
      promote( INT_2U Target,
	       promote_arg_type Obj,
	       Common::IStream* Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Number of bytes needed to write this structure
      ///
      /// \param[in] Stream
      ///     The stream from which to the object is being read or
      ///     written.
      ///
      /// \return
      ///     The number of bytes need to read or write this object.
      //-----------------------------------------------------------------
      cmn_streamsize_type pBytes( const Common::StreamBase& Stream) const;

      //-----------------------------------------------------------------
      /// \brief Virtual constructor
      ///
      /// \param[in] Stream
      ///     The input stream from where the object is being read.
      //-----------------------------------------------------------------
      virtual FrSummary* pCreate( Common::IStream& Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Write the structure to the stream
      ///
      /// \param[in] Stream
      ///     The output stream where the object is to be written.
      //-----------------------------------------------------------------
      virtual void pWrite( Common::OStream& Stream ) const;

#if ! WORKING_VIRTUAL_TOCQUERY
      //-----------------------------------------------------------------
      /// \brief Answer requests for information as needed by TOC
      ///
      /// \param[in] InfoClass
      ///     Class of information being requested.
      /// \param[in] vl
      ///     Variable argument list of requested data.
      //-----------------------------------------------------------------
      virtual void vTOCQuery( int InfoClass, va_list vl ) const;
#endif /* ! WORKING_VIRTUAL_TOCQUERY */

    private:
      //-----------------------------------------------------------------
      /// \brief Object with checksum data.
      //-----------------------------------------------------------------
      typedef ObjectWithChecksum< Common::FrameSpec::Object > object_type;

      using Common::FrameSpec::Object::Create;

      struct fr_summary_data_type {
	bool operator==( const fr_summary_data_type& RHS ) const;

	//---------------------------------------------------------------
	/// Name of summary statistic.
	//---------------------------------------------------------------
	STRING		name;
	//---------------------------------------------------------------
	/// Comment.
	//---------------------------------------------------------------
	STRING		comment;
	//---------------------------------------------------------------
	/// Statistical test(s) used on raw data.
	//---------------------------------------------------------------
	STRING		test;
	//---------------------------------------------------------------
	/// GPS time corresponding to this object.
	//---------------------------------------------------------------
	GPSTime		GTime;
	//---------------------------------------------------------------
	/// Identifier for vector containing statistical descriptors.
	//---------------------------------------------------------------
	moments_type	moments;
	//---------------------------------------------------------------
	/// Indentifier for table containing additional summary
	/// information.
	//---------------------------------------------------------------
	table_type	table;
      };

      fr_summary_data_type	m_data;

      //-----------------------------------------------------------------
      /// \brief Stream Constructor
      ///
      /// \param[in] Stream
      ///     The stream from which the object is being read.
      ///
      /// \return
      ///    A new instance of this object.
      //-----------------------------------------------------------------
      FrSummary( Common::IStream& Stream );

    };

    inline FrSummary::promote_ret_type FrSummary::
    Promote( INT_2U Source,
	     promote_arg_type Obj,
	     Common::IStream* Stream )
    {
      return Object::PromoteObject< Previous::FrSummary, FrSummary >
	( DATA_FORMAT_VERSION, Source, Obj, Stream );
    }

    inline const char* FrSummary::
    StructName( )
    {
      static const char* name = "FrSummary";
      return name;
    }

    inline const std::string& FrSummary::
    GetComment() const
    {
      return m_data.comment;
    }
  
    inline const std::string& FrSummary::
    GetTest() const
    {
      return m_data.test;
    }
  
    inline const GPSTime& FrSummary::
    GetGTime() const
    {
      return m_data.GTime;
    }
  
    inline const FrSummary::moments_type& FrSummary::
    RefMoments( ) const
    {
      return m_data.moments;
    }
  
    inline FrSummary::moments_type& FrSummary::
    RefMoments( )
    {
      return m_data.moments;
    }
  
    inline const FrSummary::table_type& FrSummary::
    RefTable( ) const
    {
      return m_data.table;
    }
  
    inline FrSummary::table_type& FrSummary::
    RefTable( )
    {
      return m_data.table;
    }   
   
    inline bool FrSummary::
    operator==( const FrSummary& RHS ) const
    {
      return ( m_data == RHS.m_data );
    }
  } // namespace - Version_8
} // namespace FrameCPP
#endif /* FrameCPP_VERSION_8_FrSummary_HH */
