#ifndef FrameCPP_VERSION_8_FrMsg_HH
#define FrameCPP_VERSION_8_FrMsg_HH

#ifndef SWIGIMPORTED
#include <vector>
#endif /* SWIGIMPORTED */

#include "general/types.hh"

#include "framecpp/Common/TOCInfo.hh"

#include "framecpp/Version7/FrMsg.hh"

#include "framecpp/Version8/FrameSpec.hh"
#include "framecpp/Version8/GPSTime.hh"
#include "framecpp/Version8/STRING.hh"

namespace FrameCPP
{
  namespace Version_8
  {
    //===================================================================
    /// \brief Message Log Data Structure Definition
    //===================================================================
    class FrMsg
      : public ObjectWithChecksum< Common::FrameSpec::Object >,
	public Common::TOCInfo
    {
    public:
      //-----------------------------------------------------------------
      /// \brief Default constructor
      ///
      /// \return
      ///    A new instance of this object.
      //-----------------------------------------------------------------
      FrMsg( );

      //-----------------------------------------------------------------
      /// \brief Copy Constructor
      ///
      /// \param[in] Source
      ///     The object from which to copy the information.
      ///
      /// \return
      ///    A new instance of this object.
      //-----------------------------------------------------------------
      FrMsg( const FrMsg& Source );

      //-----------------------------------------------------------------
      /// \brief Copy Constructor
      ///
      /// \param[in] Alarm
      ///     Name of message, error flag or alarm state.
      /// \param[in] Message
      ///     Message body
      /// \param[in] Severity
      ///     Message severity level.
      /// \param[in] GTime
      ///     GPS time corresponding to this FrMsg.
      ///
      /// \return
      ///    A new instance of this object.
      //-----------------------------------------------------------------
      FrMsg( const std::string& Alarm,
	     const std::string& Message,
	     INT_4U Severity,
	     const GPSTime& GTime );

      //-----------------------------------------------------------------
      /// \brief Promotion Constructor
      ///
      /// \param[in] Source
      ///     An FrAdcData structure from a previous frame specification.
      /// \param[in] Stream
      ///     The stream from which the earliest version of the
      ///     FrAdcData structure was read.
      ///
      /// \return
      ///     A new instance of an FrAdcData object
      //-----------------------------------------------------------------
      FrMsg( Previous::FrMsg& Source, istream_type* Stream );

      //-----------------------------------------------------------------
      /// \brief The name structure name of this object.
      ///
      /// \return
      ///     The name of the structure as specified by the frame
      ///     specification.
      //-----------------------------------------------------------------
      static const char* StructName( );

      //-----------------------------------------------------------------
      /// \brief The description of structure
      ///
      /// \return
      ///     A Description object which describes this structure as
      ///     specified by the frame specification.
      //-----------------------------------------------------------------
      static const Common::Description* StructDescription( );

      //-----------------------------------------------------------------
      /// \brief The name structure name of this object.
      ///
      /// \return
      ///     The name of the structure as specified by the frame
      ///     specification.
      //-----------------------------------------------------------------
      virtual const char* ObjectStructName( ) const;

#if WORKING_VIRTUAL_TOCQUERY
      //-----------------------------------------------------------------
      /// \brief Answer requests for information as needed by TOC
      ///
      /// \param[in] InfoClass
      ///     Class of information being requested.
      //-----------------------------------------------------------------
      void TOCQuery( int InfoClass, ... ) const;
#endif /*  WORKING_VIRTUAL_TOCQUERY */

      //-----------------------------------------------------------------
      /// \brief Retrieve the name of message, error flag, or alarm state.
      ///
      /// \return
      ///     The name of message, error flag, or alarm state.
      //-----------------------------------------------------------------
      const std::string& GetAlarm() const;

      //-----------------------------------------------------------------
      /// \brief Retieve the message body.
      ///
      /// \return
      ///     The message body.
      //-----------------------------------------------------------------
      const std::string& GetMessage( ) const { return m_data.message; }

      //-----------------------------------------------------------------
      /// \brief Retieve the message severity level.
      ///
      /// \return
      ///     The message severity level.
      //-----------------------------------------------------------------
      INT_4U GetSeverity( ) const { return m_data.severity; }

      //-----------------------------------------------------------------
      /// \brief Retrieve the GPS time corresponding to this FrMsg.
      ///
      /// \return
      ///     The GPS time corresponding to this FrMsg.
      //-----------------------------------------------------------------
      const GPSTime& GetGTime( ) const { return m_data.GTime; }   
   
      //-----------------------------------------------------------------
      /// \brief Merge with another FrMsg
      ///
      /// \param[in] RHS
      ///     The source of the information to append to this FrMsg
      ///     structure.
      ///
      /// \return
      ///     A reference to this object
      //-----------------------------------------------------------------
      FrMsg& Merge( const FrMsg& RHS );

      //-----------------------------------------------------------------
      /// \brief Promote previous version of object to this version
      ///
      /// \param[in] Source
      ///     The frame specification version of the object.
      /// \param[in] Obj
      ///     The previous version of the object.
      /// \param[in] Stream
      ///     Stream from which the origional object was created.
      ///
      /// \return
      ///     A new instance of FrameH based on Obj
      //-----------------------------------------------------------------
      static promote_ret_type
      Promote( INT_2U Source,
	       promote_arg_type Obj,
	       istream_type* Stream );

      //-----------------------------------------------------------------
      /// \brief equality operator
      ///
      /// \param[in] RHS
      ///     The FrAdcData object to be compared.
      ///
      /// \return
      ///     The value true is returned if this object is equivelent
      ///     to the RHS instance; false otherwise.
      //-----------------------------------------------------------------
      bool operator==( const FrMsg& RHS ) const;

      //-----------------------------------------------------------------
      /// \brief equality operator for abstract data type
      ///
      /// \param[in] RHS
      ///     The object to be compared.
      ///
      /// \return
      ///     The value true is returned if this object is equivelent
      ///     to the RHS instance; false otherwise.
      //-----------------------------------------------------------------
      virtual bool operator==( const Common::FrameSpec::Object& RHS ) const;

    protected:
      //-----------------------------------------------------------------
      /// \brief Demotes object to previous version of the frame spec
      ///
      /// \param[in] Target
      ///     The version of the frame specification to demote too.
      /// \param[in] Obj
      ///     The version of the object to demote.
      /// \param[in] Stream
      ///     The input stream from which the original object was read.
      ///
      /// \return
      ///     An object of the previous generation.
      //-----------------------------------------------------------------
      virtual demote_ret_type
      demote( INT_2U Target,
	      demote_arg_type Obj,
	      istream_type* Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Promotes object to another version of the frame spec
      ///
      /// \param[in] Target
      ///     The version of the promoted frame specification.
      /// \param[in] Obj
      ///     The object to be promoted.
      /// \param[in] Stream
      ///     The input stream from which the original object was read.
      ///
      /// \return
      ///     An object promoted to the next generation.
      //-----------------------------------------------------------------
      virtual promote_ret_type
      promote( INT_2U Target,
	       promote_arg_type Obj,
	       istream_type* Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Number of bytes needed to write this structure
      ///
      /// \param[in] Stream
      ///     The stream from which to the object is being read or
      ///     written.
      ///
      /// \return
      ///     The number of bytes need to read or write this object.
      //-----------------------------------------------------------------
      cmn_streamsize_type pBytes( const Common::StreamBase& Stream) const;

      //-----------------------------------------------------------------
      /// \brief Virtual constructor
      ///
      /// \param[in] Stream
      ///     The input stream from where the object is being read.
      //-----------------------------------------------------------------
      virtual FrMsg* pCreate( istream_type& Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Write the structure to the stream
      ///
      /// \param[in] Stream
      ///     The output stream where the object is to be written.
      //-----------------------------------------------------------------
      virtual void pWrite( ostream_type& Stream ) const;

#if ! WORKING_VIRTUAL_TOCQUERY
      //-----------------------------------------------------------------
      /// \brief Answer requests for information as needed by TOC
      ///
      /// \param[in] InfoClass
      ///     Class of information being requested.
      /// \param[in] vl
      ///     Variable argument list of requested data.
      //-----------------------------------------------------------------
      virtual void vTOCQuery( int InfoClass, va_list vl ) const;
#endif /* ! WORKING_VIRTUAL_TOCQUERY */

    private:
      //-----------------------------------------------------------------
      //-----------------------------------------------------------------
      typedef ObjectWithChecksum< Common::FrameSpec::Object > object_type;

      using Common::FrameSpec::Object::Create;

      struct fr_msg_data_type {
	//---------------------------------------------------------------
	/// \brief equality operator
        ///
        /// \param[in] RHS
        ///     The fr_msg_data_type object to be compared.
        ///
        /// \return
        ///     The value true is returned if this object is equivelent
        ///     to the RHS instance; false otherwise.
        //---------------------------------------------------------------
	bool operator==( const fr_msg_data_type& RHS ) const;

        //---------------------------------------------------------------
	/// Name o message, error flag or alarm state.
        //---------------------------------------------------------------
	STRING	alarm;
        //---------------------------------------------------------------
	/// Message body.
        //---------------------------------------------------------------
	STRING	message;
        //---------------------------------------------------------------
	/// Message severity level.
	/// The default level is zero.
	/// \note
	///     These are yet to be defined.
        //---------------------------------------------------------------
	INT_4U	severity;
        //---------------------------------------------------------------
	/// The GPS time corresponding alarm.
        //---------------------------------------------------------------
	GPSTime	GTime;
      };

      fr_msg_data_type	m_data;

      //-----------------------------------------------------------------
      /// \brief Stream Constructor
      ///
      /// \param[in] Stream
      ///     The stream from which the object is being read.
      ///
      /// \return
      ///    A new instance of this object.
      //-----------------------------------------------------------------
      FrMsg( istream_type& Stream );
    };

    inline const char* FrMsg::
    StructName( )
    {
      static const char* name = "FrMsg";
      return name;
    }

    inline bool FrMsg::
    operator==( const FrMsg& RHS ) const
    {
      return ( m_data == RHS.m_data );
    }

  } // namespace - Version_8
} // namespace FrameCPP
#endif /* FrameCPP_VERSION_8_FrMsg_HH */
